"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var JobRepository_1;
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobRepository = void 0;
const bullmq_1 = require("@nestjs/bullmq");
const common_1 = require("@nestjs/common");
const core_1 = require("@nestjs/core");
const bullmq_2 = require("bullmq");
const promises_1 = require("node:timers/promises");
const enum_1 = require("../enum");
const config_repository_1 = require("./config.repository");
const event_repository_1 = require("./event.repository");
const logging_repository_1 = require("./logging.repository");
const misc_1 = require("../utils/misc");
let JobRepository = JobRepository_1 = class JobRepository {
    moduleRef;
    configRepository;
    eventRepository;
    logger;
    workers = {};
    handlers = {};
    constructor(moduleRef, configRepository, eventRepository, logger) {
        this.moduleRef = moduleRef;
        this.configRepository = configRepository;
        this.eventRepository = eventRepository;
        this.logger = logger;
        this.logger.setContext(JobRepository_1.name);
    }
    setup(services) {
        const reflector = this.moduleRef.get(core_1.Reflector, { strict: false });
        for (const Service of services) {
            const instance = this.moduleRef.get(Service);
            for (const methodName of (0, misc_1.getMethodNames)(instance)) {
                const handler = instance[methodName];
                const config = reflector.get(enum_1.MetadataKey.JobConfig, handler);
                if (!config) {
                    continue;
                }
                const { name: jobName, queue: queueName } = config;
                const label = `${Service.name}.${handler.name}`;
                if (this.handlers[jobName]) {
                    const jobKey = (0, misc_1.getKeyByValue)(enum_1.JobName, jobName);
                    const errorMessage = `Failed to add job handler for ${label}`;
                    this.logger.error(`${errorMessage}. JobName.${jobKey} is already handled by ${this.handlers[jobName].label}.`);
                    throw new misc_1.ImmichStartupError(errorMessage);
                }
                this.handlers[jobName] = {
                    label,
                    jobName,
                    queueName,
                    handler: handler.bind(instance),
                };
                this.logger.verbose(`Added job handler: ${jobName} => ${label}`);
            }
        }
        for (const [jobKey, jobName] of Object.entries(enum_1.JobName)) {
            const item = this.handlers[jobName];
            if (!item) {
                const errorMessage = `Failed to find job handler for Job.${jobKey} ("${jobName}")`;
                this.logger.error(`${errorMessage}. Make sure to add the @OnJob({ name: JobName.${jobKey}, queue: QueueName.XYZ }) decorator for the new job.`);
                throw new misc_1.ImmichStartupError(errorMessage);
            }
        }
    }
    startWorkers() {
        const { bull } = this.configRepository.getEnv();
        for (const queueName of Object.values(enum_1.QueueName)) {
            this.logger.debug(`Starting worker for queue: ${queueName}`);
            this.workers[queueName] = new bullmq_2.Worker(queueName, (job) => this.eventRepository.emit('JobRun', queueName, job), { ...bull.config, concurrency: 1 });
        }
    }
    async run({ name, data }) {
        const item = this.handlers[name];
        if (!item) {
            this.logger.warn(`Skipping unknown job: "${name}"`);
            return enum_1.JobStatus.Skipped;
        }
        return item.handler(data);
    }
    setConcurrency(queueName, concurrency) {
        const worker = this.workers[queueName];
        if (!worker) {
            this.logger.warn(`Unable to set queue concurrency, worker not found: '${queueName}'`);
            return;
        }
        worker.concurrency = concurrency;
    }
    async getQueueStatus(name) {
        const queue = this.getQueue(name);
        return {
            isActive: !!(await queue.getActiveCount()),
            isPaused: await queue.isPaused(),
        };
    }
    pause(name) {
        return this.getQueue(name).pause();
    }
    resume(name) {
        return this.getQueue(name).resume();
    }
    empty(name) {
        return this.getQueue(name).drain();
    }
    clear(name, type) {
        return this.getQueue(name).clean(0, 1000, type);
    }
    getJobCounts(name) {
        return this.getQueue(name).getJobCounts('active', 'completed', 'failed', 'delayed', 'waiting', 'paused');
    }
    getQueueName(name) {
        return this.handlers[name].queueName;
    }
    async queueAll(items) {
        if (items.length === 0) {
            return;
        }
        const promises = [];
        const itemsByQueue = {};
        for (const item of items) {
            const queueName = this.getQueueName(item.name);
            const job = {
                name: item.name,
                data: item.data || {},
                options: this.getJobOptions(item) || undefined,
            };
            if (job.options?.jobId) {
                promises.push(this.getQueue(queueName).add(item.name, item.data, job.options));
            }
            else {
                itemsByQueue[queueName] = itemsByQueue[queueName] || [];
                itemsByQueue[queueName].push(job);
            }
        }
        for (const [queueName, jobs] of Object.entries(itemsByQueue)) {
            const queue = this.getQueue(queueName);
            promises.push(queue.addBulk(jobs));
        }
        await Promise.all(promises);
    }
    async queue(item) {
        return this.queueAll([item]);
    }
    async waitForQueueCompletion(...queues) {
        let activeQueue;
        do {
            const statuses = await Promise.all(queues.map((name) => this.getQueueStatus(name)));
            activeQueue = statuses.find((status) => status.isActive);
        } while (activeQueue);
        {
            this.logger.verbose(`Waiting for ${activeQueue} queue to stop...`);
            await (0, promises_1.setTimeout)(1000);
        }
    }
    getJobOptions(item) {
        switch (item.name) {
            case enum_1.JobName.NotifyAlbumUpdate: {
                return {
                    jobId: `${item.data.id}/${item.data.recipientId}`,
                    delay: item.data?.delay,
                };
            }
            case enum_1.JobName.StorageTemplateMigrationSingle: {
                return { jobId: item.data.id };
            }
            case enum_1.JobName.PersonGenerateThumbnail: {
                return { priority: 1 };
            }
            case enum_1.JobName.FacialRecognitionQueueAll: {
                return { jobId: enum_1.JobName.FacialRecognitionQueueAll };
            }
            default: {
                return null;
            }
        }
    }
    getQueue(queue) {
        return this.moduleRef.get((0, bullmq_1.getQueueToken)(queue), { strict: false });
    }
    async removeJob(name, jobID) {
        const existingJob = await this.getQueue(this.getQueueName(name)).getJob(jobID);
        if (existingJob) {
            await existingJob.remove();
        }
    }
};
exports.JobRepository = JobRepository;
exports.JobRepository = JobRepository = JobRepository_1 = __decorate([
    (0, common_1.Injectable)(),
    __metadata("design:paramtypes", [core_1.ModuleRef,
        config_repository_1.ConfigRepository,
        event_repository_1.EventRepository,
        logging_repository_1.LoggingRepository])
], JobRepository);
//# sourceMappingURL=job.repository.js.map