"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.PluginHostFunctions = void 0;
const common_1 = require("@nestjs/common");
const enum_1 = require("../enum");
const access_1 = require("../utils/access");
class PluginHostFunctions {
    assetRepository;
    albumRepository;
    accessRepository;
    cryptoRepository;
    logger;
    pluginJwtSecret;
    constructor(assetRepository, albumRepository, accessRepository, cryptoRepository, logger, pluginJwtSecret) {
        this.assetRepository = assetRepository;
        this.albumRepository = albumRepository;
        this.accessRepository = accessRepository;
        this.cryptoRepository = cryptoRepository;
        this.logger = logger;
        this.pluginJwtSecret = pluginJwtSecret;
    }
    getHostFunctions() {
        return {
            'extism:host/user': {
                updateAsset: (cp, offs) => this.handleUpdateAsset(cp, offs),
                addAssetToAlbum: (cp, offs) => this.handleAddAssetToAlbum(cp, offs),
            },
        };
    }
    async handleUpdateAsset(cp, offs) {
        const input = JSON.parse(cp.read(offs).text());
        await this.updateAsset(input);
    }
    async handleAddAssetToAlbum(cp, offs) {
        const input = JSON.parse(cp.read(offs).text());
        await this.addAssetToAlbum(input);
    }
    validateToken(authToken) {
        try {
            const auth = this.cryptoRepository.verifyJwt(authToken, this.pluginJwtSecret);
            if (!auth.userId) {
                throw new common_1.UnauthorizedException('Invalid token: missing userId');
            }
            return auth;
        }
        catch (error) {
            this.logger.error('Token validation failed:', error);
            throw new common_1.UnauthorizedException('Invalid token');
        }
    }
    async updateAsset(input) {
        const { authToken, id, ...assetData } = input;
        const auth = this.validateToken(authToken);
        await (0, access_1.requireAccess)(this.accessRepository, {
            auth: { user: { id: auth.userId } },
            permission: enum_1.Permission.AssetUpdate,
            ids: [id],
        });
        this.logger.log(`Updating asset ${id} -- ${JSON.stringify(assetData)}`);
        await this.assetRepository.update({ id, ...assetData });
    }
    async addAssetToAlbum(input) {
        const { authToken, assetId, albumId } = input;
        const auth = this.validateToken(authToken);
        await (0, access_1.requireAccess)(this.accessRepository, {
            auth: { user: { id: auth.userId } },
            permission: enum_1.Permission.AssetRead,
            ids: [assetId],
        });
        await (0, access_1.requireAccess)(this.accessRepository, {
            auth: { user: { id: auth.userId } },
            permission: enum_1.Permission.AlbumUpdate,
            ids: [albumId],
        });
        this.logger.log(`Adding asset ${assetId} to album ${albumId}`);
        await this.albumRepository.addAssetIds(albumId, [assetId]);
        return 0;
    }
}
exports.PluginHostFunctions = PluginHostFunctions;
//# sourceMappingURL=plugin-host.functions.js.map