"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.StorageTemplateService = void 0;
const common_1 = require("@nestjs/common");
const handlebars_1 = __importDefault(require("handlebars"));
const luxon_1 = require("luxon");
const node_path_1 = __importDefault(require("node:path"));
const sanitize_filename_1 = __importDefault(require("sanitize-filename"));
const storage_core_1 = require("../cores/storage.core");
const decorators_1 = require("../decorators");
const enum_1 = require("../enum");
const base_service_1 = require("./base.service");
const file_1 = require("../utils/file");
const storageTokens = {
    secondOptions: ['s', 'ss', 'SSS'],
    minuteOptions: ['m', 'mm'],
    dayOptions: ['d', 'dd'],
    weekOptions: ['W', 'WW'],
    hourOptions: ['h', 'hh', 'H', 'HH'],
    yearOptions: ['y', 'yy'],
    monthOptions: ['M', 'MM', 'MMM', 'MMMM'],
};
const storagePresets = [
    '{{y}}/{{y}}-{{MM}}-{{dd}}/{{filename}}',
    '{{y}}/{{MM}}-{{dd}}/{{filename}}',
    '{{y}}/{{MMMM}}-{{dd}}/{{filename}}',
    '{{y}}/{{MM}}/{{filename}}',
    '{{y}}/{{#if album}}{{album}}{{else}}Other/{{MM}}{{/if}}/{{filename}}',
    '{{#if album}}{{album-startDate-y}}/{{album}}{{else}}{{y}}/Other/{{MM}}{{/if}}/{{filename}}',
    '{{y}}/{{MMM}}/{{filename}}',
    '{{y}}/{{MMMM}}/{{filename}}',
    '{{y}}/{{MM}}/{{dd}}/{{filename}}',
    '{{y}}/{{MMMM}}/{{dd}}/{{filename}}',
    '{{y}}/{{y}}-{{MM}}/{{y}}-{{MM}}-{{dd}}/{{filename}}',
    '{{y}}-{{MM}}-{{dd}}/{{filename}}',
    '{{y}}-{{MMM}}-{{dd}}/{{filename}}',
    '{{y}}-{{MMMM}}-{{dd}}/{{filename}}',
    '{{y}}/{{y}}-{{MM}}/{{filename}}',
    '{{y}}/{{y}}-{{WW}}/{{filename}}',
    '{{y}}/{{y}}-{{MM}}-{{dd}}/{{assetId}}',
    '{{y}}/{{y}}-{{MM}}/{{assetId}}',
    '{{y}}/{{y}}-{{WW}}/{{assetId}}',
    '{{album}}/{{filename}}',
];
let StorageTemplateService = class StorageTemplateService extends base_service_1.BaseService {
    _template = null;
    get template() {
        if (!this._template) {
            throw new Error('Template not initialized');
        }
        return this._template;
    }
    onConfigInit({ newConfig }) {
        const template = newConfig.storageTemplate.template;
        if (!this._template || template !== this.template.raw) {
            this.logger.debug(`Compiling new storage template: ${template}`);
            this._template = this.compile(template);
        }
    }
    onConfigUpdate({ newConfig }) {
        this.onConfigInit({ newConfig });
    }
    onConfigValidate({ newConfig }) {
        try {
            const { compiled } = this.compile(newConfig.storageTemplate.template);
            this.render(compiled, {
                asset: {
                    fileCreatedAt: new Date(),
                    originalPath: '/upload/test/IMG_123.jpg',
                    type: enum_1.AssetType.Image,
                    id: 'd587e44b-f8c0-4832-9ba3-43268bbf5d4e',
                },
                filename: 'IMG_123',
                extension: 'jpg',
                albumName: 'album',
                albumStartDate: new Date(),
                albumEndDate: new Date(),
            });
        }
        catch (error) {
            this.logger.warn(`Storage template validation failed: ${JSON.stringify(error)}`);
            throw new Error(`Invalid storage template: ${error}`);
        }
    }
    getStorageTemplateOptions() {
        return { ...storageTokens, presetOptions: storagePresets };
    }
    async onAssetMetadataExtracted({ source, assetId }) {
        await this.jobRepository.queue({ name: enum_1.JobName.StorageTemplateMigrationSingle, data: { source, id: assetId } });
    }
    async handleMigrationSingle({ id }) {
        const config = await this.getConfig({ withCache: true });
        const storageTemplateEnabled = config.storageTemplate.enabled;
        if (!storageTemplateEnabled) {
            return enum_1.JobStatus.Skipped;
        }
        const asset = await this.assetJobRepository.getForStorageTemplateJob(id);
        if (!asset) {
            return enum_1.JobStatus.Failed;
        }
        const user = await this.userRepository.get(asset.ownerId, {});
        const storageLabel = user?.storageLabel || null;
        const filename = asset.originalFileName || asset.id;
        await this.moveAsset(asset, { storageLabel, filename });
        if (asset.livePhotoVideoId) {
            const livePhotoVideo = await this.assetJobRepository.getForStorageTemplateJob(asset.livePhotoVideoId);
            if (!livePhotoVideo) {
                return enum_1.JobStatus.Failed;
            }
            const motionFilename = (0, file_1.getLivePhotoMotionFilename)(filename, livePhotoVideo.originalPath);
            await this.moveAsset(livePhotoVideo, { storageLabel, filename: motionFilename });
        }
        return enum_1.JobStatus.Success;
    }
    async handleMigration() {
        this.logger.log('Starting storage template migration');
        const { storageTemplate } = await this.getConfig({ withCache: true });
        const { enabled } = storageTemplate;
        if (!enabled) {
            this.logger.log('Storage template migration disabled, skipping');
            return enum_1.JobStatus.Skipped;
        }
        await this.moveRepository.cleanMoveHistory();
        const assets = this.assetJobRepository.streamForStorageTemplateJob();
        const users = await this.userRepository.getList();
        for await (const asset of assets) {
            const user = users.find((user) => user.id === asset.ownerId);
            const storageLabel = user?.storageLabel || null;
            const filename = asset.originalFileName || asset.id;
            await this.moveAsset(asset, { storageLabel, filename });
        }
        this.logger.debug('Cleaning up empty directories...');
        const libraryFolder = storage_core_1.StorageCore.getBaseFolder(enum_1.StorageFolder.Library);
        await this.storageRepository.removeEmptyDirs(libraryFolder);
        this.logger.log('Finished storage template migration');
        return enum_1.JobStatus.Success;
    }
    async handleMoveHistoryCleanup({ assetId }) {
        this.logger.debug(`Cleaning up move history for asset ${assetId}`);
        await this.moveRepository.cleanMoveHistorySingle(assetId);
    }
    async moveAsset(asset, metadata) {
        if (asset.isExternal || storage_core_1.StorageCore.isAndroidMotionPath(asset.originalPath)) {
            return;
        }
        return this.databaseRepository.withLock(enum_1.DatabaseLock.StorageTemplateMigration, async () => {
            const { id, sidecarPath, originalPath, checksum, fileSizeInByte } = asset;
            const oldPath = originalPath;
            const newPath = await this.getTemplatePath(asset, metadata);
            if (!fileSizeInByte) {
                this.logger.error(`Asset ${id} missing exif info, skipping storage template migration`);
                return;
            }
            try {
                await this.storageCore.moveFile({
                    entityId: id,
                    pathType: enum_1.AssetPathType.Original,
                    oldPath,
                    newPath,
                    assetInfo: { sizeInBytes: fileSizeInByte, checksum },
                });
                if (sidecarPath) {
                    await this.storageCore.moveFile({
                        entityId: id,
                        pathType: enum_1.AssetPathType.Sidecar,
                        oldPath: sidecarPath,
                        newPath: `${newPath}.xmp`,
                    });
                }
            }
            catch (error) {
                this.logger.error(`Problem applying storage template`, error?.stack, { id, oldPath, newPath });
            }
        });
    }
    async getTemplatePath(asset, metadata) {
        const { storageLabel, filename } = metadata;
        try {
            const filenameWithoutExtension = node_path_1.default.basename(filename, node_path_1.default.extname(filename));
            const source = asset.originalPath;
            let extension = node_path_1.default.extname(source).split('.').pop();
            const sanitized = (0, sanitize_filename_1.default)(node_path_1.default.basename(filenameWithoutExtension, `.${extension}`));
            extension = extension?.toLowerCase();
            const rootPath = storage_core_1.StorageCore.getLibraryFolder({ id: asset.ownerId, storageLabel });
            switch (extension) {
                case 'jpeg':
                case 'jpe': {
                    extension = 'jpg';
                    break;
                }
                case 'tif': {
                    extension = 'tiff';
                    break;
                }
                case '3gpp': {
                    extension = '3gp';
                    break;
                }
                case 'mpeg':
                case 'mpe': {
                    extension = 'mpg';
                    break;
                }
                case 'm2ts':
                case 'm2t': {
                    extension = 'mts';
                    break;
                }
            }
            let albumName = null;
            let albumStartDate = null;
            let albumEndDate = null;
            if (this.template.needsAlbum) {
                const albums = await this.albumRepository.getByAssetId(asset.ownerId, asset.id);
                const album = albums?.[0];
                if (album) {
                    albumName = album.albumName || null;
                    if (this.template.needsAlbumMetadata) {
                        const [metadata] = await this.albumRepository.getMetadataForIds([album.id]);
                        albumStartDate = metadata?.startDate || null;
                        albumEndDate = metadata?.endDate || null;
                    }
                }
            }
            const storagePath = this.render(this.template.compiled, {
                asset,
                filename: sanitized,
                extension,
                albumName,
                albumStartDate,
                albumEndDate,
            });
            const fullPath = node_path_1.default.normalize(node_path_1.default.join(rootPath, storagePath));
            let destination = `${fullPath}.${extension}`;
            if (!fullPath.startsWith(rootPath)) {
                this.logger.warn(`Skipped attempt to access an invalid path: ${fullPath}. Path should start with ${rootPath}`);
                return source;
            }
            if (source === destination) {
                return source;
            }
            if (source.startsWith(fullPath) && source.endsWith(`.${extension}`)) {
                const diff = source.replace(fullPath, '').replace(`.${extension}`, '');
                const hasDuplicationAnnotation = /^\+\d+$/.test(diff);
                if (hasDuplicationAnnotation) {
                    return source;
                }
            }
            let duplicateCount = 0;
            while (true) {
                const exists = await this.storageRepository.checkFileExists(destination);
                if (!exists) {
                    break;
                }
                duplicateCount++;
                destination = `${fullPath}+${duplicateCount}.${extension}`;
            }
            return destination;
        }
        catch (error) {
            this.logger.error(`Unable to get template path for ${filename}: ${error}`);
            return asset.originalPath;
        }
    }
    compile(template) {
        return {
            raw: template,
            compiled: handlebars_1.default.compile(template, { knownHelpers: undefined, strict: true }),
            needsAlbum: template.includes('album'),
            needsAlbumMetadata: template.includes('album-startDate') || template.includes('album-endDate'),
        };
    }
    render(template, options) {
        const { filename, extension, asset, albumName, albumStartDate, albumEndDate } = options;
        const substitutions = {
            filename,
            ext: extension,
            filetype: asset.type == enum_1.AssetType.Image ? 'IMG' : 'VID',
            filetypefull: asset.type == enum_1.AssetType.Image ? 'IMAGE' : 'VIDEO',
            assetId: asset.id,
            assetIdShort: asset.id.slice(-12),
            album: (albumName && (0, sanitize_filename_1.default)(albumName.replaceAll(/\.+/g, ''))) || '',
        };
        const systemTimeZone = Intl.DateTimeFormat().resolvedOptions().timeZone;
        const zone = asset.timeZone || systemTimeZone;
        const dt = luxon_1.DateTime.fromJSDate(asset.fileCreatedAt, { zone });
        for (const token of Object.values(storageTokens).flat()) {
            substitutions[token] = dt.toFormat(token);
            if (albumName) {
                substitutions['album-startDate-' + token] = albumStartDate
                    ? luxon_1.DateTime.fromJSDate(albumStartDate, { zone: systemTimeZone }).toFormat(token)
                    : '';
                substitutions['album-endDate-' + token] = albumEndDate
                    ? luxon_1.DateTime.fromJSDate(albumEndDate, { zone: systemTimeZone }).toFormat(token)
                    : '';
            }
        }
        return template(substitutions).replaceAll(/\/{2,}/gm, '/');
    }
};
exports.StorageTemplateService = StorageTemplateService;
__decorate([
    (0, decorators_1.OnEvent)({ name: 'ConfigInit' }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", void 0)
], StorageTemplateService.prototype, "onConfigInit", null);
__decorate([
    (0, decorators_1.OnEvent)({ name: 'ConfigUpdate', server: true }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", void 0)
], StorageTemplateService.prototype, "onConfigUpdate", null);
__decorate([
    (0, decorators_1.OnEvent)({ name: 'ConfigValidate' }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", void 0)
], StorageTemplateService.prototype, "onConfigValidate", null);
__decorate([
    (0, decorators_1.OnEvent)({ name: 'AssetMetadataExtracted' }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], StorageTemplateService.prototype, "onAssetMetadataExtracted", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.StorageTemplateMigrationSingle, queue: enum_1.QueueName.StorageTemplateMigration }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], StorageTemplateService.prototype, "handleMigrationSingle", null);
__decorate([
    (0, decorators_1.OnJob)({ name: enum_1.JobName.StorageTemplateMigration, queue: enum_1.QueueName.StorageTemplateMigration }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", Promise)
], StorageTemplateService.prototype, "handleMigration", null);
__decorate([
    (0, decorators_1.OnEvent)({ name: 'AssetDelete' }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], StorageTemplateService.prototype, "handleMoveHistoryCleanup", null);
exports.StorageTemplateService = StorageTemplateService = __decorate([
    (0, common_1.Injectable)()
], StorageTemplateService);
//# sourceMappingURL=storage-template.service.js.map