"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.asRenameKey = exports.asJsonString = exports.asColumnList = exports.asColumnComment = exports.getColumnModifiers = exports.getColumnType = exports.isDefaultEqual = exports.compare = exports.haveEqualOverrides = exports.haveEqualColumns = exports.setIsEqual = exports.fromColumnValue = exports.sha1 = exports.asOptions = void 0;
const node_crypto_1 = require("node:crypto");
const asOptions = (options) => {
    if (typeof options === 'string') {
        return { name: options };
    }
    return options;
};
exports.asOptions = asOptions;
const sha1 = (value) => (0, node_crypto_1.createHash)('sha1').update(value).digest('hex');
exports.sha1 = sha1;
const fromColumnValue = (columnValue) => {
    if (columnValue === undefined) {
        return;
    }
    if (typeof columnValue === 'function') {
        return columnValue();
    }
    const value = columnValue;
    if (value === null) {
        return value;
    }
    if (typeof value === 'number') {
        return String(value);
    }
    if (typeof value === 'boolean') {
        return value ? 'true' : 'false';
    }
    if (value instanceof Date) {
        return `'${value.toISOString()}'`;
    }
    return `'${String(value)}'`;
};
exports.fromColumnValue = fromColumnValue;
const setIsEqual = (source, target) => source.size === target.size && [...source].every((x) => target.has(x));
exports.setIsEqual = setIsEqual;
const haveEqualColumns = (sourceColumns, targetColumns) => {
    return (0, exports.setIsEqual)(new Set(sourceColumns ?? []), new Set(targetColumns ?? []));
};
exports.haveEqualColumns = haveEqualColumns;
const haveEqualOverrides = (source, target) => {
    if (!source.override || !target.override) {
        return false;
    }
    const sourceValue = source.override.value;
    const targetValue = target.override.value;
    return sourceValue.name === targetValue.name && sourceValue.sql === targetValue.sql;
};
exports.haveEqualOverrides = haveEqualOverrides;
const compare = (sources, targets, options, comparer) => {
    options = options || {};
    const sourceMap = Object.fromEntries(sources.map((table) => [table.name, table]));
    const targetMap = Object.fromEntries(targets.map((table) => [table.name, table]));
    const items = [];
    const keys = new Set([...Object.keys(sourceMap), ...Object.keys(targetMap)]);
    const missingKeys = new Set();
    const extraKeys = new Set();
    for (const key of keys) {
        const source = sourceMap[key];
        const target = targetMap[key];
        if (isIgnored(source, target, options ?? true)) {
            continue;
        }
        if (isSynchronizeDisabled(source, target)) {
            continue;
        }
        if (source && !target) {
            missingKeys.add(key);
            continue;
        }
        if (!source && target) {
            extraKeys.add(key);
            continue;
        }
        if ((0, exports.haveEqualOverrides)(source, target)) {
            continue;
        }
        items.push(...comparer.onCompare(source, target));
    }
    if (comparer.getRenameKey && comparer.onRename) {
        const renameMap = {};
        for (const sourceKey of missingKeys) {
            const source = sourceMap[sourceKey];
            const renameKey = comparer.getRenameKey(source);
            renameMap[renameKey] = sourceKey;
        }
        for (const targetKey of extraKeys) {
            const target = targetMap[targetKey];
            const renameKey = comparer.getRenameKey(target);
            const sourceKey = renameMap[renameKey];
            if (!sourceKey) {
                continue;
            }
            const source = sourceMap[sourceKey];
            items.push(...comparer.onRename(source, target));
            missingKeys.delete(sourceKey);
            extraKeys.delete(targetKey);
        }
    }
    for (const key of missingKeys) {
        items.push(...comparer.onMissing(sourceMap[key]));
    }
    for (const key of extraKeys) {
        items.push(...comparer.onExtra(targetMap[key]));
    }
    return items;
};
exports.compare = compare;
const isIgnored = (source, target, options) => {
    if (typeof options === 'boolean') {
        return !options;
    }
    return (options.ignoreExtra && !source) || (options.ignoreMissing && !target);
};
const isSynchronizeDisabled = (source, target) => {
    return source?.synchronize === false || target?.synchronize === false;
};
const isDefaultEqual = (source, target) => {
    if (source.default === target.default) {
        return true;
    }
    if (source.default === undefined || target.default === undefined) {
        return false;
    }
    if (withTypeCast(source.default, (0, exports.getColumnType)(source)) === target.default ||
        withTypeCast(target.default, (0, exports.getColumnType)(target)) === source.default) {
        return true;
    }
    return false;
};
exports.isDefaultEqual = isDefaultEqual;
const getColumnType = (column) => {
    let type = column.enumName || column.type;
    if (column.isArray) {
        type += `[${column.length ?? ''}]`;
    }
    else if (column.length !== undefined) {
        type += `(${column.length})`;
    }
    return type;
};
exports.getColumnType = getColumnType;
const withTypeCast = (value, type) => {
    if (!value.startsWith(`'`)) {
        value = `'${value}'`;
    }
    return `${value}::${type}`;
};
const getColumnModifiers = (column) => {
    const modifiers = [];
    if (!column.nullable) {
        modifiers.push('NOT NULL');
    }
    if (column.default) {
        modifiers.push(`DEFAULT ${column.default}`);
    }
    if (column.identity) {
        modifiers.push(`GENERATED ALWAYS AS IDENTITY`);
    }
    return modifiers.length === 0 ? '' : ' ' + modifiers.join(' ');
};
exports.getColumnModifiers = getColumnModifiers;
const asColumnComment = (tableName, columnName, comment) => {
    return `COMMENT ON COLUMN "${tableName}"."${columnName}" IS '${comment}';`;
};
exports.asColumnComment = asColumnComment;
const asColumnList = (columns) => columns.map((column) => `"${column}"`).join(', ');
exports.asColumnList = asColumnList;
const asJsonString = (value) => {
    return `'${escape(JSON.stringify(value))}'::jsonb`;
};
exports.asJsonString = asJsonString;
const escape = (value) => {
    return value
        .replaceAll("'", "''")
        .replaceAll(/[\\]/g, '\\\\')
        .replaceAll(/[\b]/g, String.raw `\b`)
        .replaceAll(/[\f]/g, String.raw `\f`)
        .replaceAll(/[\n]/g, String.raw `\n`)
        .replaceAll(/[\r]/g, String.raw `\r`)
        .replaceAll(/[\t]/g, String.raw `\t`);
};
const asRenameKey = (values) => values.map((value) => value ?? '').join('|');
exports.asRenameKey = asRenameKey;
//# sourceMappingURL=helpers.js.map