"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.isAssetChecksumConstraint = exports.ASSET_CHECKSUM_CONSTRAINT = exports.removeUndefinedKeys = exports.unnest = exports.asVector = exports.anyUuid = exports.asUuid = exports.getKyselyConfig = exports.asPostgresConnectionConfig = void 0;
exports.toJson = toJson;
exports.withDefaultVisibility = withDefaultVisibility;
exports.withExif = withExif;
exports.withExifInner = withExifInner;
exports.withSmartSearch = withSmartSearch;
exports.withFaces = withFaces;
exports.withFiles = withFiles;
exports.withFilePath = withFilePath;
exports.withFacesAndPeople = withFacesAndPeople;
exports.hasPeople = hasPeople;
exports.inAlbums = inAlbums;
exports.hasTags = hasTags;
exports.withOwner = withOwner;
exports.withLibrary = withLibrary;
exports.withTags = withTags;
exports.truncatedDate = truncatedDate;
exports.withTagId = withTagId;
exports.searchAssetBuilder = searchAssetBuilder;
exports.vectorIndexQuery = vectorIndexQuery;
const kysely_1 = require("kysely");
const kysely_postgres_js_1 = require("kysely-postgres-js");
const postgres_1 = require("kysely/helpers/postgres");
const pg_connection_string_1 = require("pg-connection-string");
const postgres_2 = __importDefault(require("postgres"));
const database_1 = require("../database");
const enum_1 = require("../enum");
const isValidSsl = (ssl) => typeof ssl !== 'string' || ssl === 'require' || ssl === 'allow' || ssl === 'prefer' || ssl === 'verify-full';
const asPostgresConnectionConfig = (params) => {
    if (params.connectionType === 'parts') {
        return {
            host: params.host,
            port: params.port,
            username: params.username,
            password: params.password,
            database: params.database,
            ssl: params.ssl === enum_1.DatabaseSslMode.Disable ? false : params.ssl,
        };
    }
    const { host, port, user, password, database, ...rest } = (0, pg_connection_string_1.parse)(params.url);
    let ssl;
    if (rest.ssl) {
        if (!isValidSsl(rest.ssl)) {
            throw new Error(`Invalid ssl option: ${rest.ssl}`);
        }
        ssl = rest.ssl;
    }
    return {
        host: host ?? undefined,
        port: port ? Number(port) : undefined,
        username: user,
        password,
        database: database ?? undefined,
        ssl,
    };
};
exports.asPostgresConnectionConfig = asPostgresConnectionConfig;
const getKyselyConfig = (params, options = {}) => {
    const config = (0, exports.asPostgresConnectionConfig)(params);
    return {
        dialect: new kysely_postgres_js_1.PostgresJSDialect({
            postgres: (0, postgres_2.default)({
                onnotice: (notice) => {
                    if (notice['severity'] !== 'NOTICE') {
                        console.warn('Postgres notice:', notice);
                    }
                },
                max: 10,
                types: {
                    date: {
                        to: 1184,
                        from: [1082, 1114, 1184],
                        serialize: (x) => (x instanceof Date ? x.toISOString() : x),
                        parse: (x) => new Date(x),
                    },
                    bigint: {
                        to: 20,
                        from: [20, 1700],
                        parse: (value) => Number.parseInt(value),
                        serialize: (value) => value.toString(),
                    },
                },
                connection: {
                    TimeZone: 'UTC',
                },
                host: config.host,
                port: config.port,
                username: config.username,
                password: config.password,
                database: config.database,
                ssl: config.ssl,
                ...options,
            }),
        }),
        log(event) {
            if (event.level === 'error') {
                console.error('Query failed :', {
                    durationMs: event.queryDurationMillis,
                    error: event.error,
                    sql: event.query.sql,
                    params: event.query.parameters,
                });
            }
        },
    };
};
exports.getKyselyConfig = getKyselyConfig;
const asUuid = (id) => (0, kysely_1.sql) `${id}::uuid`;
exports.asUuid = asUuid;
const anyUuid = (ids) => (0, kysely_1.sql) `any(${`{${ids}}`}::uuid[])`;
exports.anyUuid = anyUuid;
const asVector = (embedding) => (0, kysely_1.sql) `${`[${embedding}]`}::vector`;
exports.asVector = asVector;
const unnest = (array) => (0, kysely_1.sql) `unnest(array[${kysely_1.sql.join(array)}]::text[])`;
exports.unnest = unnest;
const removeUndefinedKeys = (update, template) => {
    for (const key in update) {
        if (template[key] === undefined) {
            delete update[key];
        }
    }
    return update;
};
exports.removeUndefinedKeys = removeUndefinedKeys;
function toJson(eb, table) {
    return eb.fn.toJson(table);
}
exports.ASSET_CHECKSUM_CONSTRAINT = 'UQ_assets_owner_checksum';
const isAssetChecksumConstraint = (error) => {
    return error?.constraint_name === 'UQ_assets_owner_checksum';
};
exports.isAssetChecksumConstraint = isAssetChecksumConstraint;
function withDefaultVisibility(qb) {
    return qb.where('asset.visibility', 'in', [kysely_1.sql.lit(enum_1.AssetVisibility.Archive), kysely_1.sql.lit(enum_1.AssetVisibility.Timeline)]);
}
function withExif(qb) {
    return qb
        .leftJoin('asset_exif', 'asset.id', 'asset_exif.assetId')
        .select((eb) => eb.fn.toJson(eb.table('asset_exif')).$castTo().as('exifInfo'));
}
function withExifInner(qb) {
    return qb
        .innerJoin('asset_exif', 'asset.id', 'asset_exif.assetId')
        .select((eb) => eb.fn.toJson(eb.table('asset_exif')).$castTo().as('exifInfo'));
}
function withSmartSearch(qb) {
    return qb
        .leftJoin('smart_search', 'asset.id', 'smart_search.assetId')
        .select((eb) => toJson(eb, 'smart_search').as('smartSearch'));
}
function withFaces(eb, withDeletedFace) {
    return (0, postgres_1.jsonArrayFrom)(eb
        .selectFrom('asset_face')
        .selectAll('asset_face')
        .whereRef('asset_face.assetId', '=', 'asset.id')
        .$if(!withDeletedFace, (qb) => qb.where('asset_face.deletedAt', 'is', null))).as('faces');
}
function withFiles(eb, type) {
    return (0, postgres_1.jsonArrayFrom)(eb
        .selectFrom('asset_file')
        .select(database_1.columns.assetFiles)
        .whereRef('asset_file.assetId', '=', 'asset.id')
        .$if(!!type, (qb) => qb.where('asset_file.type', '=', type))).as('files');
}
function withFilePath(eb, type) {
    return eb
        .selectFrom('asset_file')
        .select('asset_file.path')
        .whereRef('asset_file.assetId', '=', 'asset.id')
        .where('asset_file.type', '=', type);
}
function withFacesAndPeople(eb, withDeletedFace) {
    return (0, postgres_1.jsonArrayFrom)(eb
        .selectFrom('asset_face')
        .leftJoinLateral((eb) => eb.selectFrom('person').selectAll('person').whereRef('asset_face.personId', '=', 'person.id').as('person'), (join) => join.onTrue())
        .selectAll('asset_face')
        .select((eb) => eb.table('person').$castTo().as('person'))
        .whereRef('asset_face.assetId', '=', 'asset.id')
        .$if(!withDeletedFace, (qb) => qb.where('asset_face.deletedAt', 'is', null))).as('faces');
}
function hasPeople(qb, personIds) {
    return qb.innerJoin((eb) => eb
        .selectFrom('asset_face')
        .select('assetId')
        .where('personId', '=', (0, exports.anyUuid)(personIds))
        .where('deletedAt', 'is', null)
        .groupBy('assetId')
        .having((eb) => eb.fn.count('personId').distinct(), '=', personIds.length)
        .as('has_people'), (join) => join.onRef('has_people.assetId', '=', 'asset.id'));
}
function inAlbums(qb, albumIds) {
    return qb.innerJoin((eb) => eb
        .selectFrom('album_asset')
        .select('assetId')
        .where('albumId', '=', (0, exports.anyUuid)(albumIds))
        .groupBy('assetId')
        .having((eb) => eb.fn.count('albumId').distinct(), '=', albumIds.length)
        .as('has_album'), (join) => join.onRef('has_album.assetId', '=', 'asset.id'));
}
function hasTags(qb, tagIds) {
    return qb.innerJoin((eb) => eb
        .selectFrom('tag_asset')
        .select('assetId')
        .innerJoin('tag_closure', 'tag_asset.tagId', 'tag_closure.id_descendant')
        .where('tag_closure.id_ancestor', '=', (0, exports.anyUuid)(tagIds))
        .groupBy('assetId')
        .having((eb) => eb.fn.count('tag_closure.id_ancestor').distinct(), '>=', tagIds.length)
        .as('has_tags'), (join) => join.onRef('has_tags.assetId', '=', 'asset.id'));
}
function withOwner(eb) {
    return (0, postgres_1.jsonObjectFrom)(eb.selectFrom('user').select(database_1.columns.user).whereRef('user.id', '=', 'asset.ownerId')).as('owner');
}
function withLibrary(eb) {
    return (0, postgres_1.jsonObjectFrom)(eb.selectFrom('library').selectAll('library').whereRef('library.id', '=', 'asset.libraryId')).as('library');
}
function withTags(eb) {
    return (0, postgres_1.jsonArrayFrom)(eb
        .selectFrom('tag')
        .select(database_1.columns.tag)
        .innerJoin('tag_asset', 'tag.id', 'tag_asset.tagId')
        .whereRef('asset.id', '=', 'tag_asset.assetId')).as('tags');
}
function truncatedDate() {
    return (0, kysely_1.sql) `date_trunc(${kysely_1.sql.lit('MONTH')}, "localDateTime" AT TIME ZONE 'UTC') AT TIME ZONE 'UTC'`;
}
function withTagId(qb, tagId) {
    return qb.where((eb) => eb.exists(eb
        .selectFrom('tag_closure')
        .innerJoin('tag_asset', 'tag_asset.tagId', 'tag_closure.id_descendant')
        .whereRef('tag_asset.assetId', '=', 'asset.id')
        .where('tag_closure.id_ancestor', '=', tagId)));
}
const joinDeduplicationPlugin = new kysely_1.DeduplicateJoinsPlugin();
function searchAssetBuilder(kysely, options) {
    options.withDeleted ||= !!(options.trashedAfter || options.trashedBefore || options.isOffline);
    const visibility = options.visibility == null ? enum_1.AssetVisibility.Timeline : options.visibility;
    return kysely
        .withPlugin(joinDeduplicationPlugin)
        .selectFrom('asset')
        .where('asset.visibility', '=', visibility)
        .$if(!!options.albumIds && options.albumIds.length > 0, (qb) => inAlbums(qb, options.albumIds))
        .$if(!!options.tagIds && options.tagIds.length > 0, (qb) => hasTags(qb, options.tagIds))
        .$if(options.tagIds === null, (qb) => qb.where((eb) => eb.not(eb.exists((eb) => eb.selectFrom('tag_asset').whereRef('assetId', '=', 'asset.id')))))
        .$if(!!options.personIds && options.personIds.length > 0, (qb) => hasPeople(qb, options.personIds))
        .$if(!!options.createdBefore, (qb) => qb.where('asset.createdAt', '<=', options.createdBefore))
        .$if(!!options.createdAfter, (qb) => qb.where('asset.createdAt', '>=', options.createdAfter))
        .$if(!!options.updatedBefore, (qb) => qb.where('asset.updatedAt', '<=', options.updatedBefore))
        .$if(!!options.updatedAfter, (qb) => qb.where('asset.updatedAt', '>=', options.updatedAfter))
        .$if(!!options.trashedBefore, (qb) => qb.where('asset.deletedAt', '<=', options.trashedBefore))
        .$if(!!options.trashedAfter, (qb) => qb.where('asset.deletedAt', '>=', options.trashedAfter))
        .$if(!!options.takenBefore, (qb) => qb.where('asset.fileCreatedAt', '<=', options.takenBefore))
        .$if(!!options.takenAfter, (qb) => qb.where('asset.fileCreatedAt', '>=', options.takenAfter))
        .$if(options.city !== undefined, (qb) => qb
        .innerJoin('asset_exif', 'asset.id', 'asset_exif.assetId')
        .where('asset_exif.city', options.city === null ? 'is' : '=', options.city))
        .$if(options.state !== undefined, (qb) => qb
        .innerJoin('asset_exif', 'asset.id', 'asset_exif.assetId')
        .where('asset_exif.state', options.state === null ? 'is' : '=', options.state))
        .$if(options.country !== undefined, (qb) => qb
        .innerJoin('asset_exif', 'asset.id', 'asset_exif.assetId')
        .where('asset_exif.country', options.country === null ? 'is' : '=', options.country))
        .$if(options.make !== undefined, (qb) => qb
        .innerJoin('asset_exif', 'asset.id', 'asset_exif.assetId')
        .where('asset_exif.make', options.make === null ? 'is' : '=', options.make))
        .$if(options.model !== undefined, (qb) => qb
        .innerJoin('asset_exif', 'asset.id', 'asset_exif.assetId')
        .where('asset_exif.model', options.model === null ? 'is' : '=', options.model))
        .$if(options.lensModel !== undefined, (qb) => qb
        .innerJoin('asset_exif', 'asset.id', 'asset_exif.assetId')
        .where('asset_exif.lensModel', options.lensModel === null ? 'is' : '=', options.lensModel))
        .$if(options.rating !== undefined, (qb) => qb
        .innerJoin('asset_exif', 'asset.id', 'asset_exif.assetId')
        .where('asset_exif.rating', options.rating === null ? 'is' : '=', options.rating))
        .$if(!!options.checksum, (qb) => qb.where('asset.checksum', '=', options.checksum))
        .$if(!!options.deviceAssetId, (qb) => qb.where('asset.deviceAssetId', '=', options.deviceAssetId))
        .$if(!!options.deviceId, (qb) => qb.where('asset.deviceId', '=', options.deviceId))
        .$if(!!options.id, (qb) => qb.where('asset.id', '=', (0, exports.asUuid)(options.id)))
        .$if(!!options.libraryId, (qb) => qb.where('asset.libraryId', '=', (0, exports.asUuid)(options.libraryId)))
        .$if(!!options.userIds, (qb) => qb.where('asset.ownerId', '=', (0, exports.anyUuid)(options.userIds)))
        .$if(!!options.encodedVideoPath, (qb) => qb.where('asset.encodedVideoPath', '=', options.encodedVideoPath))
        .$if(!!options.originalPath, (qb) => qb.where((0, kysely_1.sql) `f_unaccent(asset."originalPath")`, 'ilike', (0, kysely_1.sql) `'%' || f_unaccent(${options.originalPath}) || '%'`))
        .$if(!!options.originalFileName, (qb) => qb.where((0, kysely_1.sql) `f_unaccent(asset."originalFileName")`, 'ilike', (0, kysely_1.sql) `'%' || f_unaccent(${options.originalFileName}) || '%'`))
        .$if(!!options.description, (qb) => qb
        .innerJoin('asset_exif', 'asset.id', 'asset_exif.assetId')
        .where((0, kysely_1.sql) `f_unaccent(asset_exif.description)`, 'ilike', (0, kysely_1.sql) `'%' || f_unaccent(${options.description}) || '%'`))
        .$if(!!options.ocr, (qb) => qb
        .innerJoin('ocr_search', 'asset.id', 'ocr_search.assetId')
        .where(() => (0, kysely_1.sql) `f_unaccent(ocr_search.text) %>> f_unaccent(${options.ocr})`))
        .$if(!!options.type, (qb) => qb.where('asset.type', '=', options.type))
        .$if(options.isFavorite !== undefined, (qb) => qb.where('asset.isFavorite', '=', options.isFavorite))
        .$if(options.isOffline !== undefined, (qb) => qb.where('asset.isOffline', '=', options.isOffline))
        .$if(options.isEncoded !== undefined, (qb) => qb.where('asset.encodedVideoPath', options.isEncoded ? 'is not' : 'is', null))
        .$if(options.isMotion !== undefined, (qb) => qb.where('asset.livePhotoVideoId', options.isMotion ? 'is not' : 'is', null))
        .$if(!!options.isNotInAlbum && (!options.albumIds || options.albumIds.length === 0), (qb) => qb.where((eb) => eb.not(eb.exists((eb) => eb.selectFrom('album_asset').whereRef('assetId', '=', 'asset.id')))))
        .$if(!!options.withExif, withExifInner)
        .$if(!!(options.withFaces || options.withPeople || options.personIds), (qb) => qb.select(withFacesAndPeople))
        .$if(!options.withDeleted, (qb) => qb.where('asset.deletedAt', 'is', null));
}
function vectorIndexQuery({ vectorExtension, table, indexName, lists }) {
    switch (vectorExtension) {
        case enum_1.DatabaseExtension.VectorChord: {
            return `
        CREATE INDEX IF NOT EXISTS ${indexName} ON ${table} USING vchordrq (embedding vector_cosine_ops) WITH (options = $$
        residual_quantization = false
        [build.internal]
        lists = [${lists ?? 1}]
        spherical_centroids = true
        build_threads = 4
        sampling_factor = 1024
        $$)`;
        }
        case enum_1.DatabaseExtension.Vectors: {
            return `
        CREATE INDEX IF NOT EXISTS ${indexName} ON ${table}
        USING vectors (embedding vector_cos_ops) WITH (options = $$
        optimizing.optimizing_threads = 4
        [indexing.hnsw]
        m = 16
        ef_construction = 300
        $$)`;
        }
        case enum_1.DatabaseExtension.Vector: {
            return `
        CREATE INDEX IF NOT EXISTS ${indexName} ON ${table}
        USING hnsw (embedding vector_cosine_ops)
        WITH (ef_construction = 300, m = 16)`;
        }
        default: {
            throw new Error(`Unsupported vector extension: '${vectorExtension}'`);
        }
    }
}
//# sourceMappingURL=database.js.map