import { BatchClusterEmitter } from "./BatchClusterEmitter";
import { HealthCheckStrategy } from "./HealthCheckStrategy";
import { InternalBatchProcessOptions } from "./InternalBatchProcessOptions";
import { Task } from "./Task";
import { WhyNotHealthy, WhyNotReady } from "./WhyNotHealthy";
/**
 * Interface for objects that can be health checked
 */
export interface HealthCheckable {
    readonly pid: number;
    readonly start: number;
    readonly taskCount: number;
    readonly failedTaskCount: number;
    readonly idleMs: number;
    readonly idle: boolean;
    readonly ending: boolean;
    readonly ended: boolean;
    readonly proc: {
        stdin?: {
            destroyed?: boolean;
        } | null;
    };
    readonly currentTask?: Task<unknown> | null | undefined;
}
/**
 * Manages health checking logic for processes.
 * Provides centralized health assessment and monitoring capabilities.
 */
export declare class ProcessHealthMonitor {
    #private;
    private readonly options;
    private readonly emitter;
    private readonly healthStrategy;
    constructor(options: InternalBatchProcessOptions, emitter: BatchClusterEmitter, healthStrategy?: HealthCheckStrategy);
    /**
     * Initialize health monitoring for a process
     */
    initializeProcess(pid: number): void;
    /**
     * Clean up health monitoring for a process
     */
    cleanupProcess(pid: number): void;
    /**
     * Record that a job failed for a process
     */
    recordJobFailure(pid: number): void;
    /**
     * Record that a job succeeded for a process
     */
    recordJobSuccess(pid: number): void;
    /**
     * Assess the health of a process and return why it's not healthy, or null if healthy
     */
    assessHealth(process: HealthCheckable, overrideReason?: WhyNotHealthy): WhyNotHealthy | null;
    /**
     * Check if a process is healthy
     */
    isHealthy(process: HealthCheckable, overrideReason?: WhyNotHealthy): boolean;
    /**
     * Assess why a process is not ready (combines health and business)
     */
    assessReadiness(process: HealthCheckable, overrideReason?: WhyNotHealthy): WhyNotReady | null;
    /**
     * Check if a process is ready to handle tasks
     */
    isReady(process: HealthCheckable, overrideReason?: WhyNotHealthy): boolean;
    /**
     * Run a health check on a process if needed
     */
    maybeRunHealthCheck(process: HealthCheckable & {
        execTask: (task: Task<unknown>) => boolean;
    }): Task<unknown> | undefined;
    /**
     * Get health statistics for monitoring
     */
    getHealthStats(): {
        monitoredProcesses: number;
        totalHealthCheckFailures: number;
        processesWithFailures: number;
    };
    /**
     * Reset health check failures for a process (useful for recovery scenarios)
     */
    resetHealthCheckFailures(pid: number): void;
    /**
     * Get health check state for a specific process
     */
    getProcessHealthState(pid: number): {
        lastHealthCheck: number;
        healthCheckFailures: number;
        lastJobFailed: boolean;
    } | undefined;
}
