import { BatchClusterEmitter } from "./BatchClusterEmitter";
import { BatchProcess } from "./BatchProcess";
import { CombinedBatchProcessOptions } from "./CombinedBatchProcessOptions";
import { Task } from "./Task";
/**
 * Manages the lifecycle of a pool of BatchProcess instances.
 * Handles spawning, health monitoring, and cleanup of child processes.
 */
export declare class ProcessPoolManager {
    #private;
    private readonly options;
    private readonly emitter;
    private readonly onIdle;
    constructor(options: CombinedBatchProcessOptions, emitter: BatchClusterEmitter, onIdle: () => void);
    /**
     * Get all current processes
     */
    get processes(): readonly BatchProcess[];
    /**
     * Get the current number of spawned child processes
     */
    get procCount(): number;
    /**
     * Alias for procCount to match BatchCluster interface
     */
    get processCount(): number;
    /**
     * Get the current number of child processes currently servicing tasks
     */
    get busyProcCount(): number;
    /**
     * Get the current number of starting processes
     */
    get startingProcCount(): number;
    /**
     * Get the current number of ready processes
     */
    get readyProcCount(): number;
    /**
     * Get the total number of child processes created by this instance
     */
    get spawnedProcCount(): number;
    /**
     * Get the milliseconds until the next spawn is allowed
     */
    get msBeforeNextSpawn(): number;
    /**
     * Get all currently running tasks from all processes
     */
    currentTasks(): Task<unknown>[];
    /**
     * Find the first ready process that can handle a new task
     */
    findReadyProcess(): BatchProcess | undefined;
    /**
     * Verify that each BatchProcess PID is actually alive.
     * @return the spawned PIDs that are still in the process table.
     */
    pids(): number[];
    /**
     * Shut down any currently-running child processes.
     */
    closeChildProcesses(gracefully?: boolean): Promise<void>;
    /**
     * Run maintenance on currently spawned child processes.
     * Removes unhealthy processes and enforces maxProcs limit.
     */
    vacuumProcs(): Promise<void[]>;
    /**
     * Spawn new processes if needed based on pending task count and capacity
     */
    maybeSpawnProcs(pendingTaskCount: number, ended: boolean): Promise<void>;
    /**
     * Update the maximum number of processes allowed
     */
    setMaxProcs(maxProcs: number): void;
}
