#!/usr/bin/env node
import { createRequire } from "node:module";
import { program } from "commander";
import { spawn } from "node:child_process";
import fs, { existsSync, promises, statSync, unlinkSync, writeFileSync } from "node:fs";
import path from "node:path";
import logSymbols from "log-symbols";
import ora from "ora";
import url from "node:url";
import { createJiti } from "jiti";
import { addDevDependency } from "nypm";
import prompts from "prompts";
import { watch } from "chokidar";
import debounce from "debounce";
import { Server } from "socket.io";
import { parse } from "@babel/parser";
import traverseModule from "@babel/traverse";
import { createMatchPath, loadConfig } from "tsconfig-paths";
import http from "node:http";
import { styleText } from "node:util";
import { lookup } from "mime-types";
import os from "node:os";
import { build } from "esbuild";
import { glob } from "glob";
import normalize from "normalize-path";

//#region src/utils/get-emails-directory-metadata.ts
const isFileAnEmail = async (fullPath) => {
	let fileHandle;
	try {
		fileHandle = await fs.promises.open(fullPath, "r");
	} catch (exception) {
		console.warn(exception);
		return false;
	}
	if ((await fileHandle.stat()).isDirectory()) {
		await fileHandle.close();
		return false;
	}
	const { ext } = path.parse(fullPath);
	if (![
		".js",
		".tsx",
		".jsx"
	].includes(ext)) {
		await fileHandle.close();
		return false;
	}
	const fileContents = await fileHandle.readFile("utf8");
	await fileHandle.close();
	const hasES6DefaultExport = /\bexport\s+default\b/gm.test(fileContents);
	const hasCommonJSExport = /\bmodule\.exports\s*=/gm.test(fileContents);
	const hasNamedExport = /\bexport\s+\{[^}]*\bdefault\b[^}]*\}/gm.test(fileContents);
	return hasES6DefaultExport || hasCommonJSExport || hasNamedExport;
};
const mergeDirectoriesWithSubDirectories = (emailsDirectoryMetadata) => {
	let currentResultingMergedDirectory = emailsDirectoryMetadata;
	while (currentResultingMergedDirectory.emailFilenames.length === 0 && currentResultingMergedDirectory.subDirectories.length === 1) {
		const onlySubDirectory = currentResultingMergedDirectory.subDirectories[0];
		currentResultingMergedDirectory = {
			...onlySubDirectory,
			directoryName: path.join(currentResultingMergedDirectory.directoryName, onlySubDirectory.directoryName)
		};
	}
	return currentResultingMergedDirectory;
};
const getEmailsDirectoryMetadata = async (absolutePathToEmailsDirectory, keepFileExtensions = false, isSubDirectory = false, baseDirectoryPath = absolutePathToEmailsDirectory) => {
	if (!fs.existsSync(absolutePathToEmailsDirectory)) return;
	const dirents = await fs.promises.readdir(absolutePathToEmailsDirectory, { withFileTypes: true });
	const isEmailPredicates = await Promise.all(dirents.map((dirent) => isFileAnEmail(path.join(absolutePathToEmailsDirectory, dirent.name))));
	const emailFilenames = dirents.filter((_, i) => isEmailPredicates[i]).map((dirent) => keepFileExtensions ? dirent.name : dirent.name.replace(path.extname(dirent.name), ""));
	const subDirectories = await Promise.all(dirents.filter((dirent) => dirent.isDirectory() && !dirent.name.startsWith("_") && dirent.name !== "static").map((dirent) => {
		return getEmailsDirectoryMetadata(path.join(absolutePathToEmailsDirectory, dirent.name), keepFileExtensions, true, baseDirectoryPath);
	}));
	const emailsMetadata = {
		absolutePath: absolutePathToEmailsDirectory,
		relativePath: path.relative(baseDirectoryPath, absolutePathToEmailsDirectory),
		directoryName: absolutePathToEmailsDirectory.split(path.sep).pop(),
		emailFilenames,
		subDirectories
	};
	return isSubDirectory ? mergeDirectoriesWithSubDirectories(emailsMetadata) : emailsMetadata;
};

//#endregion
//#region package.json
var name = "react-email";
var version = "4.3.2";
var description = "A live preview of your emails right in your browser.";
var bin = { "email": "./dist/index.js" };
var type = "module";
var scripts = {
	"build": "tsdown",
	"build:watch": "tsdown --watch src",
	"clean": "rm -rf dist",
	"test": "vitest run",
	"test:watch": "vitest"
};
var license = "MIT";
var repository = {
	"type": "git",
	"url": "https://github.com/resend/react-email.git",
	"directory": "packages/react-email"
};
var keywords = ["react", "email"];
var engines = { "node": ">=18.0.0" };
var dependencies = {
	"@babel/parser": "^7.27.0",
	"@babel/traverse": "^7.27.0",
	"chokidar": "^4.0.3",
	"commander": "^13.0.0",
	"debounce": "^2.0.0",
	"esbuild": "^0.25.0",
	"glob": "^11.0.0",
	"jiti": "2.4.2",
	"log-symbols": "^7.0.0",
	"mime-types": "^3.0.0",
	"normalize-path": "^3.0.0",
	"nypm": "0.6.0",
	"ora": "^8.0.0",
	"prompts": "2.4.2",
	"socket.io": "^4.8.1",
	"tsconfig-paths": "4.2.0"
};
var devDependencies = {
	"@react-email/components": "workspace:*",
	"@types/babel__core": "7.20.5",
	"@types/babel__traverse": "7.20.7",
	"@types/mime-types": "2.1.4",
	"@types/prompts": "2.4.9",
	"next": "^15.3.2",
	"react": "19.0.0",
	"react-dom": "19.0.0",
	"typescript": "5.8.3"
};
var package_default = {
	name,
	version,
	description,
	bin,
	type,
	scripts,
	license,
	repository,
	keywords,
	engines,
	dependencies,
	devDependencies
};

//#endregion
//#region src/utils/get-preview-server-location.ts
const ensurePreviewServerInstalled = async (message) => {
	if ((await prompts({
		type: "confirm",
		name: "installPreviewServer",
		message,
		initial: true
	})).installPreviewServer) {
		console.log("Installing \"@react-email/preview-server\"");
		await addDevDependency(`@react-email/preview-server@${package_default.version}`);
		process.exit(0);
	} else process.exit(0);
};
const getPreviewServerLocation = async () => {
	const usersProject = createJiti(process.cwd());
	let previewServerLocation;
	try {
		previewServerLocation = path.dirname(url.fileURLToPath(usersProject.esmResolve("@react-email/preview-server")));
	} catch (_exception) {
		await ensurePreviewServerInstalled("To run the preview server, the package \"@react-email/preview-server\" must be installed. Would you like to install it?");
	}
	const { version: version$1 } = await usersProject.import("@react-email/preview-server");
	if (version$1 !== package_default.version) await ensurePreviewServerInstalled(`To run the preview server, the version of "@react-email/preview-server" must match the version of "react-email" (${package_default.version}). Would you like to install it?`);
	return previewServerLocation;
};

//#endregion
//#region src/utils/register-spinner-autostopping.ts
const spinners = /* @__PURE__ */ new Set();
process.on("SIGINT", () => {
	spinners.forEach((spinner) => {
		if (spinner.isSpinning) spinner.stop();
	});
});
process.on("exit", (code) => {
	if (code !== 0) spinners.forEach((spinner) => {
		if (spinner.isSpinning) spinner.stopAndPersist({ symbol: logSymbols.error });
	});
});
const registerSpinnerAutostopping = (spinner) => {
	spinners.add(spinner);
};

//#endregion
//#region src/commands/build.ts
const buildPreviewApp = (absoluteDirectory) => {
	return new Promise((resolve, reject) => {
		const nextBuild = spawn("npm", ["run", "build"], {
			cwd: absoluteDirectory,
			shell: true
		});
		nextBuild.stdout.pipe(process.stdout);
		nextBuild.stderr.pipe(process.stderr);
		nextBuild.on("close", (code) => {
			if (code === 0) resolve();
			else reject(/* @__PURE__ */ new Error(`Unable to build the Next app and it exited with code: ${code}`));
		});
	});
};
const npmInstall = async (builtPreviewAppPath, packageManager) => {
	return new Promise((resolve, reject) => {
		const childProc = spawn(packageManager, [
			"install",
			packageManager === "deno" ? "" : "--include=dev",
			packageManager === "deno" ? "--quiet" : "--silent"
		], {
			cwd: builtPreviewAppPath,
			shell: true
		});
		childProc.stdout.pipe(process.stdout);
		childProc.stderr.pipe(process.stderr);
		childProc.on("close", (code) => {
			if (code === 0) resolve();
			else reject(/* @__PURE__ */ new Error(`Unable to install the dependencies and it exited with code: ${code}`));
		});
	});
};
const setNextEnvironmentVariablesForBuild = async (emailsDirRelativePath, builtPreviewAppPath) => {
	const nextConfigContents = `
const path = require('path');
const emailsDirRelativePath = path.normalize('${emailsDirRelativePath}');
const userProjectLocation = '${process.cwd().replace(/\\/g, "/")}';
/** @type {import('next').NextConfig} */
module.exports = {
  env: {
    NEXT_PUBLIC_IS_BUILDING: 'true',
    EMAILS_DIR_RELATIVE_PATH: emailsDirRelativePath,
    EMAILS_DIR_ABSOLUTE_PATH: path.resolve(userProjectLocation, emailsDirRelativePath),
    PREVIEW_SERVER_LOCATION: '${builtPreviewAppPath.replace(/\\/g, "/")}',
    USER_PROJECT_LOCATION: userProjectLocation
  },
  // this is needed so that the code for building emails works properly
  webpack: (
    /** @type {import('webpack').Configuration & { externals: string[] }} */
    config,
    { isServer }
  ) => {
    if (isServer) {
      config.externals.push('esbuild');
    }

    return config;
  },
  typescript: {
    ignoreBuildErrors: true
  },
  eslint: {
    ignoreDuringBuilds: true
  },
  experimental: {
    webpackBuildWorker: true
  },
}`;
	await fs.promises.writeFile(path.resolve(builtPreviewAppPath, "./next.config.js"), nextConfigContents, "utf8");
};
const getEmailSlugsFromEmailDirectory = (emailDirectory, emailsDirectoryAbsolutePath) => {
	const directoryPathRelativeToEmailsDirectory = emailDirectory.absolutePath.replace(emailsDirectoryAbsolutePath, "").trim();
	const slugs = [];
	emailDirectory.emailFilenames.forEach((filename) => slugs.push(path.join(directoryPathRelativeToEmailsDirectory, filename).split(path.sep).filter((segment) => segment.length > 0)));
	emailDirectory.subDirectories.forEach((directory) => {
		slugs.push(...getEmailSlugsFromEmailDirectory(directory, emailsDirectoryAbsolutePath));
	});
	return slugs;
};
const forceSSGForEmailPreviews = async (emailsDirPath, builtPreviewAppPath) => {
	const parameters = getEmailSlugsFromEmailDirectory(await getEmailsDirectoryMetadata(emailsDirPath), emailsDirPath).map((slug) => ({ slug }));
	const removeForceDynamic = async (filePath) => {
		const contents = await fs.promises.readFile(filePath, "utf8");
		await fs.promises.writeFile(filePath, contents.replace("export const dynamic = 'force-dynamic';", ""), "utf8");
	};
	await removeForceDynamic(path.resolve(builtPreviewAppPath, "./src/app/layout.tsx"));
	await removeForceDynamic(path.resolve(builtPreviewAppPath, "./src/app/preview/[...slug]/page.tsx"));
	await fs.promises.appendFile(path.resolve(builtPreviewAppPath, "./src/app/preview/[...slug]/page.tsx"), `

export function generateStaticParams() { 
  return Promise.resolve(
    ${JSON.stringify(parameters)}
  );
}`, "utf8");
};
const updatePackageJson = async (builtPreviewAppPath) => {
	const packageJsonPath = path.resolve(builtPreviewAppPath, "./package.json");
	const packageJson = JSON.parse(await fs.promises.readFile(packageJsonPath, "utf8"));
	packageJson.scripts.build = "next build";
	packageJson.scripts.start = "next start";
	delete packageJson.scripts.postbuild;
	packageJson.name = "preview-server";
	delete packageJson.devDependencies["@react-email/render"];
	delete packageJson.devDependencies["@react-email/components"];
	delete packageJson.scripts.prepare;
	await fs.promises.writeFile(packageJsonPath, JSON.stringify(packageJson), "utf8");
};
const build$1 = async ({ dir: emailsDirRelativePath, packageManager }) => {
	try {
		const previewServerLocation = await getPreviewServerLocation();
		const spinner = ora({
			text: "Starting build process...",
			prefixText: "  "
		}).start();
		registerSpinnerAutostopping(spinner);
		spinner.text = `Checking if ${emailsDirRelativePath} folder exists`;
		if (!fs.existsSync(emailsDirRelativePath)) process.exit(1);
		const emailsDirPath = path.join(process.cwd(), emailsDirRelativePath);
		const staticPath = path.join(emailsDirPath, "static");
		const builtPreviewAppPath = path.join(process.cwd(), ".react-email");
		if (fs.existsSync(builtPreviewAppPath)) {
			spinner.text = "Deleting pre-existing `.react-email` folder";
			await fs.promises.rm(builtPreviewAppPath, { recursive: true });
		}
		spinner.text = "Copying preview app from CLI to `.react-email`";
		await fs.promises.cp(previewServerLocation, builtPreviewAppPath, {
			recursive: true,
			filter: (source) => {
				return !/(\/|\\)cli(\/|\\)?/.test(source) && !/(\/|\\)\.next(\/|\\)?/.test(source) && !/(\/|\\)\.turbo(\/|\\)?/.test(source) && !/(\/|\\)node_modules(\/|\\)?$/.test(source);
			}
		});
		if (fs.existsSync(staticPath)) {
			spinner.text = "Copying `static` folder into `.react-email/public/static`";
			const builtStaticDirectory = path.resolve(builtPreviewAppPath, "./public/static");
			await fs.promises.cp(staticPath, builtStaticDirectory, { recursive: true });
		}
		spinner.text = "Setting Next environment variables for preview app to work properly";
		await setNextEnvironmentVariablesForBuild(emailsDirRelativePath, builtPreviewAppPath);
		spinner.text = "Setting server side generation for the email preview pages";
		await forceSSGForEmailPreviews(emailsDirPath, builtPreviewAppPath);
		spinner.text = "Updating package.json's build and start scripts";
		await updatePackageJson(builtPreviewAppPath);
		spinner.text = "Installing dependencies on `.react-email`";
		await npmInstall(builtPreviewAppPath, packageManager);
		spinner.stopAndPersist({
			text: "Successfully prepared `.react-email` for `next build`",
			symbol: logSymbols.success
		});
		await buildPreviewApp(builtPreviewAppPath);
	} catch (error) {
		console.log(error);
		process.exit(1);
	}
};

//#endregion
//#region src/utils/preview/hot-reloading/get-imported-modules.ts
const traverse = typeof traverseModule === "function" ? traverseModule : traverseModule.default;
const getImportedModules = (contents) => {
	const importedPaths = [];
	traverse(parse(contents, {
		sourceType: "unambiguous",
		strictMode: false,
		errorRecovery: true,
		plugins: [
			"jsx",
			"typescript",
			"decorators"
		]
	}), {
		ImportDeclaration({ node }) {
			importedPaths.push(node.source.value);
		},
		ExportAllDeclaration({ node }) {
			importedPaths.push(node.source.value);
		},
		ExportNamedDeclaration({ node }) {
			if (node.source) importedPaths.push(node.source.value);
		},
		TSExternalModuleReference({ node }) {
			importedPaths.push(node.expression.value);
		},
		CallExpression({ node }) {
			if ("name" in node.callee && node.callee.name === "require") {
				if (node.arguments.length === 1) {
					const importPathNode = node.arguments[0];
					if (importPathNode.type === "StringLiteral") importedPaths.push(importPathNode.value);
				}
			}
		}
	});
	return importedPaths;
};

//#endregion
//#region src/utils/preview/hot-reloading/resolve-path-aliases.ts
const resolvePathAliases = (importPaths, projectPath) => {
	const configLoadResult = loadConfig(projectPath);
	if (configLoadResult.resultType === "success") {
		const matchPath = createMatchPath(configLoadResult.absoluteBaseUrl, configLoadResult.paths);
		return importPaths.map((importedPath) => {
			const unaliasedPath = matchPath(importedPath, void 0, void 0, [
				".tsx",
				".ts",
				".js",
				".jsx",
				".cjs",
				".mjs"
			]);
			if (unaliasedPath) return `./${path.relative(projectPath, unaliasedPath)}`;
			return importedPath;
		});
	}
	return importPaths;
};

//#endregion
//#region src/utils/preview/hot-reloading/create-dependency-graph.ts
const readAllFilesInsideDirectory = async (directory) => {
	let allFilePaths = [];
	const topLevelDirents = await promises.readdir(directory, { withFileTypes: true });
	for await (const dirent of topLevelDirents) {
		const pathToDirent = path.join(directory, dirent.name);
		if (dirent.isDirectory()) allFilePaths = allFilePaths.concat(await readAllFilesInsideDirectory(pathToDirent));
		else allFilePaths.push(pathToDirent);
	}
	return allFilePaths;
};
const javascriptExtensions = [
	".js",
	".ts",
	".jsx",
	".tsx",
	".mjs",
	".cjs"
];
const isJavascriptModule = (filePath) => {
	const extensionName = path.extname(filePath);
	return javascriptExtensions.includes(extensionName);
};
const checkFileExtensionsUntilItExists = (pathWithoutExtension) => {
	if (existsSync(`${pathWithoutExtension}.ts`)) return `${pathWithoutExtension}.ts`;
	if (existsSync(`${pathWithoutExtension}.tsx`)) return `${pathWithoutExtension}.tsx`;
	if (existsSync(`${pathWithoutExtension}.js`)) return `${pathWithoutExtension}.js`;
	if (existsSync(`${pathWithoutExtension}.jsx`)) return `${pathWithoutExtension}.jsx`;
	if (existsSync(`${pathWithoutExtension}.mjs`)) return `${pathWithoutExtension}.mjs`;
	if (existsSync(`${pathWithoutExtension}.cjs`)) return `${pathWithoutExtension}.cjs`;
};
/**
* Creates a stateful dependency graph that is structured in a way that you can get
* the dependents of a module from its path.
*
* Stateful in the sense that it provides a `getter` and an "`updater`". The updater
* will receive changes to the files, that can be perceived through some file watching mechanism,
* so that it doesn't need to recompute the entire dependency graph but only the parts changed.
*/
const createDependencyGraph = async (directory) => {
	const modulePaths = (await readAllFilesInsideDirectory(directory)).filter(isJavascriptModule);
	const graph = Object.fromEntries(modulePaths.map((path$1) => [path$1, {
		path: path$1,
		dependencyPaths: [],
		dependentPaths: [],
		moduleDependencies: []
	}]));
	const getDependencyPaths = async (filePath) => {
		const contents = await promises.readFile(filePath, "utf8");
		const importedPathsRelativeToDirectory = (isJavascriptModule(filePath) ? resolvePathAliases(getImportedModules(contents), path.dirname(filePath)) : []).map((dependencyPath) => {
			if (!dependencyPath.startsWith(".") || path.isAbsolute(dependencyPath)) return dependencyPath;
			let pathToDependencyFromDirectory = path.resolve(path.dirname(filePath), dependencyPath);
			let isDirectory = false;
			try {
				isDirectory = statSync(pathToDependencyFromDirectory).isDirectory();
			} catch (_) {}
			if (isDirectory) {
				const pathWithExtension = checkFileExtensionsUntilItExists(`${pathToDependencyFromDirectory}/index`);
				if (pathWithExtension) pathToDependencyFromDirectory = pathWithExtension;
				else console.warn(`Could not find index file for directory at ${pathToDependencyFromDirectory}. This is probably going to cause issues with both hot reloading and your code.`);
			}
			const extension = path.extname(pathToDependencyFromDirectory);
			const pathWithEnsuredExtension = (() => {
				if (extension.length > 0 && existsSync(pathToDependencyFromDirectory)) return pathToDependencyFromDirectory;
				if (javascriptExtensions.includes(extension)) return checkFileExtensionsUntilItExists(pathToDependencyFromDirectory.replace(extension, ""));
				return checkFileExtensionsUntilItExists(pathToDependencyFromDirectory);
			})();
			if (pathWithEnsuredExtension) pathToDependencyFromDirectory = pathWithEnsuredExtension;
			else console.warn(`Could not find file at ${pathToDependencyFromDirectory}`);
			return pathToDependencyFromDirectory;
		});
		const moduleDependencies = importedPathsRelativeToDirectory.filter((dependencyPath) => !dependencyPath.startsWith(".") && !path.isAbsolute(dependencyPath));
		return {
			dependencyPaths: importedPathsRelativeToDirectory.filter((dependencyPath) => dependencyPath.startsWith(".") || path.isAbsolute(dependencyPath)),
			moduleDependencies
		};
	};
	const updateModuleDependenciesInGraph = async (moduleFilePath) => {
		if (graph[moduleFilePath] === void 0) graph[moduleFilePath] = {
			path: moduleFilePath,
			dependencyPaths: [],
			dependentPaths: [],
			moduleDependencies: []
		};
		const { moduleDependencies, dependencyPaths: newDependencyPaths } = await getDependencyPaths(moduleFilePath);
		graph[moduleFilePath].moduleDependencies = moduleDependencies;
		for (const dependencyPath of graph[moduleFilePath].dependencyPaths) {
			if (newDependencyPaths.includes(dependencyPath)) continue;
			const dependencyModule = graph[dependencyPath];
			if (dependencyModule !== void 0) dependencyModule.dependentPaths = dependencyModule.dependentPaths.filter((dependentPath) => dependentPath !== moduleFilePath);
		}
		graph[moduleFilePath].dependencyPaths = newDependencyPaths;
		for await (const dependencyPath of newDependencyPaths) {
			if (graph[dependencyPath] === void 0) await updateModuleDependenciesInGraph(dependencyPath);
			const dependencyModule = graph[dependencyPath];
			if (dependencyModule === void 0) throw new Error(`Loading the dependency path ${dependencyPath} did not initialize it at all. This is a bug in React Email.`);
			if (!dependencyModule.dependentPaths.includes(moduleFilePath)) dependencyModule.dependentPaths.push(moduleFilePath);
		}
	};
	for (const filePath of modulePaths) await updateModuleDependenciesInGraph(filePath);
	const removeModuleFromGraph = (filePath) => {
		const module = graph[filePath];
		if (module) {
			for (const dependencyPath of module.dependencyPaths) if (graph[dependencyPath]) graph[dependencyPath].dependentPaths = graph[dependencyPath].dependentPaths.filter((dependentPath) => dependentPath !== filePath);
			delete graph[filePath];
		}
	};
	return [
		graph,
		async (event, pathToModified) => {
			switch (event) {
				case "change":
					if (isJavascriptModule(pathToModified)) await updateModuleDependenciesInGraph(pathToModified);
					break;
				case "add":
					if (isJavascriptModule(pathToModified)) await updateModuleDependenciesInGraph(pathToModified);
					break;
				case "addDir": {
					const modulesInsideAddedDirectory = (await readAllFilesInsideDirectory(pathToModified)).filter(isJavascriptModule);
					for await (const filePath of modulesInsideAddedDirectory) await updateModuleDependenciesInGraph(filePath);
					break;
				}
				case "unlink":
					if (isJavascriptModule(pathToModified)) removeModuleFromGraph(pathToModified);
					break;
				case "unlinkDir": {
					const modulesInsideDeletedDirectory = (await readAllFilesInsideDirectory(pathToModified)).filter(isJavascriptModule);
					for await (const filePath of modulesInsideDeletedDirectory) removeModuleFromGraph(filePath);
					break;
				}
			}
		},
		{ resolveDependentsOf: function resolveDependentsOf(pathToModule) {
			const dependentPaths = /* @__PURE__ */ new Set();
			const stack = [pathToModule];
			while (stack.length > 0) {
				const moduleEntry = graph[stack.pop()];
				if (!moduleEntry) continue;
				for (const dependentPath of moduleEntry.dependentPaths) {
					if (dependentPaths.has(dependentPath) || dependentPath === pathToModule) continue;
					dependentPaths.add(dependentPath);
					stack.push(dependentPath);
				}
			}
			return [...dependentPaths.values()];
		} }
	];
};

//#endregion
//#region src/utils/preview/hot-reloading/setup-hot-reloading.ts
const setupHotreloading = async (devServer$1, emailDirRelativePath) => {
	let clients = [];
	new Server(devServer$1).on("connection", (client) => {
		clients.push(client);
		client.on("disconnect", () => {
			clients = clients.filter((item) => item !== client);
		});
	});
	let changes = [];
	const reload = debounce(() => {
		clients.forEach((client) => {
			client.emit("reload", changes.filter((change) => path.resolve(absolutePathToEmailsDirectory, change.filename).startsWith(absolutePathToEmailsDirectory)));
		});
		changes = [];
	}, 150);
	const absolutePathToEmailsDirectory = path.resolve(process.cwd(), emailDirRelativePath);
	const [dependencyGraph, updateDependencyGraph, { resolveDependentsOf }] = await createDependencyGraph(absolutePathToEmailsDirectory);
	const watcher = watch("", {
		ignoreInitial: true,
		cwd: absolutePathToEmailsDirectory
	});
	const getFilesOutsideEmailsDirectory = () => Object.keys(dependencyGraph).filter((p) => path.relative(absolutePathToEmailsDirectory, p).startsWith(".."));
	let filesOutsideEmailsDirectory = getFilesOutsideEmailsDirectory();
	for (const p of filesOutsideEmailsDirectory) watcher.add(p);
	const exit = async () => {
		await watcher.close();
	};
	process.on("SIGINT", exit);
	process.on("uncaughtException", exit);
	watcher.on("all", async (event, relativePathToChangeTarget) => {
		if (relativePathToChangeTarget.split(path.sep).length === 0) return;
		const pathToChangeTarget = path.resolve(absolutePathToEmailsDirectory, relativePathToChangeTarget);
		await updateDependencyGraph(event, pathToChangeTarget);
		const newFilesOutsideEmailsDirectory = getFilesOutsideEmailsDirectory();
		for (const p of filesOutsideEmailsDirectory) if (!newFilesOutsideEmailsDirectory.includes(p)) watcher.unwatch(p);
		for (const p of newFilesOutsideEmailsDirectory) if (!filesOutsideEmailsDirectory.includes(p)) watcher.add(p);
		filesOutsideEmailsDirectory = newFilesOutsideEmailsDirectory;
		changes.push({
			event,
			filename: relativePathToChangeTarget
		});
		for (const dependentPath of resolveDependentsOf(pathToChangeTarget)) changes.push({
			event: "change",
			filename: path.relative(absolutePathToEmailsDirectory, dependentPath)
		});
		reload();
	});
	return watcher;
};

//#endregion
//#region src/utils/preview/get-env-variables-for-preview-app.ts
const getEnvVariablesForPreviewApp = (relativePathToEmailsDirectory, previewServerLocation, cwd) => {
	return {
		EMAILS_DIR_RELATIVE_PATH: relativePathToEmailsDirectory,
		EMAILS_DIR_ABSOLUTE_PATH: path.resolve(cwd, relativePathToEmailsDirectory),
		PREVIEW_SERVER_LOCATION: previewServerLocation,
		USER_PROJECT_LOCATION: cwd
	};
};

//#endregion
//#region src/utils/preview/serve-static-file.ts
const serveStaticFile = async (res, parsedUrl, staticDirRelativePath) => {
	const pathname = parsedUrl.pathname.replace("/static", "./static");
	const ext = path.parse(pathname).ext;
	const staticBaseDir = path.resolve(process.cwd(), staticDirRelativePath);
	const fileAbsolutePath = path.resolve(staticBaseDir, pathname);
	if (!fileAbsolutePath.startsWith(staticBaseDir)) {
		res.statusCode = 403;
		res.end();
		return;
	}
	try {
		const fileHandle = await promises.open(fileAbsolutePath, "r");
		const fileData = await promises.readFile(fileHandle);
		res.setHeader("Content-type", lookup(ext) || "text/plain");
		res.end(fileData);
		fileHandle.close();
	} catch (exception) {
		if (!existsSync(fileAbsolutePath)) {
			res.statusCode = 404;
			res.end();
		} else {
			const sanitizedFilePath = fileAbsolutePath.replace(/\n|\r/g, "");
			console.error(`Could not read file at %s to be served, here's the exception:`, sanitizedFilePath, exception);
			res.statusCode = 500;
			res.end("Could not read file to be served! Check your terminal for more information.");
		}
	}
};

//#endregion
//#region src/utils/preview/start-dev-server.ts
let devServer;
const safeAsyncServerListen = (server, port) => {
	return new Promise((resolve) => {
		server.listen(port, () => {
			resolve({ portAlreadyInUse: false });
		});
		server.on("error", (e) => {
			if (e.code === "EADDRINUSE") resolve({ portAlreadyInUse: true });
		});
	});
};
const startDevServer = async (emailsDirRelativePath, staticBaseDirRelativePath, port) => {
	const [majorNodeVersion] = process.versions.node.split(".");
	if (majorNodeVersion && Number.parseInt(majorNodeVersion) < 18) {
		console.error(` ${logSymbols.error}  Node ${majorNodeVersion} is not supported. Please upgrade to Node 18 or higher.`);
		process.exit(1);
	}
	const previewServerLocation = await getPreviewServerLocation();
	const previewServer = createJiti(previewServerLocation);
	devServer = http.createServer((req, res) => {
		if (!req.url) {
			res.end(404);
			return;
		}
		const parsedUrl = url.parse(req.url, true);
		res.setHeader("Cache-Control", "no-cache, max-age=0, must-revalidate, no-store");
		res.setHeader("Pragma", "no-cache");
		res.setHeader("Expires", "-1");
		try {
			if (parsedUrl.path?.includes("static/") && !parsedUrl.path.includes("_next/static/")) serveStaticFile(res, parsedUrl, staticBaseDirRelativePath);
			else if (!isNextReady) nextReadyPromise.then(() => nextHandleRequest?.(req, res, parsedUrl));
			else nextHandleRequest?.(req, res, parsedUrl);
		} catch (e) {
			console.error("caught error", e);
			res.writeHead(500);
			res.end();
		}
	});
	const { portAlreadyInUse } = await safeAsyncServerListen(devServer, port);
	if (!portAlreadyInUse) {
		console.log(styleText("greenBright", `    React Email ${package_default.version}`));
		console.log(`    Running preview at:          http://localhost:${port}\n`);
	} else {
		const nextPortToTry = port + 1;
		console.warn(` ${logSymbols.warning} Port ${port} is already in use, trying ${nextPortToTry}`);
		return startDevServer(emailsDirRelativePath, staticBaseDirRelativePath, nextPortToTry);
	}
	devServer.on("close", async () => {
		await app.close();
	});
	devServer.on("error", (e) => {
		spinner.stopAndPersist({
			symbol: logSymbols.error,
			text: `Preview Server had an error: ${e}`
		});
		process.exit(1);
	});
	const spinner = ora({
		text: "Getting react-email preview server ready...\n",
		prefixText: " "
	}).start();
	registerSpinnerAutostopping(spinner);
	const timeBeforeNextReady = performance.now();
	process.env = {
		NODE_ENV: "development",
		...process.env,
		...getEnvVariablesForPreviewApp(path.normalize(emailsDirRelativePath), previewServerLocation, process.cwd())
	};
	const app = (await previewServer.import("next", { default: true }))({
		dev: false,
		conf: { images: { unoptimized: true } },
		hostname: "localhost",
		port,
		dir: previewServerLocation
	});
	let isNextReady = false;
	const nextReadyPromise = app.prepare();
	try {
		await nextReadyPromise;
	} catch (exception) {
		spinner.stopAndPersist({
			symbol: logSymbols.error,
			text: ` Preview Server had an error: ${exception}`
		});
		process.exit(1);
	}
	isNextReady = true;
	const nextHandleRequest = app.getRequestHandler();
	const secondsToNextReady = ((performance.now() - timeBeforeNextReady) / 1e3).toFixed(1);
	spinner.stopAndPersist({
		text: `Ready in ${secondsToNextReady}s\n`,
		symbol: logSymbols.success
	});
	return devServer;
};
const makeExitHandler = (options) => (codeSignalOrError) => {
	if (typeof devServer !== "undefined") {
		console.log("\nshutting down dev server");
		devServer.close();
		devServer = void 0;
	}
	if (codeSignalOrError instanceof Error) console.error(codeSignalOrError);
	if (options?.shouldKillProcess) process.exit(options.killWithErrorCode ? 1 : 0);
};
process.on("exit", makeExitHandler());
process.on("SIGINT", makeExitHandler({
	shouldKillProcess: true,
	killWithErrorCode: false
}));
process.on("SIGUSR1", makeExitHandler({
	shouldKillProcess: true,
	killWithErrorCode: false
}));
process.on("SIGUSR2", makeExitHandler({
	shouldKillProcess: true,
	killWithErrorCode: false
}));
process.on("uncaughtException", makeExitHandler({
	shouldKillProcess: true,
	killWithErrorCode: true
}));

//#endregion
//#region src/utils/tree.ts
const SYMBOLS = {
	BRANCH: "├── ",
	EMPTY: "",
	INDENT: "    ",
	LAST_BRANCH: "└── ",
	VERTICAL: "│   "
};
const getTreeLines = async (dirPath, depth, currentDepth = 0) => {
	const base = process.cwd();
	const dirFullpath = path.resolve(base, dirPath);
	let lines = [path.basename(dirFullpath)];
	if ((await promises.stat(dirFullpath)).isDirectory() && currentDepth < depth) {
		const childDirents = await promises.readdir(dirFullpath, { withFileTypes: true });
		childDirents.sort((a, b) => {
			if (a.isDirectory() && b.isFile()) return -1;
			if (a.isFile() && b.isDirectory()) return 1;
			return b.name > a.name ? -1 : 1;
		});
		for (let i = 0; i < childDirents.length; i++) {
			const dirent = childDirents[i];
			const isLast = i === childDirents.length - 1;
			const branchingSymbol = isLast ? SYMBOLS.LAST_BRANCH : SYMBOLS.BRANCH;
			const verticalSymbol = isLast ? SYMBOLS.INDENT : SYMBOLS.VERTICAL;
			if (dirent.isFile()) lines.push(`${branchingSymbol}${dirent.name}`);
			else {
				const treeLinesForSubDirectory = await getTreeLines(path.join(dirFullpath, dirent.name), depth, currentDepth + 1);
				lines = lines.concat(treeLinesForSubDirectory.map((line, index) => index === 0 ? `${branchingSymbol}${line}` : `${verticalSymbol}${line}`));
			}
		}
	}
	return lines;
};
const tree = async (dirPath, depth) => {
	return (await getTreeLines(dirPath, depth)).join(os.EOL);
};

//#endregion
//#region src/commands/dev.ts
const dev = async ({ dir: emailsDirRelativePath, port }) => {
	try {
		if (!fs.existsSync(emailsDirRelativePath)) {
			console.error(`Missing ${emailsDirRelativePath} folder`);
			process.exit(1);
		}
		await setupHotreloading(await startDevServer(emailsDirRelativePath, emailsDirRelativePath, Number.parseInt(port)), emailsDirRelativePath);
	} catch (error) {
		console.log(error);
		process.exit(1);
	}
};

//#endregion
//#region src/utils/esbuild/escape-string-for-regex.ts
function escapeStringForRegex(string) {
	return string.replace(/[|\\{}()[\]^$+*?.]/g, "\\$&").replace(/-/g, "\\x2d");
}

//#endregion
//#region src/utils/esbuild/renderring-utilities-exporter.ts
/**
* Made to export the `render` function out of the user's email template
* so that issues like https://github.com/resend/react-email/issues/649 don't
* happen.
*
* This also exports the `createElement` from the user's React version as well
* to avoid mismatches.
*
* This avoids multiple versions of React being involved, i.e., the version
* in the CLI vs. the version the user has on their emails.
*/
const renderingUtilitiesExporter = (emailTemplates) => ({
	name: "rendering-utilities-exporter",
	setup: (b) => {
		b.onLoad({ filter: new RegExp(emailTemplates.map((emailPath) => escapeStringForRegex(emailPath)).join("|")) }, async ({ path: pathToFile }) => {
			return {
				contents: `${await promises.readFile(pathToFile, "utf8")};
          export { render } from 'react-email-module-that-will-export-render'
          export { createElement as reactEmailCreateReactElement } from 'react';
        `,
				loader: path.extname(pathToFile).slice(1)
			};
		});
		b.onResolve({ filter: /^react-email-module-that-will-export-render$/ }, async (args) => {
			const options = {
				kind: "import-statement",
				importer: args.importer,
				resolveDir: args.resolveDir,
				namespace: args.namespace
			};
			let result = await b.resolve("@react-email/render", options);
			if (result.errors.length === 0) return result;
			result = await b.resolve("@react-email/components", options);
			if (result.errors.length > 0 && result.errors[0]) result.errors[0].text = "Failed trying to import `render` from either `@react-email/render` or `@react-email/components` to be able to render your email template.\n Maybe you don't have either of them installed?";
			return result;
		});
	}
});

//#endregion
//#region src/commands/export.ts
const getEmailTemplatesFromDirectory = (emailDirectory) => {
	const templatePaths = [];
	emailDirectory.emailFilenames.forEach((filename) => templatePaths.push(path.join(emailDirectory.absolutePath, filename)));
	emailDirectory.subDirectories.forEach((directory) => {
		templatePaths.push(...getEmailTemplatesFromDirectory(directory));
	});
	return templatePaths;
};
const require = createRequire(url.fileURLToPath(import.meta.url));
const exportTemplates = async (pathToWhereEmailMarkupShouldBeDumped, emailsDirectoryPath, options) => {
	if (fs.existsSync(pathToWhereEmailMarkupShouldBeDumped)) fs.rmSync(pathToWhereEmailMarkupShouldBeDumped, { recursive: true });
	let spinner;
	if (!options.silent) {
		spinner = ora("Preparing files...\n").start();
		registerSpinnerAutostopping(spinner);
	}
	const emailsDirectoryMetadata = await getEmailsDirectoryMetadata(path.resolve(process.cwd(), emailsDirectoryPath), true);
	if (typeof emailsDirectoryMetadata === "undefined") {
		if (spinner) spinner.stopAndPersist({
			symbol: logSymbols.error,
			text: `Could not find the directory at ${emailsDirectoryPath}`
		});
		return;
	}
	const allTemplates = getEmailTemplatesFromDirectory(emailsDirectoryMetadata);
	try {
		await build({
			bundle: true,
			entryPoints: allTemplates,
			format: "cjs",
			jsx: "automatic",
			loader: { ".js": "jsx" },
			logLevel: "silent",
			outExtension: { ".js": ".cjs" },
			outdir: pathToWhereEmailMarkupShouldBeDumped,
			platform: "node",
			plugins: [renderingUtilitiesExporter(allTemplates)],
			write: true
		});
	} catch (exception) {
		if (spinner) spinner.stopAndPersist({
			symbol: logSymbols.error,
			text: "Failed to build emails"
		});
		const buildFailure = exception;
		console.error(`\n${buildFailure.message}`);
		process.exit(1);
	}
	if (spinner) spinner.succeed();
	const allBuiltTemplates = glob.sync(normalize(`${pathToWhereEmailMarkupShouldBeDumped}/**/*.cjs`), { absolute: true });
	for await (const template of allBuiltTemplates) try {
		if (spinner) {
			spinner.text = `rendering ${template.split("/").pop()}`;
			spinner.render();
		}
		delete require.cache[template];
		const emailModule = require(template);
		const rendered = await emailModule.render(emailModule.reactEmailCreateReactElement(emailModule.default, {}), options);
		writeFileSync(template.replace(".cjs", options.plainText ? ".txt" : ".html"), rendered);
		unlinkSync(template);
	} catch (exception) {
		if (spinner) spinner.stopAndPersist({
			symbol: logSymbols.error,
			text: `failed when rendering ${template.split("/").pop()}`
		});
		console.error(exception);
		process.exit(1);
	}
	if (spinner) {
		spinner.succeed("Rendered all files");
		spinner.text = "Copying static files";
		spinner.render();
	}
	const staticDirectoryPath = path.join(emailsDirectoryPath, "static");
	if (fs.existsSync(staticDirectoryPath)) {
		const pathToDumpStaticFilesInto = path.join(pathToWhereEmailMarkupShouldBeDumped, "static");
		if (fs.existsSync(pathToDumpStaticFilesInto)) await fs.promises.rm(pathToDumpStaticFilesInto, { recursive: true });
		try {
			await fs.promises.cp(staticDirectoryPath, pathToDumpStaticFilesInto, { recursive: true });
		} catch (exception) {
			console.error(exception);
			if (spinner) spinner.stopAndPersist({
				symbol: logSymbols.error,
				text: "Failed to copy static files"
			});
			console.error(`Something went wrong while copying the file to ${pathToWhereEmailMarkupShouldBeDumped}/static, ${exception}`);
			process.exit(1);
		}
	}
	if (spinner && !options.silent) {
		spinner.succeed();
		const fileTree = await tree(pathToWhereEmailMarkupShouldBeDumped, 4);
		console.log(fileTree);
		spinner.stopAndPersist({
			symbol: logSymbols.success,
			text: "Successfully exported emails"
		});
	}
};

//#endregion
//#region src/commands/start.ts
const start = async () => {
	try {
		const previewServerLocation = await getPreviewServerLocation();
		const usersProjectLocation = process.cwd();
		const builtPreviewPath = path.resolve(usersProjectLocation, "./.react-email");
		if (!fs.existsSync(builtPreviewPath)) {
			console.error("Could not find .react-email, maybe you haven't ran email build?");
			process.exit(1);
		}
		const nextStart = spawn("npx", [
			"next",
			"start",
			builtPreviewPath
		], {
			cwd: previewServerLocation,
			stdio: "inherit"
		});
		process.on("SIGINT", () => {
			nextStart.kill("SIGINT");
		});
		nextStart.on("exit", (code) => {
			process.exit(code ?? 0);
		});
	} catch (error) {
		console.log(error);
		process.exit(1);
	}
};

//#endregion
//#region src/index.ts
program.name("react-email").description("A live preview of your emails right in your browser").version(package_default.version);
program.command("dev").description("Starts the preview email development app").option("-d, --dir <path>", "Directory with your email templates", "./emails").option("-p --port <port>", "Port to run dev server on", "3000").action(dev);
program.command("build").description("Copies the preview app for onto .react-email and builds it").option("-d, --dir <path>", "Directory with your email templates", "./emails").option("-p --packageManager <name>", "Package name to use on installation on `.react-email`", "npm").action(build$1);
program.command("start").description("Runs the built preview app that is inside of \".react-email\"").action(start);
program.command("export").description("Build the templates to the `out` directory").option("--outDir <path>", "Output directory", "out").option("-p, --pretty", "Pretty print the output", false).option("-t, --plainText", "Set output format as plain text", false).option("-d, --dir <path>", "Directory with your email templates", "./emails").option("-s, --silent", "To, or not to show a spinner with process information", false).action(({ outDir, pretty, plainText, silent, dir: srcDir }) => exportTemplates(outDir, srcDir, {
	silent,
	plainText,
	pretty
}));
program.parse();

//#endregion
export {  };