import { WebSocketServer } from "ws";
import { getResponder, } from "@homebridge/ciao";
import { ZWaveError, ZWaveErrorCodes, getEnumMemberName, } from "zwave-js";
import { libVersion } from "zwave-js";
import { EventForwarder } from "./forward.js";
import { dumpLogConfig, dumpState } from "./state.js";
import { createServer } from "http";
import { EventEmitter, once } from "events";
import { dnssdServiceType, version, minSchemaVersion, maxSchemaVersion, applicationName, } from "./const.js";
import { NodeMessageHandler } from "./node/message_handler.js";
import { ControllerMessageHandler } from "./controller/message_handler.js";
import { BaseError, ErrorCode, SchemaIncompatibleError, UnknownCommandError, } from "./error.js";
import { Instance } from "./instance.js";
import { ServerCommand } from "./command.js";
import { DriverMessageHandler } from "./driver/message_handler.js";
import { LoggingEventForwarder } from "./logging.js";
import { BroadcastNodeMessageHandler } from "./broadcast_node/message_handler.js";
import { MulticastGroupMessageHandler } from "./multicast_group/message_handler.js";
import { EndpointMessageHandler } from "./endpoint/message_handler.js";
import { UtilsMessageHandler } from "./utils/message_handler.js";
import { inclusionUserCallbacks } from "./inclusion_user_callbacks.js";
import { ConfigManagerMessageHandler } from "./config_manager/message_handler.js";
import { ZnifferMessageHandler } from "./zniffer/message_handler.js";
import { stringifyReplacer } from "../util/stringify.js";
function getVersionData(driver) {
    return {
        homeId: driver.controller.homeId,
        driverVersion: libVersion,
        serverVersion: version,
        minSchemaVersion: minSchemaVersion,
        maxSchemaVersion: maxSchemaVersion,
    };
}
export class Client {
    socket;
    clientsController;
    driver;
    logger;
    remoteController;
    receiveEvents = false;
    _outstandingPing = false;
    schemaVersion = minSchemaVersion;
    receiveLogs = false;
    additionalUserAgentComponents;
    instanceHandlers;
    constructor(socket, clientsController, driver, logger, remoteController) {
        this.socket = socket;
        this.clientsController = clientsController;
        this.driver = driver;
        this.logger = logger;
        this.remoteController = remoteController;
        socket.on("pong", () => {
            this._outstandingPing = false;
        });
        socket.on("message", (data) => this.receiveMessage(data));
        this.instanceHandlers = {
            [Instance.config_manager]: new ConfigManagerMessageHandler(this.driver),
            [Instance.controller]: new ControllerMessageHandler(this.clientsController, this.driver, this),
            [Instance.driver]: new DriverMessageHandler(this.remoteController, this.clientsController, this.logger, this.driver, this),
            [Instance.node]: new NodeMessageHandler(this.clientsController, this.driver, this),
            [Instance.multicast_group]: new MulticastGroupMessageHandler(this.driver, this),
            [Instance.broadcast_node]: new BroadcastNodeMessageHandler(this.driver, this),
            [Instance.endpoint]: new EndpointMessageHandler(this.driver, this),
            [Instance.utils]: new UtilsMessageHandler(),
            [Instance.zniffer]: new ZnifferMessageHandler(driver, clientsController),
        };
    }
    get isConnected() {
        return this.socket.readyState === this.socket.OPEN;
    }
    setSchemaVersion(schemaVersion) {
        // Handle schema version
        this.schemaVersion = schemaVersion;
        if (this.schemaVersion < minSchemaVersion ||
            this.schemaVersion > maxSchemaVersion) {
            throw new SchemaIncompatibleError(this.schemaVersion);
        }
    }
    async receiveMessage(data) {
        let msg;
        try {
            msg = JSON.parse(data);
        }
        catch (err) {
            // We don't have the message ID. Just close it.
            this.logger.debug(`Unable to parse data: ${data}`);
            this.socket.close();
            return;
        }
        try {
            if (msg.command === ServerCommand.initialize) {
                this.setSchemaVersion(msg.schemaVersion);
                this.additionalUserAgentComponents = msg.additionalUserAgentComponents;
                this.sendResultSuccess(msg.messageId, {});
                return;
            }
            if (msg.command === ServerCommand.setApiSchema) {
                this.setSchemaVersion(msg.schemaVersion);
                this.sendResultSuccess(msg.messageId, {});
                return;
            }
            if (msg.command === ServerCommand.startListening) {
                this.sendResultSuccess(msg.messageId, {
                    state: dumpState(this.driver, this.schemaVersion),
                }, true);
                this.receiveEvents = true;
                return;
            }
            if (msg.command === ServerCommand.updateLogConfig) {
                this.driver.updateLogConfig(msg.config);
                this.sendResultSuccess(msg.messageId, {});
                return;
            }
            if (msg.command === ServerCommand.getLogConfig) {
                this.sendResultSuccess(msg.messageId, {
                    config: dumpLogConfig(this.driver, this.schemaVersion),
                });
                return;
            }
            if (msg.command === ServerCommand.startListeningLogs) {
                this.receiveLogs = true;
                this.clientsController.configureLoggingEventForwarder(msg.filter);
                this.sendResultSuccess(msg.messageId, {});
                return;
            }
            if (msg.command === ServerCommand.stopListeningLogs) {
                this.receiveLogs = false;
                this.clientsController.cleanupLoggingEventForwarder();
                this.sendResultSuccess(msg.messageId, {});
                return;
            }
            const instance = msg.command.split(".")[0];
            if (this.instanceHandlers[instance]) {
                return this.sendResultSuccess(msg.messageId, await this.instanceHandlers[instance].handle(msg));
            }
            throw new UnknownCommandError(msg.command);
        }
        catch (err) {
            if (err instanceof BaseError) {
                this.logger.error("Message error", err);
                const { errorCode, name, message, stack, ...args } = err;
                return this.sendResultError(msg.messageId, errorCode, message, args);
            }
            if (err instanceof ZWaveError) {
                this.logger.error("Z-Wave error", err);
                return this.sendResultZWaveError(msg.messageId, err.code, err.message);
            }
            let error;
            if (err instanceof Error) {
                error = err;
            }
            else {
                error = new Error(`${err}`);
            }
            this.logger.error("Unexpected error", error);
            this.sendResultError(msg.messageId, ErrorCode.unknownError, error.stack ?? error.message, {});
        }
    }
    sendVersion() {
        this.sendData({
            type: "version",
            ...getVersionData(this.driver),
        });
    }
    sendResultSuccess(messageId, result, compress = false) {
        this.sendData({
            type: "result",
            success: true,
            messageId,
            result,
        }, compress);
    }
    sendResultError(messageId, errorCode, message, args) {
        if (this.schemaVersion <= 31) {
            // `sendResultError` didn't support passing the error message before schema 32.
            // We `sendResultZWaveError` instead so that we can pass the error message in
            // for the client to consume and display.
            this.sendResultZWaveError(messageId, -1, `${errorCode}: ${message}`);
        }
        else {
            this.sendData({
                type: "result",
                success: false,
                messageId,
                errorCode,
                message,
                args,
            });
        }
    }
    sendResultZWaveError(messageId, zjsErrorCode, message) {
        if (this.schemaVersion <= 31) {
            this.sendData({
                type: "result",
                success: false,
                messageId,
                errorCode: ErrorCode.zwaveError,
                zwaveErrorCode: zjsErrorCode,
                zwaveErrorMessage: message,
            });
        }
        else {
            this.sendData({
                type: "result",
                success: false,
                messageId,
                errorCode: ErrorCode.zwaveError,
                zwaveErrorCode: zjsErrorCode,
                zwaveErrorCodeName: getEnumMemberName(ZWaveErrorCodes, zjsErrorCode),
                zwaveErrorMessage: message,
            });
        }
    }
    sendEvent(event) {
        this.sendData({
            type: "event",
            event,
        });
    }
    sendData(data, compress = false) {
        this.socket.send(JSON.stringify(data, stringifyReplacer), { compress });
    }
    checkAlive() {
        if (this._outstandingPing) {
            this.disconnect();
            return;
        }
        this._outstandingPing = true;
        this.socket.ping();
    }
    disconnect() {
        this.socket.close();
    }
}
export class ClientsController extends EventEmitter {
    driver;
    logger;
    remoteController;
    clients = [];
    pingInterval;
    eventForwarder;
    cleanupScheduled = false;
    loggingEventForwarder;
    grantSecurityClassesPromise;
    validateDSKAndEnterPinPromise;
    constructor(driver, logger, remoteController) {
        super();
        this.driver = driver;
        this.logger = logger;
        this.remoteController = remoteController;
    }
    addSocket(socket) {
        this.logger.debug("New client");
        const client = new Client(socket, this, this.driver, this.logger, this.remoteController);
        socket.on("error", (error) => {
            this.logger.error("Client socket error", error);
        });
        socket.on("close", (code, reason) => {
            this.logger.info("Client disconnected");
            this.logger.debug(`Code ${code}: ${reason}`);
            this.scheduleClientCleanup();
        });
        client.sendVersion();
        this.clients.push(client);
        if (this.pingInterval === undefined) {
            this.pingInterval = setInterval(() => {
                const newClients = [];
                for (const client of this.clients) {
                    if (client.isConnected) {
                        newClients.push(client);
                    }
                    else {
                        client.disconnect();
                    }
                }
                this.clients = newClients;
            }, 30000);
        }
        if (this.eventForwarder === undefined) {
            this.eventForwarder = new EventForwarder(this);
            this.eventForwarder.start();
        }
    }
    get loggingEventForwarderStarted() {
        return this.loggingEventForwarder?.started === true;
    }
    restartLoggingEventForwarderIfNeeded() {
        this.loggingEventForwarder?.restartIfNeeded();
    }
    configureLoggingEventForwarder(filter) {
        if (this.loggingEventForwarder === undefined) {
            this.loggingEventForwarder = new LoggingEventForwarder(this, this.driver, this.logger);
        }
        if (!this.loggingEventForwarderStarted) {
            this.loggingEventForwarder?.start(filter);
        }
    }
    cleanupLoggingEventForwarder() {
        if (this.clients.filter((cl) => cl.receiveLogs).length == 0 &&
            this.loggingEventForwarderStarted) {
            this.loggingEventForwarder?.stop();
        }
    }
    scheduleClientCleanup() {
        if (this.cleanupScheduled) {
            return;
        }
        this.cleanupScheduled = true;
        setTimeout(() => this.cleanupClients(), 0);
    }
    cleanupClients() {
        this.cleanupScheduled = false;
        this.clients = this.clients.filter((cl) => cl.isConnected);
        this.cleanupLoggingEventForwarder();
    }
    disconnect() {
        if (this.pingInterval !== undefined) {
            clearInterval(this.pingInterval);
        }
        this.pingInterval = undefined;
        this.clients.forEach((client) => client.disconnect());
        this.clients = [];
        this.cleanupLoggingEventForwarder();
    }
}
/**
 * This class allows the hard reset driver command to be passed to the
 * ClientsController, Client, and Message Handler instances without
 * providing access to the base server and eventing system.
 */
export class ZwavejsServerRemoteController {
    driver;
    zwaveJsServer;
    constructor(driver, zwaveJsServer) {
        this.driver = driver;
        this.zwaveJsServer = zwaveJsServer;
    }
    async hardResetController() {
        await this.driver.hardReset();
        this.zwaveJsServer.emit("hard reset");
    }
}
export class ZwavejsServer extends EventEmitter {
    driver;
    options;
    server;
    wsServer;
    sockets;
    logger;
    defaultPort = 3000;
    responder;
    service;
    remoteController;
    constructor(driver, options = {}) {
        super();
        this.driver = driver;
        this.options = options;
        this.remoteController = new ZwavejsServerRemoteController(driver, this);
        this.logger = options.logger ?? console;
    }
    async start(shouldSetInclusionUserCallbacks = false) {
        if (!this.driver.ready) {
            throw new Error("Cannot start server when driver not ready");
        }
        this.driver.updateUserAgent({ [applicationName]: version });
        this.server = createServer();
        this.wsServer = new WebSocketServer({
            server: this.server,
            perMessageDeflate: true,
        });
        this.sockets = new ClientsController(this.driver, this.logger, this.remoteController);
        if (shouldSetInclusionUserCallbacks) {
            this.setInclusionUserCallbacks();
        }
        this.wsServer.on("connection", (socket) => this.sockets.addSocket(socket));
        const port = this.options.port || this.defaultPort;
        const host = this.options.host;
        const localEndpointString = `${host ?? "<all interfaces>"}:${port}`;
        this.logger.debug(`Starting server on ${localEndpointString}`);
        this.wsServer.on("error", this.onError.bind(this, this.wsServer));
        this.server.on("error", this.onError.bind(this, this.server));
        this.server.listen(port, host);
        await once(this.server, "listening");
        this.emit("listening");
        this.logger.info(`ZwaveJS server listening on ${localEndpointString}`);
        if (this.options.enableDNSServiceDiscovery) {
            this.responder = getResponder();
            this.service = this.responder.createService({
                name: this.driver.controller.homeId.toString(),
                port,
                type: dnssdServiceType,
                protocol: "tcp" /* Protocol.TCP */,
                txt: getVersionData(this.driver),
            });
            this.service.advertise().then(() => {
                this.logger.info(`DNS Service Discovery enabled`);
            });
        }
    }
    setInclusionUserCallbacks() {
        if (this.sockets === undefined) {
            throw new Error("Server must be started before setting the inclusion user callbacks");
        }
        this.driver.updateOptions({
            inclusionUserCallbacks: inclusionUserCallbacks(this.sockets),
        });
    }
    onError(sourceClass, error) {
        this.emit("error", error, sourceClass);
        this.logger.error(error);
    }
    async destroy() {
        this.logger.debug(`Closing server...`);
        if (this.sockets) {
            this.sockets.disconnect();
            this.sockets.removeAllListeners();
            delete this.sockets;
        }
        if (this.wsServer) {
            this.wsServer.close();
            await once(this.wsServer, "close");
            this.wsServer.removeAllListeners();
            delete this.wsServer;
        }
        if (this.server) {
            this.server.close();
            await once(this.server, "close");
            this.server.removeAllListeners();
            delete this.server;
        }
        if (this.service) {
            await this.service.end();
            await this.service.destroy();
            this.service.removeAllListeners();
            delete this.service;
        }
        if (this.responder) {
            await this.responder.shutdown();
            delete this.responder;
        }
        this.logger.info(`Server closed`);
    }
}
