import { type CommandClasses, NodeIDType, type NodeProtocolInfo, RFRegion } from "@zwave-js/core";
import { type BytesView } from "@zwave-js/shared";
import { type NVM3Meta } from "./lib/NVM3.js";
import { type Route } from "./lib/common/routeCache.js";
import type { SUCUpdateEntry } from "./lib/common/sucUpdateEntry.js";
import type { NVM3Object } from "./lib/nvm3/object.js";
import type { NVM500JSON } from "./nvm500/NVMParser.js";
export interface NVMJSON {
    format: number;
    applicationFileFormat?: number;
    meta?: NVM3Meta;
    controller: NVMJSONController;
    nodes: Record<number, NVMJSONNode>;
    lrNodes?: Record<number, NVMJSONLRNode>;
}
export type NVMJSONWithMeta = NVMJSON & {
    meta: NVM3Meta;
};
export interface NVMJSONController {
    protocolVersion: string;
    applicationVersion: string;
    homeId: string;
    nodeId: number;
    lastNodeId: number;
    staticControllerNodeId: number;
    sucLastIndex: number;
    controllerConfiguration: number;
    sucUpdateEntries: SUCUpdateEntry[];
    sucAwarenessPushNeeded?: number | null;
    maxNodeId: number;
    reservedId: number;
    systemState: number;
    lastNodeIdLR?: number | null;
    maxNodeIdLR?: number | null;
    reservedIdLR?: number | null;
    primaryLongRangeChannelId?: number | null;
    dcdcConfig?: number | null;
    rfConfig?: NVMJSONControllerRFConfig | null;
    preferredRepeaters?: number[] | null;
    isListening: boolean;
    optionalFunctionality: boolean;
    genericDeviceClass: number;
    specificDeviceClass: number;
    commandClasses: {
        includedInsecurely: CommandClasses[];
        includedSecurelyInsecureCCs: CommandClasses[];
        includedSecurelySecureCCs: CommandClasses[];
    };
    applicationData?: string | null;
    applicationName?: string | null;
}
export interface NVMJSONControllerRFConfig {
    rfRegion: RFRegion;
    txPower: number;
    measured0dBm: number;
    enablePTI: number | null;
    maxTXPower: number | null;
    nodeIdType: NodeIDType | null;
}
export interface NVMJSONNodeWithInfo extends Omit<NodeProtocolInfo, "hasSpecificDeviceClass"> {
    isVirtual: boolean;
    genericDeviceClass: number;
    specificDeviceClass: number | null;
    neighbors: number[];
    sucUpdateIndex: number;
    appRouteLock: boolean;
    routeSlaveSUC: boolean;
    sucPendingUpdate: boolean;
    pendingDiscovery: boolean;
    lwr?: Route | null;
    nlwr?: Route | null;
}
export interface NVMJSONVirtualNode {
    isVirtual: true;
}
export interface NVMJSONLRNode extends Omit<NodeProtocolInfo, "hasSpecificDeviceClass"> {
    genericDeviceClass: number;
    specificDeviceClass?: number | null;
}
export type NVMJSONNode = NVMJSONNodeWithInfo | NVMJSONVirtualNode;
/**
 * Options influencing how NVM contents should be migrated.
 * By default, all data will be preserved.
 */
export interface MigrateNVMOptions {
    /** Whether application data will be preserved */
    preserveApplicationData?: boolean;
    /** Whether SUC update entries will be preserved */
    preserveSUCUpdateEntries?: boolean;
    /** Whether LWR, NLWR and the priority route flag will be preserved */
    preserveRoutes?: boolean;
    /** Whether the neighbor table will be preserved */
    preserveNeighbors?: boolean;
}
export declare function nodeHasInfo(node: NVMJSONNode): node is NVMJSONNodeWithInfo;
/** Converts a compressed set of NVM objects to a JSON representation */
export declare function nvmObjectsToJSON(objects: ReadonlyMap<number, NVM3Object>): NVMJSON;
/** Reads an NVM buffer of a 700+ series stick and returns its JSON representation */
export declare function nvmToJSON(buffer: BytesView, debugLogs?: boolean): Promise<NVMJSONWithMeta>;
/** Reads an NVM buffer of a 500-series stick and returns its JSON representation */
export declare function nvm500ToJSON(buffer: BytesView): Promise<Required<NVM500JSON>>;
export declare function jsonToNVM(json: NVMJSON, targetSDKVersion: string): Promise<BytesView>;
/** Takes a JSON represented 500 series NVM and converts it to binary */
export declare function jsonToNVM500(json: Required<NVM500JSON>, protocolVersion: string): Promise<BytesView>;
export declare function json500To700(json: NVM500JSON, truncateApplicationData?: boolean): NVMJSON;
export declare function json700To500(json: NVMJSON): NVM500JSON;
/** Converts the given source NVM into a format that is compatible with the given target NVM */
export declare function migrateNVM(sourceNVM: BytesView, targetNVM: BytesView, options?: MigrateNVMOptions): Promise<BytesView>;
//# sourceMappingURL=convert.d.ts.map