import { type BytesView } from "@zwave-js/shared";
/**
 * Protobuf wire types
 */
export declare const enum WireType {
    Varint = 0,
    Fixed64 = 1,
    LengthDelimited = 2,
    StartGroup = 3,// Deprecated
    EndGroup = 4,// Deprecated
    Fixed32 = 5
}
/**
 * Helper functions for protobuf wire format encoding/decoding
 */
/**
 * Encodes a protobuf field tag (field number and wire type)
 */
export declare function encodeTag(fieldNumber: number, wireType: WireType): BytesView;
/**
 * Decodes a protobuf field tag
 */
export declare function decodeTag(data: BytesView, offset?: number): {
    fieldNumber: number;
    wireType: WireType;
    bytesRead: number;
};
/**
 * Encodes a VarInt according to the Protocol Buffers specification
 */
export declare function encodeVarInt(value: number): BytesView;
/**
 * Decodes a VarInt according to the Protocol Buffers specification
 * Returns the decoded value and the number of bytes consumed
 */
export declare function decodeVarInt(data: BytesView, offset?: number): {
    value: number;
    bytesRead: number;
};
/**
 * Encodes a string field
 */
export declare function encodeStringField(fieldNumber: number, value: string | BytesView): BytesView;
/**
 * Decodes a string field at the current position
 */
export declare function decodeStringField(data: BytesView, offset: number): {
    value: string;
    bytesRead: number;
};
/**
 * Encodes a varint field (uint32, int32, bool, enum)
 */
export declare function encodeVarintField(fieldNumber: number, value: number): BytesView;
/**
 * Encodes a boolean field
 */
export declare function encodeBoolField(fieldNumber: number, value: boolean): BytesView;
/**
 * Encodes a fixed32 field
 */
export declare function encodeFixed32Field(fieldNumber: number, value: number): BytesView;
/**
 * Decodes a fixed32 field at the current position
 */
export declare function decodeFixed32Field(data: BytesView, offset: number): {
    value: number;
    bytesRead: number;
};
/**
 * Encodes a length-delimited field (for nested messages)
 */
export declare function encodeLengthDelimitedField(fieldNumber: number, value: BytesView): BytesView;
/**
 * Decodes a length-delimited field at the current position
 */
export declare function decodeLengthDelimitedField(data: BytesView, offset: number): {
    value: BytesView;
    bytesRead: number;
};
/**
 * Encodes a signed integer using zigzag encoding
 */
export declare function encodeSignedVarint(value: number): BytesView;
/**
 * Decodes a signed integer using zigzag encoding
 */
export declare function decodeSignedVarint(data: BytesView, offset?: number): {
    value: number;
    bytesRead: number;
};
/**
 * Encodes a signed varint field
 */
export declare function encodeSignedVarintField(fieldNumber: number, value: number): BytesView;
/**
 * Skips an unknown field based on its wire type
 */
export declare function skipField(data: BytesView, offset: number, wireType: WireType): number;
/**
 * Parses a protobuf message and calls a field handler for each field
 */
export declare function parseProtobufMessage(data: BytesView, fieldHandler: (fieldNumber: number, wireType: WireType, data: BytesView, offset: number) => number): void;
//# sourceMappingURL=ProtobufHelpers.d.ts.map