// This is basically a duplex transform stream wrapper around a
// readable and a writable stream, both compatible with the Web Streams API.
// This allows supporting different low-level connections to a Z-Wave controller,
// whether it's a serial port, a network connection, or something else.
//
// 0 ┌─────────────────┐ ┌─────────────────┐ ┌──
// 1 <--               <--   PassThrough   <-- write
// 1 │    any stream   │ │ ZWaveSerialPort │ │
// 0 -->               -->     Parsers     --> read
// 1 └─────────────────┘ └─────────────────┘ └──
import { noop } from "@zwave-js/shared";
import { SerialLogger } from "../log/Logger.js";
import { MessageHeaders } from "../message/MessageHeaders.js";
import { ZWaveSerialParser } from "../plumbing/ZWaveSerialParser.js";
import { ZWaveSerialMode } from "./definitions.js";
/** Tests if `obj` is (probably) a ZWaveSerialBindingFactory */
export function isZWaveSerialBindingFactory(obj) {
    return typeof obj === "function" && obj.length === 0;
}
/** Re-usable stream factory to create new serial streams */
export class ZWaveSerialStreamFactory {
    constructor(binding, loggers) {
        this.binding = binding;
        this.logger = new SerialLogger(loggers);
    }
    binding;
    logger;
    async createStream() {
        // Set up streams for the underlying resource
        const { source, sink } = await this.binding();
        return new ZWaveSerialStream(source, sink, this.logger);
    }
}
/** Single-use serial stream. Has to be re-created after being closed. */
export class ZWaveSerialStream {
    constructor(source, sink, logger) {
        this.logger = logger;
        this.#abort = new AbortController();
        // Expose the underlying sink as the writable side of this stream.
        // We use an identity stream in the middle to pipe through, so we
        // can properly abort the stream
        const { readable: input, writable } = new TransformStream();
        this.writable = writable;
        const sinkStream = new WritableStream(sink);
        void input
            .pipeTo(sinkStream, { signal: this.#abort.signal })
            .catch(noop);
        // Pipe the underlying source through the parser to the readable side
        this.parser = new ZWaveSerialParser(logger, this.#abort.signal);
        this.readable = this.parser.readable;
        const sourceStream = new ReadableStream(source);
        void sourceStream.pipeTo(this.parser.writable, {
            signal: this.#abort.signal,
        }).catch((_e) => {
            this._isOpen = false;
        });
    }
    logger;
    // Public interface to let consumers read from and write to this stream
    readable;
    writable;
    // Signal to close the underlying stream
    #abort;
    // Serial API parser
    parser;
    // Allow switching between modes
    get mode() {
        return this.parser.mode;
    }
    set mode(mode) {
        this.parser.mode = mode;
    }
    // Allow ignoring the high nibble of an ACK once to work around an issue in the 700 series firmware
    ignoreAckHighNibbleOnce() {
        this.parser.ignoreAckHighNibbleOnce();
    }
    async close() {
        this._isOpen = false;
        // Close the underlying stream
        if (this._writer) {
            try {
                this._writer?.releaseLock();
                this._writer = undefined;
                await this.writable.close();
            }
            catch {
                // ignore
            }
        }
        this.#abort.abort();
        return Promise.resolve();
        // // Wait for streams to finish
        // await this.#pipePromise;
    }
    _isOpen = true;
    get isOpen() {
        return this._isOpen;
    }
    _writer;
    async writeAsync(data) {
        if (!this.isOpen) {
            throw new Error("The serial port is not open!");
        }
        // Only log in Serial API mode
        if (this.mode === ZWaveSerialMode.SerialAPI && data.length === 1) {
            switch (data[0]) {
                case MessageHeaders.ACK:
                    this.logger.ACK("outbound");
                    break;
                case MessageHeaders.CAN:
                    this.logger.CAN("outbound");
                    break;
                case MessageHeaders.NAK:
                    this.logger.NAK("outbound");
                    break;
            }
        }
        else {
            this.logger.data("outbound", data);
        }
        // Keep a writer instance to avoid locking and unlocking the
        // writable stream for every write, as this can cause errors
        // when writing in quick succession.
        this._writer ??= this.writable.getWriter();
        await this._writer.write(data);
    }
}
//# sourceMappingURL=ZWaveSerialStream.js.map