import { createDeferredPromise, } from "alcalzone-shared/deferred-promise";
export class AsyncQueue {
    /** Adds one or more items onto the queue */
    add(...items) {
        if (items.length === 0 || this.ended)
            return;
        // Resolve any pending promises first
        while (items.length > 0 && this.listeners.length > 0) {
            const promise = this.listeners.shift();
            const item = items.shift();
            promise.resolve(item);
        }
        // Add the remaining items to the backlog
        this.backlog.push(...items);
    }
    /**
     * Removes an item from the queue if it was not processed yet.
     * The return value indicates whether the item was removed.
     */
    remove(item) {
        if (this.ended)
            return false;
        // Remove the item from the backlog
        const index = this.backlog.indexOf(item);
        if (index !== -1) {
            this.backlog.splice(index, 1);
            return true;
        }
        return false;
    }
    get length() {
        return this.backlog.length;
    }
    // A list of items that have been pushed but not pulled
    backlog = [];
    // A list of Promises that are waiting to be resolved
    listeners = [];
    // Whether the queue was ended
    ended = false;
    /** Ends the queue after it has been drained */
    end() {
        this.ended = true;
    }
    /** Ends the queue and discards all pending items */
    abort() {
        this.ended = true;
        while (this.backlog.length > 0) {
            const removed = this.backlog.pop();
            if (typeof removed === "object"
                && removed !== null
                && Symbol.dispose in removed
                && typeof removed[Symbol.dispose] === "function") {
                console.log("dispose");
                removed[Symbol.dispose]();
            }
        }
        for (const p of this.listeners) {
            p.resolve(undefined);
        }
    }
    [Symbol.asyncIterator]() {
        return {
            next: async () => {
                let value;
                if (this.backlog.length > 0) {
                    // If there are items in the backlog, return the first one
                    value = this.backlog.shift();
                }
                else if (!this.ended) {
                    // Otherwise create a new promise and add it to the pending list
                    const promise = createDeferredPromise();
                    this.listeners.push(promise);
                    value = await promise;
                }
                if (value) {
                    // We have a value, return it
                    return { value, done: false };
                }
                else {
                    // No value means the queue was ended
                    return { value: undefined, done: true };
                }
            },
        };
    }
}
//# sourceMappingURL=AsyncQueue.js.map