import { createDeferredPromise, } from "alcalzone-shared/deferred-promise";
import { SortedList } from "alcalzone-shared/sorted-list";
import { createWrappingCounter } from "./wrappingCounter.js";
import { evalOrStatic, highResTimestamp, noop } from "./utils.js";
function getTaskName(task) {
    return `${task.name ?? (task.tag && JSON.stringify(task.tag)) ?? "unnamed"} (ID ${task.id})`;
}
function isTaskBuilder(obj) {
    return (typeof obj === "object" &&
        typeof obj.task === "function" &&
        typeof obj.priority === "number");
}
/**
 * The priority of a task.
 *
 * Higher priority tasks are executed first and interrupt lower priority tasks.
 * The recommended priority for application-initiated communication is `Normal`.
 * `Low` and `Lower` are recommended for internal long-running tasks that should not interfere with user-initiated tasks.
 * `Idle` is recommended for tasks that should only run when no other tasks are pending.
 */
export var TaskPriority;
(function (TaskPriority) {
    TaskPriority[TaskPriority["Highest"] = 0] = "Highest";
    TaskPriority[TaskPriority["High"] = 1] = "High";
    TaskPriority[TaskPriority["Normal"] = 2] = "Normal";
    TaskPriority[TaskPriority["Low"] = 3] = "Low";
    TaskPriority[TaskPriority["Lower"] = 4] = "Lower";
    TaskPriority[TaskPriority["Idle"] = 5] = "Idle";
})(TaskPriority || (TaskPriority = {}));
export var TaskState;
(function (TaskState) {
    /** The task has not been created yet */
    TaskState[TaskState["None"] = 0] = "None";
    /** The task is being executed */
    TaskState[TaskState["Active"] = 1] = "Active";
    /** The task is waiting for a Promise to resolve */
    TaskState[TaskState["AwaitingPromise"] = 2] = "AwaitingPromise";
    /** The task is waiting for another task to finish */
    TaskState[TaskState["AwaitingTask"] = 3] = "AwaitingTask";
    /** The task is finished */
    TaskState[TaskState["Done"] = 4] = "Done";
})(TaskState || (TaskState = {}));
export var TaskInterruptBehavior;
(function (TaskInterruptBehavior) {
    /** The task may not be interrupted */
    TaskInterruptBehavior[TaskInterruptBehavior["Forbidden"] = 0] = "Forbidden";
    /** The task will be resumed after being interrupted (default) */
    TaskInterruptBehavior[TaskInterruptBehavior["Resume"] = 1] = "Resume";
    /** The task needs to be restarted after being interrupted */
    TaskInterruptBehavior[TaskInterruptBehavior["Restart"] = 2] = "Restart";
})(TaskInterruptBehavior || (TaskInterruptBehavior = {}));
function compareTasks(a, b) {
    // In the same concurrency group, only one task may be active at a time.
    const aConcurrencyGroup = a.group?.id;
    const bConcurrencyGroup = b.group?.id;
    if (aConcurrencyGroup &&
        bConcurrencyGroup &&
        aConcurrencyGroup === bConcurrencyGroup) {
        const aActiveOrWaiting = a.state === TaskState.Active ||
            a.state === TaskState.AwaitingPromise ||
            a.state === TaskState.AwaitingTask;
        const bActiveOrWaiting = b.state === TaskState.Active ||
            b.state === TaskState.AwaitingPromise ||
            b.state === TaskState.AwaitingTask;
        if (aActiveOrWaiting && !bActiveOrWaiting)
            return 1; // A is already active
        if (!aActiveOrWaiting && bActiveOrWaiting)
            return -1; // B is already active
        // None of the tasks is active, fall back to default sorting
    }
    // Sort by priority for tasks not in the same concurrency group
    if (a.priority < b.priority)
        return 1;
    if (a.priority > b.priority)
        return -1;
    // Deprioritize waiting tasks
    const aWaiting = a.state === TaskState.AwaitingPromise ||
        a.state === TaskState.AwaitingTask;
    const bWaiting = b.state === TaskState.AwaitingPromise ||
        b.state === TaskState.AwaitingTask;
    if (!aWaiting && bWaiting)
        return 1;
    if (aWaiting && !bWaiting)
        return -1;
    // Sort equal priority by timestamp. Newer tasks go to the end of the list
    if (a.timestamp < b.timestamp)
        return 1;
    if (a.timestamp > b.timestamp)
        return -1;
    // If all else fails, sort by ID
    return Math.sign(b.id - a.id);
}
export class TaskScheduler {
    defaultErrorFactory;
    verbose;
    constructor(defaultErrorFactory = () => new Error("Task was removed"), verbose = false) {
        this.defaultErrorFactory = defaultErrorFactory;
        this.verbose = verbose;
    }
    _tasks = new SortedList(undefined, compareTasks);
    _currentTask;
    _idGenerator = createWrappingCounter(0xff_ff_ff);
    _continueSignal;
    _stopSignal;
    _stopPromise;
    queueTask(builder) {
        const task = this.createTask(builder);
        this._tasks.add(task);
        if (this.verbose) {
            console.log(`Task queued: ${getTaskName(task)}`);
        }
        if (this._continueSignal)
            this._continueSignal.resolve();
        return task.promise;
    }
    /** Removes/stops tasks matching the given predicate. Returns `true` when a task was removed, `false` otherwise. */
    async removeTasks(predicate, reason) {
        // Collect tasks that should be removed, but in reverse order,
        // so that we handle the current task last.
        const tasksToRemove = [];
        let removeCurrentTask = false;
        for (const task of this._tasks) {
            if (predicate(task)) {
                if (task === this._currentTask) {
                    removeCurrentTask = true;
                }
                else {
                    tasksToRemove.push(task);
                }
            }
        }
        reason ??= this.defaultErrorFactory();
        for (const task of tasksToRemove) {
            if (this.verbose) {
                console.log(`Removing task: ${getTaskName(task)}`);
            }
            this._tasks.remove(task);
            if (task.state === TaskState.Active ||
                task.state === TaskState.AwaitingPromise ||
                task.state === TaskState.AwaitingTask) {
                // The task is running, clean it up
                await task.reset().catch(noop);
            }
            task.reject(reason);
            // Re-add the parent task to the list if there is one
            if (task.parent) {
                if (this.verbose) {
                    console.log(`Restoring parent task: ${getTaskName(task.parent)}`);
                }
                this._tasks.add(task.parent);
            }
        }
        if (removeCurrentTask && this._currentTask) {
            if (this.verbose) {
                console.log(`Removing task: ${getTaskName(this._currentTask)}`);
            }
            this._tasks.remove(this._currentTask);
            await this._currentTask.reset().catch(noop);
            this._currentTask.reject(reason);
            // Re-add the parent task to the list if there is one
            if (this._currentTask.parent) {
                if (this.verbose) {
                    console.log(`Restoring parent task: ${getTaskName(this._currentTask.parent)}`);
                }
                this._tasks.add(this._currentTask.parent);
            }
            this._currentTask = undefined;
        }
        if (this._continueSignal)
            this._continueSignal.resolve();
        return tasksToRemove.length > 0 || removeCurrentTask;
    }
    findTask(predicate) {
        return this._tasks.find((t) => predicate(t))?.promise;
    }
    /** Creates a task that can be executed */
    createTask(builder, parent) {
        let state = TaskState.None;
        let generator;
        let waitForPromise;
        const promise = createDeferredPromise();
        let prevResult;
        let waitError;
        const self = this;
        return {
            id: this._idGenerator(),
            timestamp: highResTimestamp(),
            builder,
            parent,
            name: builder.name,
            tag: builder.tag,
            priority: builder.priority,
            group: builder.group,
            interrupt: builder.interrupt ?? TaskInterruptBehavior.Resume,
            promise,
            async step() {
                // Do not proceed while still waiting for a Promise to resolve
                if (state === TaskState.AwaitingPromise && waitForPromise) {
                    return {
                        newState: state,
                        promise: waitForPromise,
                    };
                }
                else if (state === TaskState.AwaitingTask) {
                    throw new Error("Cannot step through a task that is waiting for another task");
                }
                generator ??= builder.task();
                state = TaskState.Active;
                const { value, done } = waitError
                    ? await generator.throw(waitError)
                    : await generator.next(prevResult);
                prevResult = undefined;
                waitError = undefined;
                if (done) {
                    state = TaskState.Done;
                    return {
                        newState: state,
                        result: value,
                    };
                }
                else if (value != undefined) {
                    const waitFor = evalOrStatic(value);
                    if (waitFor instanceof Promise) {
                        state = TaskState.AwaitingPromise;
                        waitForPromise = waitFor
                            .then((result) => {
                            prevResult = result;
                        })
                            .catch((e) => {
                            waitError = e;
                        })
                            .finally(() => {
                            waitForPromise = undefined;
                            if (state === TaskState.AwaitingPromise) {
                                state = TaskState.Active;
                            }
                        });
                        return {
                            newState: state,
                            promise: waitForPromise,
                        };
                    }
                    else if (isTaskBuilder(waitFor)) {
                        if (waitFor.priority > builder.priority) {
                            throw new Error("Tasks cannot yield to tasks with lower priority than their own");
                        }
                        // Create a sub-task with a reference to this task
                        state = TaskState.AwaitingTask;
                        const subTask = self.createTask(waitFor, this);
                        subTask.promise
                            .then((result) => {
                            prevResult = result;
                        })
                            .catch((e) => {
                            waitError = e;
                        })
                            .finally(() => {
                            if (state === TaskState.AwaitingTask) {
                                state = TaskState.Active;
                            }
                        });
                        return {
                            newState: state,
                            task: subTask,
                        };
                    }
                    else {
                        throw new Error("Invalid value yielded by task");
                    }
                }
                else {
                    return { newState: state };
                }
            },
            async reset() {
                if (state === TaskState.None)
                    return;
                state = TaskState.None;
                waitForPromise = undefined;
                generator = undefined;
                await builder.cleanup?.();
            },
            resolve(result) {
                promise.resolve(result);
            },
            reject(error) {
                promise.reject(error);
            },
            get state() {
                return state;
            },
            get generator() {
                return generator;
            },
        };
    }
    start() {
        this._stopSignal = createDeferredPromise();
        setImmediate(async () => {
            try {
                await this.run();
            }
            catch (e) {
                console.error("Task runner crashed", e);
            }
        });
    }
    async run() {
        while (true) {
            let waitFor;
            if (this._tasks.length > 0) {
                const firstTask = this._tasks.peekStart();
                if (!this._currentTask) {
                    // We're not currently executing a task, start executing the first one
                    this._currentTask = firstTask;
                }
                else if (this._currentTask !== firstTask &&
                    this._currentTask.interrupt !==
                        TaskInterruptBehavior.Forbidden) {
                    // We are executing an interruptible task, and a new task with a higher priority was added
                    if (this._currentTask.interrupt ===
                        TaskInterruptBehavior.Restart) {
                        // The current task needs to be restarted after being interrupted, so reset it
                        await this._currentTask.reset();
                    }
                    // switch to the new task
                    this._currentTask = firstTask;
                }
                if (this.verbose) {
                    console.log(`Stepping through task: ${getTaskName(this._currentTask)}`);
                }
                const cleanupCurrentTask = async () => {
                    if (this._currentTask) {
                        this._tasks.remove(this._currentTask);
                        await this._currentTask.reset();
                        // Re-add the parent task to the list if there is one
                        if (this._currentTask.parent) {
                            if (this.verbose) {
                                console.log(`Restoring parent task: ${getTaskName(this._currentTask.parent)}`);
                            }
                            this._tasks.add(this._currentTask.parent);
                        }
                        this._currentTask = undefined;
                    }
                };
                // Execute the current task one step further
                waitFor = undefined;
                let stepResult;
                try {
                    stepResult = await this._currentTask.step();
                }
                catch (e) {
                    if (this.verbose) {
                        console.error(`- Task threw an error:`, e);
                    }
                    // The task threw an error, expose the result and clean up.
                    this._currentTask.reject(e);
                    await cleanupCurrentTask();
                    // Then continue with the next iteration
                    continue;
                }
                if (stepResult.newState === TaskState.Done) {
                    if (this.verbose) {
                        console.log(`- Task finished`);
                    }
                    // The task is done, clean up
                    this._currentTask.resolve(stepResult.result);
                    await cleanupCurrentTask();
                    // Then continue with the next iteration
                    continue;
                }
                else if (stepResult.newState === TaskState.AwaitingPromise) {
                    // The task is waiting for something
                    if (this.verbose) {
                        console.log(`- Task waiting`);
                    }
                    // If the task may be interrupted, check if there are other same-priority tasks that should be executed instead
                    if (this._currentTask.interrupt !==
                        TaskInterruptBehavior.Forbidden) {
                        // Re-queue the task, so the queue gets reordered
                        this._tasks.remove(this._currentTask);
                        this._tasks.add(this._currentTask);
                        if (this._tasks.peekStart() !== this._currentTask) {
                            if (this.verbose) {
                                console.log(`-- Continuing with another task`);
                            }
                            // The task is no longer the first in the queue. Switch to the other one
                            continue;
                        }
                    }
                    // Otherwise, we got nothing to do right now than to wait
                    waitFor = stepResult.promise;
                }
                else if (stepResult.newState === TaskState.AwaitingTask) {
                    // The task spawned a sub-task. Replace it with the sub-task and continue executing
                    if (this.verbose) {
                        console.log(`- Task spawned a sub-task`);
                    }
                    this._tasks.add(stepResult.task);
                    this._tasks.remove(this._currentTask);
                    // Continue with the next iteration
                    continue;
                }
                else {
                    // The current task is not done, continue with the next iteration
                    continue;
                }
            }
            // Task queue empty. Wait for either a new task or the stop signal,
            // or the current task to finish waiting
            this._continueSignal = createDeferredPromise();
            const nextAction = await Promise.race([
                this._continueSignal.then(() => "continue"),
                this._stopSignal.then(() => "stop"),
                waitFor?.then(() => "continue"),
            ].filter((p) => p !== undefined));
            this._continueSignal = undefined;
            if (nextAction === "stop")
                break;
        }
        this._stopPromise.resolve();
    }
    async stop() {
        if (!this._stopSignal)
            return;
        // Signal to the task runner that it should stop
        this._stopPromise = createDeferredPromise();
        this._stopSignal.resolve();
        await this._stopPromise;
    }
}
//# sourceMappingURL=Task.js.map