import { CommandClasses, EncapsulationFlags, ZWaveError, ZWaveErrorCodes, } from "@zwave-js/core";
import { MAX_ASSOCIATIONS } from "./_shared.js";
export async function handleAssociationGet(ctx, controller, node, command) {
    // We only "support" the lifeline group
    const groupId = 1;
    const endpoint = node.getEndpoint(command.endpointIndex) ?? node;
    // We are being queried, so the device may actually not support the CC, just control it.
    // Using the commandClasses property would throw in that case
    const api = endpoint
        .createAPI(CommandClasses.Association, false)
        .withOptions({
        // Answer with the same encapsulation as asked, but omit
        // Supervision as it shouldn't be used for Get-Report flows
        encapsulationFlags: command.encapsulationFlags
            & ~EncapsulationFlags.Supervision,
    });
    const nodeIds = controller.associations.filter((a) => a.endpoint == undefined)
        .map((a) => a.nodeId) ?? [];
    await api.sendReport({
        groupId,
        maxNodes: MAX_ASSOCIATIONS,
        nodeIds,
        reportsToFollow: 0,
    });
}
export function handleAssociationSet(ctx, controller, node, command) {
    if (command.groupId !== 1) {
        // We only "support" the lifeline group.
        throw new ZWaveError(`Association group ${command.groupId} is not supported.`, ZWaveErrorCodes.CC_OperationFailed);
    }
    // Ignore associations that already exist
    const newAssociations = command.nodeIds.filter((newNodeId) => !controller.associations.some(({ nodeId, endpoint }) => endpoint === undefined && nodeId === newNodeId)).map((nodeId) => ({ nodeId }));
    const associations = [...controller.associations];
    associations.push(...newAssociations);
    // Report error if the association group is already full
    if (associations.length > MAX_ASSOCIATIONS) {
        throw new ZWaveError(`Association group ${command.groupId} is full`, ZWaveErrorCodes.CC_OperationFailed);
    }
    controller.associations = associations;
}
export function handleAssociationRemove(ctx, controller, node, command) {
    // Allow accessing the lifeline group or all groups (which is the same)
    if (!!command.groupId && command.groupId !== 1) {
        // We only "support" the lifeline group
        return;
    }
    if (!command.nodeIds?.length) {
        // clear
        controller.associations = [];
    }
    else {
        controller.associations = controller
            .associations.filter(({ nodeId, endpoint }) => endpoint === undefined
            && !command.nodeIds.includes(nodeId));
    }
}
export async function handleAssociationSpecificGroupGet(ctx, node, command) {
    const endpoint = node.getEndpoint(command.endpointIndex) ?? node;
    // We are being queried, so the device may actually not support the CC, just control it.
    // Using the commandClasses property would throw in that case
    const api = endpoint
        .createAPI(CommandClasses.Association, false)
        .withOptions({
        // Answer with the same encapsulation as asked, but omit
        // Supervision as it shouldn't be used for Get-Report flows
        encapsulationFlags: command.encapsulationFlags
            & ~EncapsulationFlags.Supervision,
    });
    // We don't support this feature.
    // It is RECOMMENDED that the value 0 is returned by non-supporting devices.
    await api.reportSpecificGroup(0);
}
//# sourceMappingURL=AssociationCC.js.map