#!/usr/bin/env node
"use strict";
var __import_meta_url = typeof document === "undefined" ? new (require("url".replace("", ""))).URL("file:" + __filename).href : document.currentScript && document.currentScript.src || new URL("main.js", document.baseURI).href;
var import_path = require("path");
var import_zwave_js = require("zwave-js");
var import_server = require("../lib/server.js");
var import_mock = require("../mock/index.js");
var import_parse_args = require("../util/parse-args.js");
var import_node_module = require("node:module");
const require2 = (0, import_node_module.createRequire)(__import_meta_url);
const normalizeKey = (key, keyName) => {
  if (Buffer.isBuffer(key))
    return key;
  if (key.length === 32)
    return Buffer.from(key, "hex");
  key = key.toLowerCase();
  if (key.includes("0x"))
    return Buffer.from(key.replace(/0x/g, "").replace(/, /g, ""), "hex");
  throw new Error(`Invalid key format for ${keyName} option`);
};
(async () => {
  const args = (0, import_parse_args.parseArgs)([
    "_",
    "config",
    "mock-driver",
    "port",
    "host",
    "disable-dns-sd"
  ]);
  if (args.port) {
    if (typeof args["port"] !== "number") {
      throw new Error("port must be a valid integer");
    }
  }
  if (args["mock-driver"]) {
    args._.push("mock-serial-port");
  }
  if (args._.length < 1) {
    console.error("Error: Missing path to serial port");
    return;
  }
  const serialPort = args._[0];
  let configPath = args.config;
  if (configPath && configPath.substring(0, 1) !== "/") {
    configPath = (0, import_path.resolve)(process.cwd(), configPath);
  }
  let options;
  let presetNames;
  if (configPath) {
    try {
      ({ presets: presetNames, ...options } = require2(configPath));
      if (options.securityKeys?.S0_Legacy && options.networkKey) {
        throw new Error("Both `networkKey` and `securityKeys.S0_Legacy` options are present in the config. Remove `networkKey`.");
      }
      const securityKeyNames = [
        "S0_Legacy",
        "S2_AccessControl",
        "S2_Authenticated",
        "S2_Unauthenticated"
      ];
      if (options.securityKeys) {
        for (const key of securityKeyNames) {
          if (key in options.securityKeys) {
            options.securityKeys[key] = normalizeKey(options.securityKeys[key], `securityKeys.${key}`);
          }
        }
      }
      if (options.networkKey) {
        if (!options.securityKeys)
          options.securityKeys = {};
        options.securityKeys.S0_Legacy = normalizeKey(options.networkKey, "networkKey");
        console.warn("The `networkKey` option is deprecated in favor of `securityKeys` option. To eliminate this warning, move your networkKey into the securityKeys.S0_Legacy option. Refer to the Z-Wave JS docs for more information");
        delete options.networkKey;
      } else if (!options.securityKeys?.S0_Legacy)
        throw new Error("Error: `securityKeys.S0_Legacy` key is missing.");
      const securityKeysLongRangeNames = [
        "S2_AccessControl",
        "S2_Authenticated"
      ];
      if (options.securityKeysLongRange) {
        for (const key of securityKeysLongRangeNames) {
          if (key in options.securityKeysLongRange) {
            options.securityKeysLongRange[key] = normalizeKey(options.securityKeysLongRange[key], `securityKeysLongRange.${key}`);
          }
        }
      }
    } catch (err) {
      console.error(`Error: failed loading config file ${configPath}`);
      console.error(err);
      return;
    }
  }
  if (!options) {
    options = { emitValueUpdateAfterSetValue: true };
  } else if (!("emitValueUpdateAfterSetValue" in options)) {
    options["emitValueUpdateAfterSetValue"] = true;
  } else if (!options["emitValueUpdateAfterSetValue"]) {
    console.warn("Because `emitValueUpdateAfterSetValue` is set to false, multi-client setups will not work as expected. In particular, clients will not see value updates that are initiated by another client.");
  }
  let presets;
  if (presetNames !== void 0) {
    if (typeof presetNames === "string") {
      presetNames = [presetNames];
    } else if (!Array.isArray(presetNames) || !presetNames.every((p) => typeof p === "string")) {
      throw new Error("presets must be an array of strings or a string if provided");
    }
    presets = presetNames.map((name) => import_zwave_js.driverPresets[name]).filter((preset) => preset !== void 0);
  }
  const driver = args["mock-driver"] ? (0, import_mock.createMockDriver)() : new import_zwave_js.Driver(serialPort, options, ...presets ?? []);
  driver.on("error", (e) => {
    console.error("Error in driver", e);
    if (e instanceof import_zwave_js.ZWaveError && e.code === import_zwave_js.ZWaveErrorCodes.Driver_Failed) {
      handleShutdown(1);
    }
  });
  let server;
  driver.once("driver ready", async () => {
    server = new import_server.ZwavejsServer(driver, {
      port: args.port,
      host: args.host,
      enableDNSServiceDiscovery: !args["disable-dns-sd"]
    });
    await server.start(true);
  });
  await driver.start();
  let closing = false;
  const handleShutdown = async (exitCode = 0) => {
    if (closing) {
      process.exit(exitCode);
    }
    closing = true;
    console.log("Shutting down");
    if (server) {
      await server.destroy();
    }
    if (driver) {
      await driver.destroy();
    }
    process.exit(exitCode);
  };
  process.on("SIGINT", () => handleShutdown(0));
  process.on("SIGTERM", () => handleShutdown(0));
})().catch((err) => {
  console.error("Unable to start driver", err);
  process.exit(1);
});
//# sourceMappingURL=server.js.map
