"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.openIssueCreationLink = exports.maybePrintIssueCreationPrompt = exports.shouldRecommendIssue = exports.getPackageVCSDetails = void 0;
const chalk_1 = __importDefault(require("chalk"));
const open_1 = __importDefault(require("open"));
const querystring_1 = require("querystring");
const path_1 = require("./path");
const repoSpecifier = /^([\w.-]+)\/([\w.-]+)$/;
const githubURL = /github.com(:|\/)([\w.-]+\/[\w.-]+?)(.git|\/.*)?$/;
function parseRepoString(repository) {
    if (repository.startsWith("github:")) {
        repository = repository.replace(/^github:/, "");
    }
    const urlMatch = repository.match(githubURL);
    if (urlMatch) {
        repository = urlMatch[2];
    }
    const specMatch = repository.match(repoSpecifier);
    if (!specMatch) {
        return null;
    }
    const [, org, repo] = specMatch;
    return { org, repo, provider: "GitHub" };
}
function getPackageVCSDetails(packageDetails) {
    const repository = require(path_1.resolve(path_1.join(packageDetails.path, "package.json")))
        .repository;
    if (!repository) {
        return null;
    }
    if (typeof repository === "string") {
        return parseRepoString(repository);
    }
    else if (typeof repository === "object" &&
        typeof repository.url === "string") {
        return parseRepoString(repository.url);
    }
}
exports.getPackageVCSDetails = getPackageVCSDetails;
function createIssueUrl({ vcs, packageDetails, packageVersion, diff, }) {
    return `https://github.com/${vcs === null || vcs === void 0 ? void 0 : vcs.org}/${vcs === null || vcs === void 0 ? void 0 : vcs.repo}/issues/new?${querystring_1.stringify({
        title: "",
        body: `Hi! 👋 
      
Firstly, thanks for your work on this project! 🙂

Today I used [patch-package](https://github.com/ds300/patch-package) to patch \`${packageDetails.name}@${packageVersion}\` for the project I'm working on.

<!-- 🔺️🔺️🔺️ PLEASE REPLACE THIS BLOCK with a description of your problem, and any other relevant context 🔺️🔺️🔺️ -->

Here is the diff that solved my problem:

\`\`\`diff
${diff}
\`\`\`

<em>This issue body was [partially generated by patch-package](https://github.com/ds300/patch-package/issues/296).</em>
`,
    })}`;
}
function shouldRecommendIssue(vcsDetails) {
    if (!vcsDetails) {
        return true;
    }
    const { repo, org } = vcsDetails;
    if (repo === "DefinitelyTyped" && org === "DefinitelyTyped") {
        return false;
    }
    return true;
}
exports.shouldRecommendIssue = shouldRecommendIssue;
function maybePrintIssueCreationPrompt(vcs, packageDetails, packageManager) {
    if (vcs) {
        console.log(`💡 ${chalk_1.default.bold(packageDetails.name)} is on ${vcs.provider}! To draft an issue based on your patch run

    ${packageManager === "yarn" ? "yarn" : "npx"} patch-package ${packageDetails.pathSpecifier} --create-issue
`);
    }
}
exports.maybePrintIssueCreationPrompt = maybePrintIssueCreationPrompt;
function openIssueCreationLink({ packageDetails, patchFileContents, packageVersion, patchPath, }) {
    const vcs = getPackageVCSDetails(packageDetails);
    if (!vcs) {
        console.log(`Error: Couldn't find VCS details for ${packageDetails.pathSpecifier}`);
        process.exit(1);
    }
    // trim off trailing newline since we add an extra one in the markdown block
    if (patchFileContents.endsWith("\n")) {
        patchFileContents = patchFileContents.slice(0, -1);
    }
    let issueUrl = createIssueUrl({
        vcs,
        packageDetails,
        packageVersion,
        diff: patchFileContents,
    });
    const urlExceedsLimit = patchFileContents.length > 1950;
    if (urlExceedsLimit) {
        const diffMessage = `<!-- 🔺️🔺️🔺️ PLEASE REPLACE THIS BLOCK with the diff contents of ${patchPath
            .split("/")
            .pop()}. 🔺️🔺️🔺️ -->`;
        console.log(`📋 Copy the contents in [ ${patchPath} ] and paste it in the new issue's diff section.`);
        issueUrl = createIssueUrl({
            vcs,
            packageDetails,
            packageVersion,
            diff: diffMessage,
        });
    }
    open_1.default(issueUrl);
}
exports.openIssueCreationLink = openIssueCreationLink;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiY3JlYXRlSXNzdWUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyIuLi9zcmMvY3JlYXRlSXNzdWUudHMiXSwibmFtZXMiOltdLCJtYXBwaW5ncyI6Ijs7Ozs7O0FBQUEsa0RBQXlCO0FBQ3pCLGdEQUF1QjtBQUN2Qiw2Q0FBdUM7QUFHdkMsaUNBQXNDO0FBRXRDLE1BQU0sYUFBYSxHQUFHLHdCQUF3QixDQUFBO0FBQzlDLE1BQU0sU0FBUyxHQUFHLGtEQUFrRCxDQUFBO0FBV3BFLFNBQVMsZUFBZSxDQUFDLFVBQWtCO0lBQ3pDLElBQUksVUFBVSxDQUFDLFVBQVUsQ0FBQyxTQUFTLENBQUMsRUFBRTtRQUNwQyxVQUFVLEdBQUcsVUFBVSxDQUFDLE9BQU8sQ0FBQyxVQUFVLEVBQUUsRUFBRSxDQUFDLENBQUE7S0FDaEQ7SUFDRCxNQUFNLFFBQVEsR0FBRyxVQUFVLENBQUMsS0FBSyxDQUFDLFNBQVMsQ0FBQyxDQUFBO0lBQzVDLElBQUksUUFBUSxFQUFFO1FBQ1osVUFBVSxHQUFHLFFBQVEsQ0FBQyxDQUFDLENBQUMsQ0FBQTtLQUN6QjtJQUVELE1BQU0sU0FBUyxHQUFHLFVBQVUsQ0FBQyxLQUFLLENBQUMsYUFBYSxDQUFDLENBQUE7SUFFakQsSUFBSSxDQUFDLFNBQVMsRUFBRTtRQUNkLE9BQU8sSUFBSSxDQUFBO0tBQ1o7SUFDRCxNQUFNLENBQUMsRUFBRSxHQUFHLEVBQUUsSUFBSSxDQUFDLEdBQUcsU0FBUyxDQUFBO0lBRS9CLE9BQU8sRUFBRSxHQUFHLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxRQUFRLEVBQUUsQ0FBQTtBQUMxQyxDQUFDO0FBRUQsU0FBZ0Isb0JBQW9CLENBQUMsY0FBOEI7SUFDakUsTUFBTSxVQUFVLEdBQUcsT0FBTyxDQUFDLGNBQU8sQ0FBQyxXQUFJLENBQUMsY0FBYyxDQUFDLElBQUksRUFBRSxjQUFjLENBQUMsQ0FBQyxDQUFDO1NBQzNFLFVBQWtELENBQUE7SUFFckQsSUFBSSxDQUFDLFVBQVUsRUFBRTtRQUNmLE9BQU8sSUFBSSxDQUFBO0tBQ1o7SUFDRCxJQUFJLE9BQU8sVUFBVSxLQUFLLFFBQVEsRUFBRTtRQUNsQyxPQUFPLGVBQWUsQ0FBQyxVQUFVLENBQUMsQ0FBQTtLQUNuQztTQUFNLElBQ0wsT0FBTyxVQUFVLEtBQUssUUFBUTtRQUM5QixPQUFPLFVBQVUsQ0FBQyxHQUFHLEtBQUssUUFBUSxFQUNsQztRQUNBLE9BQU8sZUFBZSxDQUFDLFVBQVUsQ0FBQyxHQUFHLENBQUMsQ0FBQTtLQUN2QztBQUNILENBQUM7QUFmRCxvREFlQztBQUVELFNBQVMsY0FBYyxDQUFDLEVBQ3RCLEdBQUcsRUFDSCxjQUFjLEVBQ2QsY0FBYyxFQUNkLElBQUksR0FNTDtJQUNDLE9BQU8sc0JBQXNCLEdBQUcsYUFBSCxHQUFHLHVCQUFILEdBQUcsQ0FBRSxHQUFHLElBQUksR0FBRyxhQUFILEdBQUcsdUJBQUgsR0FBRyxDQUFFLElBQUksZUFBZSx1QkFBUyxDQUFDO1FBQ3pFLEtBQUssRUFBRSxFQUFFO1FBQ1QsSUFBSSxFQUFFOzs7O2tGQUl3RSxjQUFjLENBQUMsSUFBSSxJQUFJLGNBQWM7Ozs7Ozs7RUFPckgsSUFBSTs7OztDQUlMO0tBQ0UsQ0FBQyxFQUFFLENBQUE7QUFDTixDQUFDO0FBRUQsU0FBZ0Isb0JBQW9CLENBQ2xDLFVBQW1EO0lBRW5ELElBQUksQ0FBQyxVQUFVLEVBQUU7UUFDZixPQUFPLElBQUksQ0FBQTtLQUNaO0lBRUQsTUFBTSxFQUFFLElBQUksRUFBRSxHQUFHLEVBQUUsR0FBRyxVQUFVLENBQUE7SUFDaEMsSUFBSSxJQUFJLEtBQUssaUJBQWlCLElBQUksR0FBRyxLQUFLLGlCQUFpQixFQUFFO1FBQzNELE9BQU8sS0FBSyxDQUFBO0tBQ2I7SUFDRCxPQUFPLElBQUksQ0FBQTtBQUNiLENBQUM7QUFaRCxvREFZQztBQUVELFNBQWdCLDZCQUE2QixDQUMzQyxHQUE0QyxFQUM1QyxjQUE4QixFQUM5QixjQUE4QjtJQUU5QixJQUFJLEdBQUcsRUFBRTtRQUNQLE9BQU8sQ0FBQyxHQUFHLENBQUMsTUFBTSxlQUFLLENBQUMsSUFBSSxDQUFDLGNBQWMsQ0FBQyxJQUFJLENBQUMsVUFDL0MsR0FBRyxDQUFDLFFBQ047O01BRUUsY0FBYyxLQUFLLE1BQU0sQ0FBQyxDQUFDLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxLQUFLLGtCQUMxQyxjQUFjLENBQUMsYUFDakI7Q0FDSCxDQUFDLENBQUE7S0FDQztBQUNILENBQUM7QUFmRCxzRUFlQztBQUVELFNBQWdCLHFCQUFxQixDQUFDLEVBQ3BDLGNBQWMsRUFDZCxpQkFBaUIsRUFDakIsY0FBYyxFQUNkLFNBQVMsR0FNVjtJQUNDLE1BQU0sR0FBRyxHQUFHLG9CQUFvQixDQUFDLGNBQWMsQ0FBQyxDQUFBO0lBRWhELElBQUksQ0FBQyxHQUFHLEVBQUU7UUFDUixPQUFPLENBQUMsR0FBRyxDQUNULHdDQUF3QyxjQUFjLENBQUMsYUFBYSxFQUFFLENBQ3ZFLENBQUE7UUFDRCxPQUFPLENBQUMsSUFBSSxDQUFDLENBQUMsQ0FBQyxDQUFBO0tBQ2hCO0lBRUQsNEVBQTRFO0lBQzVFLElBQUksaUJBQWlCLENBQUMsUUFBUSxDQUFDLElBQUksQ0FBQyxFQUFFO1FBQ3BDLGlCQUFpQixHQUFHLGlCQUFpQixDQUFDLEtBQUssQ0FBQyxDQUFDLEVBQUUsQ0FBQyxDQUFDLENBQUMsQ0FBQTtLQUNuRDtJQUVELElBQUksUUFBUSxHQUFHLGNBQWMsQ0FBQztRQUM1QixHQUFHO1FBQ0gsY0FBYztRQUNkLGNBQWM7UUFDZCxJQUFJLEVBQUUsaUJBQWlCO0tBQ3hCLENBQUMsQ0FBQTtJQUVGLE1BQU0sZUFBZSxHQUFHLGlCQUFpQixDQUFDLE1BQU0sR0FBRyxJQUFJLENBQUE7SUFFdkQsSUFBSSxlQUFlLEVBQUU7UUFDbkIsTUFBTSxXQUFXLEdBQUcsc0VBQXNFLFNBQVM7YUFDaEcsS0FBSyxDQUFDLEdBQUcsQ0FBQzthQUNWLEdBQUcsRUFBRSxpQkFBaUIsQ0FBQTtRQUN6QixPQUFPLENBQUMsR0FBRyxDQUNULDZCQUE2QixTQUFTLGtEQUFrRCxDQUN6RixDQUFBO1FBQ0QsUUFBUSxHQUFHLGNBQWMsQ0FBQztZQUN4QixHQUFHO1lBQ0gsY0FBYztZQUNkLGNBQWM7WUFDZCxJQUFJLEVBQUUsV0FBVztTQUNsQixDQUFDLENBQUE7S0FDSDtJQUNELGNBQUksQ0FBQyxRQUFRLENBQUMsQ0FBQTtBQUNoQixDQUFDO0FBakRELHNEQWlEQyIsInNvdXJjZXNDb250ZW50IjpbImltcG9ydCBjaGFsayBmcm9tIFwiY2hhbGtcIlxuaW1wb3J0IG9wZW4gZnJvbSBcIm9wZW5cIlxuaW1wb3J0IHsgc3RyaW5naWZ5IH0gZnJvbSBcInF1ZXJ5c3RyaW5nXCJcbmltcG9ydCB7IFBhY2thZ2VNYW5hZ2VyIH0gZnJvbSBcIi4vZGV0ZWN0UGFja2FnZU1hbmFnZXJcIlxuaW1wb3J0IHsgUGFja2FnZURldGFpbHMgfSBmcm9tIFwiLi9QYWNrYWdlRGV0YWlsc1wiXG5pbXBvcnQgeyBqb2luLCByZXNvbHZlIH0gZnJvbSBcIi4vcGF0aFwiXG5cbmNvbnN0IHJlcG9TcGVjaWZpZXIgPSAvXihbXFx3Li1dKylcXC8oW1xcdy4tXSspJC9cbmNvbnN0IGdpdGh1YlVSTCA9IC9naXRodWIuY29tKDp8XFwvKShbXFx3Li1dK1xcL1tcXHcuLV0rPykoLmdpdHxcXC8uKik/JC9cblxudHlwZSBWQ1MgPVxuICB8IHtcbiAgICAgIHJlcG86IHN0cmluZ1xuICAgICAgb3JnOiBzdHJpbmdcbiAgICAgIHByb3ZpZGVyOiBcIkdpdEh1YlwiXG4gICAgfVxuICB8IG51bGxcbiAgfCB1bmRlZmluZWRcblxuZnVuY3Rpb24gcGFyc2VSZXBvU3RyaW5nKHJlcG9zaXRvcnk6IHN0cmluZyk6IFZDUyB7XG4gIGlmIChyZXBvc2l0b3J5LnN0YXJ0c1dpdGgoXCJnaXRodWI6XCIpKSB7XG4gICAgcmVwb3NpdG9yeSA9IHJlcG9zaXRvcnkucmVwbGFjZSgvXmdpdGh1YjovLCBcIlwiKVxuICB9XG4gIGNvbnN0IHVybE1hdGNoID0gcmVwb3NpdG9yeS5tYXRjaChnaXRodWJVUkwpXG4gIGlmICh1cmxNYXRjaCkge1xuICAgIHJlcG9zaXRvcnkgPSB1cmxNYXRjaFsyXVxuICB9XG5cbiAgY29uc3Qgc3BlY01hdGNoID0gcmVwb3NpdG9yeS5tYXRjaChyZXBvU3BlY2lmaWVyKVxuXG4gIGlmICghc3BlY01hdGNoKSB7XG4gICAgcmV0dXJuIG51bGxcbiAgfVxuICBjb25zdCBbLCBvcmcsIHJlcG9dID0gc3BlY01hdGNoXG5cbiAgcmV0dXJuIHsgb3JnLCByZXBvLCBwcm92aWRlcjogXCJHaXRIdWJcIiB9XG59XG5cbmV4cG9ydCBmdW5jdGlvbiBnZXRQYWNrYWdlVkNTRGV0YWlscyhwYWNrYWdlRGV0YWlsczogUGFja2FnZURldGFpbHMpOiBWQ1Mge1xuICBjb25zdCByZXBvc2l0b3J5ID0gcmVxdWlyZShyZXNvbHZlKGpvaW4ocGFja2FnZURldGFpbHMucGF0aCwgXCJwYWNrYWdlLmpzb25cIikpKVxuICAgIC5yZXBvc2l0b3J5IGFzIHVuZGVmaW5lZCB8IHN0cmluZyB8IHsgdXJsOiBzdHJpbmcgfVxuXG4gIGlmICghcmVwb3NpdG9yeSkge1xuICAgIHJldHVybiBudWxsXG4gIH1cbiAgaWYgKHR5cGVvZiByZXBvc2l0b3J5ID09PSBcInN0cmluZ1wiKSB7XG4gICAgcmV0dXJuIHBhcnNlUmVwb1N0cmluZyhyZXBvc2l0b3J5KVxuICB9IGVsc2UgaWYgKFxuICAgIHR5cGVvZiByZXBvc2l0b3J5ID09PSBcIm9iamVjdFwiICYmXG4gICAgdHlwZW9mIHJlcG9zaXRvcnkudXJsID09PSBcInN0cmluZ1wiXG4gICkge1xuICAgIHJldHVybiBwYXJzZVJlcG9TdHJpbmcocmVwb3NpdG9yeS51cmwpXG4gIH1cbn1cblxuZnVuY3Rpb24gY3JlYXRlSXNzdWVVcmwoe1xuICB2Y3MsXG4gIHBhY2thZ2VEZXRhaWxzLFxuICBwYWNrYWdlVmVyc2lvbixcbiAgZGlmZixcbn06IHtcbiAgdmNzOiBWQ1NcbiAgcGFja2FnZURldGFpbHM6IFBhY2thZ2VEZXRhaWxzXG4gIHBhY2thZ2VWZXJzaW9uOiBzdHJpbmdcbiAgZGlmZjogc3RyaW5nXG59KTogc3RyaW5nIHtcbiAgcmV0dXJuIGBodHRwczovL2dpdGh1Yi5jb20vJHt2Y3M/Lm9yZ30vJHt2Y3M/LnJlcG99L2lzc3Vlcy9uZXc/JHtzdHJpbmdpZnkoe1xuICAgIHRpdGxlOiBcIlwiLFxuICAgIGJvZHk6IGBIaSEg8J+RiyBcbiAgICAgIFxuRmlyc3RseSwgdGhhbmtzIGZvciB5b3VyIHdvcmsgb24gdGhpcyBwcm9qZWN0ISDwn5mCXG5cblRvZGF5IEkgdXNlZCBbcGF0Y2gtcGFja2FnZV0oaHR0cHM6Ly9naXRodWIuY29tL2RzMzAwL3BhdGNoLXBhY2thZ2UpIHRvIHBhdGNoIFxcYCR7cGFja2FnZURldGFpbHMubmFtZX1AJHtwYWNrYWdlVmVyc2lvbn1cXGAgZm9yIHRoZSBwcm9qZWN0IEknbSB3b3JraW5nIG9uLlxuXG48IS0tIPCflLrvuI/wn5S677iP8J+Uuu+4jyBQTEVBU0UgUkVQTEFDRSBUSElTIEJMT0NLIHdpdGggYSBkZXNjcmlwdGlvbiBvZiB5b3VyIHByb2JsZW0sIGFuZCBhbnkgb3RoZXIgcmVsZXZhbnQgY29udGV4dCDwn5S677iP8J+Uuu+4j/CflLrvuI8gLS0+XG5cbkhlcmUgaXMgdGhlIGRpZmYgdGhhdCBzb2x2ZWQgbXkgcHJvYmxlbTpcblxuXFxgXFxgXFxgZGlmZlxuJHtkaWZmfVxuXFxgXFxgXFxgXG5cbjxlbT5UaGlzIGlzc3VlIGJvZHkgd2FzIFtwYXJ0aWFsbHkgZ2VuZXJhdGVkIGJ5IHBhdGNoLXBhY2thZ2VdKGh0dHBzOi8vZ2l0aHViLmNvbS9kczMwMC9wYXRjaC1wYWNrYWdlL2lzc3Vlcy8yOTYpLjwvZW0+XG5gLFxuICB9KX1gXG59XG5cbmV4cG9ydCBmdW5jdGlvbiBzaG91bGRSZWNvbW1lbmRJc3N1ZShcbiAgdmNzRGV0YWlsczogUmV0dXJuVHlwZTx0eXBlb2YgZ2V0UGFja2FnZVZDU0RldGFpbHM+LFxuKSB7XG4gIGlmICghdmNzRGV0YWlscykge1xuICAgIHJldHVybiB0cnVlXG4gIH1cblxuICBjb25zdCB7IHJlcG8sIG9yZyB9ID0gdmNzRGV0YWlsc1xuICBpZiAocmVwbyA9PT0gXCJEZWZpbml0ZWx5VHlwZWRcIiAmJiBvcmcgPT09IFwiRGVmaW5pdGVseVR5cGVkXCIpIHtcbiAgICByZXR1cm4gZmFsc2VcbiAgfVxuICByZXR1cm4gdHJ1ZVxufVxuXG5leHBvcnQgZnVuY3Rpb24gbWF5YmVQcmludElzc3VlQ3JlYXRpb25Qcm9tcHQoXG4gIHZjczogUmV0dXJuVHlwZTx0eXBlb2YgZ2V0UGFja2FnZVZDU0RldGFpbHM+LFxuICBwYWNrYWdlRGV0YWlsczogUGFja2FnZURldGFpbHMsXG4gIHBhY2thZ2VNYW5hZ2VyOiBQYWNrYWdlTWFuYWdlcixcbikge1xuICBpZiAodmNzKSB7XG4gICAgY29uc29sZS5sb2coYPCfkqEgJHtjaGFsay5ib2xkKHBhY2thZ2VEZXRhaWxzLm5hbWUpfSBpcyBvbiAke1xuICAgICAgdmNzLnByb3ZpZGVyXG4gICAgfSEgVG8gZHJhZnQgYW4gaXNzdWUgYmFzZWQgb24geW91ciBwYXRjaCBydW5cblxuICAgICR7cGFja2FnZU1hbmFnZXIgPT09IFwieWFyblwiID8gXCJ5YXJuXCIgOiBcIm5weFwifSBwYXRjaC1wYWNrYWdlICR7XG4gICAgICBwYWNrYWdlRGV0YWlscy5wYXRoU3BlY2lmaWVyXG4gICAgfSAtLWNyZWF0ZS1pc3N1ZVxuYClcbiAgfVxufVxuXG5leHBvcnQgZnVuY3Rpb24gb3Blbklzc3VlQ3JlYXRpb25MaW5rKHtcbiAgcGFja2FnZURldGFpbHMsXG4gIHBhdGNoRmlsZUNvbnRlbnRzLFxuICBwYWNrYWdlVmVyc2lvbixcbiAgcGF0Y2hQYXRoLFxufToge1xuICBwYWNrYWdlRGV0YWlsczogUGFja2FnZURldGFpbHNcbiAgcGF0Y2hGaWxlQ29udGVudHM6IHN0cmluZ1xuICBwYWNrYWdlVmVyc2lvbjogc3RyaW5nXG4gIHBhdGNoUGF0aDogc3RyaW5nXG59KSB7XG4gIGNvbnN0IHZjcyA9IGdldFBhY2thZ2VWQ1NEZXRhaWxzKHBhY2thZ2VEZXRhaWxzKVxuXG4gIGlmICghdmNzKSB7XG4gICAgY29uc29sZS5sb2coXG4gICAgICBgRXJyb3I6IENvdWxkbid0IGZpbmQgVkNTIGRldGFpbHMgZm9yICR7cGFja2FnZURldGFpbHMucGF0aFNwZWNpZmllcn1gLFxuICAgIClcbiAgICBwcm9jZXNzLmV4aXQoMSlcbiAgfVxuXG4gIC8vIHRyaW0gb2ZmIHRyYWlsaW5nIG5ld2xpbmUgc2luY2Ugd2UgYWRkIGFuIGV4dHJhIG9uZSBpbiB0aGUgbWFya2Rvd24gYmxvY2tcbiAgaWYgKHBhdGNoRmlsZUNvbnRlbnRzLmVuZHNXaXRoKFwiXFxuXCIpKSB7XG4gICAgcGF0Y2hGaWxlQ29udGVudHMgPSBwYXRjaEZpbGVDb250ZW50cy5zbGljZSgwLCAtMSlcbiAgfVxuXG4gIGxldCBpc3N1ZVVybCA9IGNyZWF0ZUlzc3VlVXJsKHtcbiAgICB2Y3MsXG4gICAgcGFja2FnZURldGFpbHMsXG4gICAgcGFja2FnZVZlcnNpb24sXG4gICAgZGlmZjogcGF0Y2hGaWxlQ29udGVudHMsXG4gIH0pXG5cbiAgY29uc3QgdXJsRXhjZWVkc0xpbWl0ID0gcGF0Y2hGaWxlQ29udGVudHMubGVuZ3RoID4gMTk1MFxuXG4gIGlmICh1cmxFeGNlZWRzTGltaXQpIHtcbiAgICBjb25zdCBkaWZmTWVzc2FnZSA9IGA8IS0tIPCflLrvuI/wn5S677iP8J+Uuu+4jyBQTEVBU0UgUkVQTEFDRSBUSElTIEJMT0NLIHdpdGggdGhlIGRpZmYgY29udGVudHMgb2YgJHtwYXRjaFBhdGhcbiAgICAgIC5zcGxpdChcIi9cIilcbiAgICAgIC5wb3AoKX0uIPCflLrvuI/wn5S677iP8J+Uuu+4jyAtLT5gXG4gICAgY29uc29sZS5sb2coXG4gICAgICBg8J+TiyBDb3B5IHRoZSBjb250ZW50cyBpbiBbICR7cGF0Y2hQYXRofSBdIGFuZCBwYXN0ZSBpdCBpbiB0aGUgbmV3IGlzc3VlJ3MgZGlmZiBzZWN0aW9uLmAsXG4gICAgKVxuICAgIGlzc3VlVXJsID0gY3JlYXRlSXNzdWVVcmwoe1xuICAgICAgdmNzLFxuICAgICAgcGFja2FnZURldGFpbHMsXG4gICAgICBwYWNrYWdlVmVyc2lvbixcbiAgICAgIGRpZmY6IGRpZmZNZXNzYWdlLFxuICAgIH0pXG4gIH1cbiAgb3Blbihpc3N1ZVVybClcbn1cbiJdfQ==