/*
 * Decompiled with CFR 0.152.
 */
package de.bluecolored.bluemap.cli;

import de.bluecolored.bluemap.api.gson.MarkerGson;
import de.bluecolored.bluemap.common.BlueMapConfiguration;
import de.bluecolored.bluemap.common.BlueMapService;
import de.bluecolored.bluemap.common.MissingResourcesException;
import de.bluecolored.bluemap.common.addons.AddonLoader;
import de.bluecolored.bluemap.common.api.BlueMapAPIImpl;
import de.bluecolored.bluemap.common.commands.TextFormat;
import de.bluecolored.bluemap.common.config.BlueMapConfigManager;
import de.bluecolored.bluemap.common.config.ConfigurationException;
import de.bluecolored.bluemap.common.config.CoreConfig;
import de.bluecolored.bluemap.common.config.MapConfig;
import de.bluecolored.bluemap.common.config.WebserverConfig;
import de.bluecolored.bluemap.common.metrics.Metrics;
import de.bluecolored.bluemap.common.plugin.MapUpdateService;
import de.bluecolored.bluemap.common.rendermanager.MapUpdatePreparationTask;
import de.bluecolored.bluemap.common.rendermanager.RenderManager;
import de.bluecolored.bluemap.common.rendermanager.RenderTask;
import de.bluecolored.bluemap.common.rendermanager.TileUpdateStrategy;
import de.bluecolored.bluemap.common.web.BlueMapResponseModifier;
import de.bluecolored.bluemap.common.web.FileRequestHandler;
import de.bluecolored.bluemap.common.web.LoggingRequestHandler;
import de.bluecolored.bluemap.common.web.MapRequestHandler;
import de.bluecolored.bluemap.common.web.RoutingRequestHandler;
import de.bluecolored.bluemap.common.web.http.HttpRequestHandler;
import de.bluecolored.bluemap.common.web.http.HttpServer;
import de.bluecolored.bluemap.core.BlueMap;
import de.bluecolored.bluemap.core.logger.Logger;
import de.bluecolored.bluemap.core.map.BmMap;
import de.bluecolored.bluemap.core.storage.MapStorage;
import de.bluecolored.bluemap.core.util.FileHelper;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.net.BindException;
import java.net.InetSocketAddress;
import java.net.UnknownHostException;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.Path;
import java.nio.file.attribute.FileAttribute;
import java.time.Duration;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Map;
import java.util.Set;
import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.TimeUnit;
import java.util.function.Predicate;
import java.util.regex.Pattern;
import org.apache.commons.cli.CommandLine;
import org.apache.commons.cli.DefaultParser;
import org.apache.commons.cli.HelpFormatter;
import org.apache.commons.cli.Option;
import org.apache.commons.cli.Options;
import org.apache.commons.cli.ParseException;
import org.checkerframework.checker.nullness.qual.Nullable;

public class BlueMapCLI {
    private static boolean shutdownInProgress = false;
    private String minecraftVersion = null;
    private Path configFolder = Path.of("config", new String[0]);
    private Path modsFolder = null;

    public void renderMaps(BlueMapService blueMap, boolean watch, TileUpdateStrategy force, boolean forceGenerateWebapp, @Nullable String mapsToRender) throws ConfigurationException, IOException, InterruptedException {
        if (blueMap.getConfig().getWebappConfig().isEnabled()) {
            blueMap.createOrUpdateWebApp(forceGenerateWebapp);
        }
        blueMap.getOrLoadResourcePack();
        final RenderManager renderManager = new RenderManager();
        Predicate<String> mapFilter = mapId -> true;
        if (mapsToRender != null) {
            Set<String> mapsToRenderSet = Set.of(mapsToRender.split(","));
            mapFilter = mapsToRenderSet::contains;
        }
        final Map<String, BmMap> maps = blueMap.getOrLoadMaps(mapFilter);
        ArrayList<MapUpdateService> mapUpdateServices = new ArrayList<MapUpdateService>();
        if (watch) {
            for (BmMap map : maps.values()) {
                try {
                    MapUpdateService watcher = new MapUpdateService(renderManager, map, blueMap.getConfig().getPluginConfig().getUpdateCooldown());
                    watcher.start();
                    mapUpdateServices.add(watcher);
                }
                catch (IOException ex) {
                    Logger.global.logError("Failed to create update-watcher for map: " + map.getId() + " (This means the map might not automatically update)", ex);
                }
                catch (UnsupportedOperationException ex) {
                    Logger.global.logWarning("Update-watcher for map '" + map.getId() + "' is not supported for the world-type. (This means the map might not automatically update)");
                }
            }
        }
        for (BmMap map : maps.values()) {
            renderManager.scheduleRenderTask(MapUpdatePreparationTask.builder().map(map).force(force).taskConsumer(renderManager::scheduleRenderTaskNext).build());
        }
        BlueMapAPIImpl api = new BlueMapAPIImpl(blueMap, null);
        api.register();
        Logger.global.logInfo("Start updating " + maps.size() + " maps ...");
        renderManager.start(blueMap.getConfig().getCoreConfig().resolveRenderThreadCount());
        Timer timer = new Timer("BlueMap-CLI-Timer", true);
        TimerTask updateInfoTask = new TimerTask(this){

            @Override
            public void run() {
                RenderTask task = renderManager.getCurrentRenderTask();
                if (task == null) {
                    return;
                }
                double progress = task.estimateProgress();
                long etaMs = renderManager.estimateCurrentRenderTaskTimeRemaining();
                String eta = "";
                if (etaMs > 0L) {
                    Duration duration = Duration.of(etaMs, ChronoUnit.MILLIS);
                    eta = " (ETA: %s)".formatted(TextFormat.duration(duration));
                }
                Logger.global.logInfo(task.getDescription() + ": " + (double)Math.round(progress * 100000.0) / 1000.0 + "%" + eta);
            }
        };
        timer.scheduleAtFixedRate(updateInfoTask, TimeUnit.SECONDS.toMillis(10L), TimeUnit.SECONDS.toMillis(10L));
        TimerTask saveTask = new TimerTask(this){

            @Override
            public void run() {
                for (BmMap map : maps.values()) {
                    map.save();
                }
            }
        };
        timer.scheduleAtFixedRate(saveTask, TimeUnit.MINUTES.toMillis(2L), TimeUnit.MINUTES.toMillis(2L));
        Runnable shutdown = () -> {
            Logger.global.logInfo("Stopping...");
            api.unregister();
            updateInfoTask.cancel();
            saveTask.cancel();
            mapUpdateServices.forEach(MapUpdateService::close);
            mapUpdateServices.clear();
            renderManager.removeAllRenderTasks();
            try {
                renderManager.awaitIdle(true);
            }
            catch (InterruptedException ex) {
                Thread.currentThread().interrupt();
            }
            renderManager.stop();
            try {
                renderManager.awaitShutdown();
            }
            catch (InterruptedException e) {
                Logger.global.logError("Unexpected interruption: ", e);
            }
            Logger.global.logInfo("Saving...");
            saveTask.run();
            Logger.global.logInfo("Stopped.");
        };
        Thread shutdownHook = new Thread(() -> {
            shutdownInProgress = true;
            shutdown.run();
        }, "BlueMap-CLI-ShutdownHook");
        Runtime.getRuntime().addShutdownHook(shutdownHook);
        if (blueMap.getConfig().getCoreConfig().isMetrics()) {
            Metrics.sendReportAsync("cli", blueMap.getOrLoadMinecraftVersion().getId());
        }
        renderManager.awaitIdle();
        if (shutdownInProgress) {
            return;
        }
        Logger.global.logInfo("Your maps are now all up-to-date!");
        if (watch) {
            updateInfoTask.cancel();
            Logger.global.logInfo("Waiting for changes on the world-files...");
        } else {
            Runtime.getRuntime().removeShutdownHook(shutdownHook);
            shutdown.run();
        }
    }

    public void updateMarkers(BlueMapService blueMap, @Nullable String mapsToUpdate) {
        Predicate<String> mapFilter = mapId -> true;
        if (mapsToUpdate != null) {
            Set<String> mapsToRenderSet = Set.of(mapsToUpdate.split(","));
            mapFilter = mapsToRenderSet::contains;
        }
        for (Map.Entry<String, MapConfig> entry : blueMap.getConfig().getMapConfigs().entrySet()) {
            String mapId2 = entry.getKey();
            MapConfig mapConfig = entry.getValue();
            if (!mapFilter.test(mapId2)) {
                return;
            }
            try {
                MapStorage storage = blueMap.getOrLoadStorage(mapConfig.getStorage()).map(mapId2);
                try (OutputStream out = storage.markers().write();
                     OutputStreamWriter writer = new OutputStreamWriter(out, StandardCharsets.UTF_8);){
                    MarkerGson.INSTANCE.toJson(mapConfig.parseMarkerSets(), (Appendable)writer);
                }
                Logger.global.logInfo("Updated markers for map '" + mapId2 + "'");
            }
            catch (Exception ex) {
                Logger.global.logError("Failed to save markers for map '" + mapId2 + "'!", ex);
            }
        }
    }

    public void startWebserver(BlueMapService blueMap, boolean verbose) throws IOException, ConfigurationException, InterruptedException {
        Logger.global.logInfo("Starting webserver ...");
        WebserverConfig config = blueMap.getConfig().getWebserverConfig();
        FileHelper.createDirectories(config.getWebroot(), new FileAttribute[0]);
        RoutingRequestHandler routingRequestHandler = new RoutingRequestHandler();
        routingRequestHandler.register(".*", (HttpRequestHandler)new FileRequestHandler(config.getWebroot()));
        for (Map.Entry<String, MapConfig> mapConfigEntry : blueMap.getConfig().getMapConfigs().entrySet()) {
            MapStorage storage = blueMap.getOrLoadStorage(mapConfigEntry.getValue().getStorage()).map(mapConfigEntry.getKey());
            routingRequestHandler.register("maps/" + Pattern.quote(mapConfigEntry.getKey()) + "/(.*)", "$1", (HttpRequestHandler)new MapRequestHandler(storage));
        }
        ArrayList<Logger> webLoggerList = new ArrayList<Logger>();
        if (verbose) {
            webLoggerList.add(Logger.stdOut(true));
        }
        if (config.getLog().getFile() != null) {
            ZonedDateTime zdt = ZonedDateTime.ofInstant(Instant.now(), ZoneId.systemDefault());
            webLoggerList.add(Logger.file(Path.of(String.format(config.getLog().getFile(), zdt), new String[0]), config.getLog().isAppend()));
        }
        HttpRequestHandler handler = new BlueMapResponseModifier(routingRequestHandler);
        handler = new LoggingRequestHandler(handler, config.getLog().getFormat(), Logger.combine(webLoggerList));
        try {
            HttpServer webServer = new HttpServer(handler);
            webServer.bind(new InetSocketAddress(config.resolveIp(), config.getPort()));
            webServer.start();
        }
        catch (UnknownHostException ex) {
            throw new ConfigurationException("BlueMap failed to resolve the ip in your webserver-config.\nCheck if that is correctly configured.", ex);
        }
        catch (BindException ex) {
            throw new ConfigurationException("BlueMap failed to bind to the configured address.\nThis usually happens when the configured port (" + config.getPort() + ") is already in use by some other program.", ex);
        }
        catch (IOException ex) {
            throw new ConfigurationException("BlueMap failed to initialize the webserver.\nCheck your webserver-config if everything is configured correctly.\n(Make sure you DON'T use the same port for bluemap that you also use for your minecraft server)", ex);
        }
    }

    public static void main(String[] args) {
        DefaultParser parser = new DefaultParser();
        BlueMapCLI cli = new BlueMapCLI();
        BlueMapService blueMap = null;
        try {
            CommandLine cmd = parser.parse(BlueMapCLI.createOptions(), args, false);
            if (cmd.hasOption("b")) {
                Logger.global.clear();
                Logger.global.put(Logger.stdOut(true));
            }
            if (cmd.hasOption("l")) {
                Logger.global.put(Logger.file(Path.of(cmd.getOptionValue("l"), new String[0]), cmd.hasOption("a")));
            }
            if (cmd.hasOption("h")) {
                BlueMapCLI.printHelp();
                return;
            }
            if (cmd.hasOption("V")) {
                BlueMapCLI.printVersion();
                return;
            }
            if (cmd.hasOption("c")) {
                cli.configFolder = Path.of(cmd.getOptionValue("c"), new String[0]);
                FileHelper.createDirectories(cli.configFolder, new FileAttribute[0]);
            }
            if (cmd.hasOption("n")) {
                cli.modsFolder = Path.of(cmd.getOptionValue("n"), new String[0]);
                if (!Files.isDirectory(cli.modsFolder, new LinkOption[0])) {
                    throw new ConfigurationException("Mods folder does not exist: " + String.valueOf(cli.modsFolder));
                }
            }
            if (cmd.hasOption("v")) {
                cli.minecraftVersion = cmd.getOptionValue("v");
            }
            Path packsFolder = cli.configFolder.resolve("packs");
            Files.createDirectories(packsFolder, new FileAttribute[0]);
            AddonLoader.INSTANCE.tryLoadAddons(packsFolder);
            BlueMapConfigManager configs = BlueMapConfigManager.builder().minecraftVersion(cli.minecraftVersion).configRoot(cli.configFolder).modsFolder(cli.modsFolder).packsFolder(packsFolder).usePluginConfig(false).defaultDataFolder(Path.of("data", new String[0])).defaultWebroot(Path.of("web", new String[0])).build();
            CoreConfig coreConfig = configs.getCoreConfig();
            if (coreConfig.getLog().getFile() != null) {
                ZonedDateTime zdt = ZonedDateTime.ofInstant(Instant.now(), ZoneId.systemDefault());
                Logger.global.put(Logger.file(Path.of(String.format(coreConfig.getLog().getFile(), zdt), new String[0]), coreConfig.getLog().isAppend()));
            }
            blueMap = new BlueMapService(configs);
            boolean noActions = true;
            if (cmd.hasOption("w")) {
                noActions = false;
                cli.startWebserver(blueMap, cmd.hasOption("b"));
                Thread.sleep(1000L);
            }
            if (cmd.hasOption("r") || cmd.hasOption("f") || cmd.hasOption("u") || cmd.hasOption("e")) {
                noActions = false;
                boolean watch = cmd.hasOption("u");
                TileUpdateStrategy force = TileUpdateStrategy.FORCE_NONE;
                if (cmd.hasOption("f")) {
                    force = TileUpdateStrategy.FORCE_ALL;
                } else if (cmd.hasOption("e")) {
                    force = TileUpdateStrategy.FORCE_EDGE;
                }
                boolean generateWebappFiles = cmd.hasOption("g");
                String mapsToRender = cmd.getOptionValue("m", null);
                cli.renderMaps(blueMap, watch, force, generateWebappFiles, mapsToRender);
            } else {
                if (cmd.hasOption("markers")) {
                    noActions = false;
                    String mapsToUpdate = cmd.getOptionValue("m", null);
                    cli.updateMarkers(blueMap, mapsToUpdate);
                }
                if (cmd.hasOption("g")) {
                    noActions = false;
                    blueMap.createOrUpdateWebApp(true);
                }
                if (cmd.hasOption("s")) {
                    noActions = false;
                    blueMap.createOrUpdateWebApp(false);
                }
            }
            if (noActions) {
                Logger.global.logInfo("Generated default config files for you, here: " + String.valueOf(cli.configFolder.toAbsolutePath().normalize()) + "\n");
                FileHelper.createDirectories(cli.configFolder.resolve("packs"), new FileAttribute[0]);
                BlueMapCLI.printHelp();
                System.exit(1);
            }
        }
        catch (MissingResourcesException e) {
            BlueMapConfiguration configProvider;
            Logger.global.logWarning("BlueMap is missing important resources!");
            Logger.global.logWarning("You must accept the required file download in order for BlueMap to work!");
            if (blueMap != null && (configProvider = blueMap.getConfig()) instanceof BlueMapConfigManager) {
                Logger.global.logWarning("Please check: " + String.valueOf(((BlueMapConfigManager)configProvider).getConfigManager().resolveConfigFile("core").toAbsolutePath().normalize()));
            }
            System.exit(2);
        }
        catch (ParseException e) {
            Logger.global.logError("Failed to parse provided arguments!", e);
            BlueMapCLI.printHelp();
            System.exit(1);
        }
        catch (ConfigurationException e) {
            e.printLog(Logger.global);
        }
        catch (IOException e) {
            Logger.global.logError("An IO-error occurred!", e);
            System.exit(1);
        }
        catch (InterruptedException ex) {
            System.exit(1);
        }
        catch (RuntimeException e) {
            Logger.global.logError("An unexpected error occurred!", e);
            System.exit(1);
        }
    }

    private static Options createOptions() {
        Options options = new Options();
        options.addOption("h", "help", false, "Displays this message");
        options.addOption(Option.builder("c").longOpt("config").hasArg().argName("config-folder").desc("Sets path of the folder containing the configuration-files to use (configurations will be generated here if they don't exist)").build());
        options.addOption(Option.builder("n").longOpt("mods").hasArg().argName("mods-folder").desc("Sets path of the folder containing the mods that contain extra resources for rendering.").build());
        options.addOption(Option.builder("v").longOpt("mc-version").hasArg().argName("mc-version").desc("Sets the minecraft-version, used e.g. to load resource-packs correctly. Defaults to the latest compatible version.").build());
        options.addOption(Option.builder("l").longOpt("log-file").hasArg().argName("file-name").desc("Sets a file to save the log to. If not specified, no log will be saved.").build());
        options.addOption("a", "append", false, "Causes log save file to be appended rather than replaced.");
        options.addOption("w", "webserver", false, "Starts the web-server, configured in the 'webserver.conf' file");
        options.addOption("b", "verbose", false, "Causes the web-server to log requests to the console");
        options.addOption("g", "generate-webapp", false, "Generates the files for the web-app to the folder configured in the 'webapp.conf' file");
        options.addOption("s", "generate-websettings", false, "Updates the settings.json for the web-app");
        options.addOption("r", "render", false, "Renders the maps configured in the 'render.conf' file");
        options.addOption("e", "fix-edges", false, "Forces rendering the map-edges, instead of only rendering chunks that have been modified since the last render");
        options.addOption("f", "force-render", false, "Forces rendering everything, instead of only rendering chunks that have been modified since the last render");
        options.addOption("m", "maps", true, "A comma-separated list of map-id's that should be rendered. Example: 'world,nether'");
        options.addOption(null, "markers", false, "Updates the map-markers based on the map configs");
        options.addOption("u", "watch", false, "Watches for file-changes after rendering and updates the map");
        options.addOption("V", "version", false, "Print the current BlueMap version");
        return options;
    }

    private static void printHelp() {
        HelpFormatter formatter = new HelpFormatter();
        String command = BlueMapCLI.getCliCommand();
        StringBuilder footer = new StringBuilder();
        footer.append("Examples:\n\n");
        footer.append(command).append(" -c './config/'\n");
        footer.append("Generates the default/example configurations in a folder named 'config' if they are not already present\n\n");
        footer.append(command).append(" -r\n");
        footer.append("Render the configured maps\n\n");
        footer.append(command).append(" -w\n");
        footer.append("Start only the webserver without doing anything else\n\n");
        footer.append(command).append(" -ru\n");
        footer.append("Render the configured maps and then keeps watching the world-files and updates the map once something changed.\n\n");
        formatter.printHelp(command + " [options]", "\nOptions:", BlueMapCLI.createOptions(), "\n" + String.valueOf(footer));
    }

    private static String getCliCommand() {
        Object filename = "bluemap-cli.jar";
        try {
            if (System.getenv("BLUEMAP_COMMAND") != null) {
                return System.getenv("BLUEMAP_COMMAND");
            }
            Path file = Path.of(BlueMapCLI.class.getProtectionDomain().getCodeSource().getLocation().toURI());
            if (Files.isRegularFile(file, new LinkOption[0])) {
                try {
                    filename = "." + file.getFileSystem().getSeparator() + String.valueOf(Path.of("", new String[0]).toRealPath(new LinkOption[0]).relativize(file.toRealPath(new LinkOption[0])));
                }
                catch (IllegalArgumentException ex) {
                    filename = file.toAbsolutePath().toString();
                }
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
        return "java -jar " + (String)filename;
    }

    private static void printVersion() {
        System.out.printf("%s\n%s\n", BlueMap.VERSION, BlueMap.GIT_HASH);
    }
}

