#!/usr/bin/env node
"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
process.env.DB_URL = process.env.DB_URL || 'postgres://postgres:postgres@localhost:5432/immich';
const kysely_1 = require("kysely");
const node_fs_1 = require("node:fs");
const node_path_1 = require("node:path");
const postgres_1 = __importDefault(require("postgres"));
const config_repository_1 = require("../repositories/config.repository");
const database_repository_1 = require("../repositories/database.repository");
const logging_repository_1 = require("../repositories/logging.repository");
require("../schema");
const sql_tools_1 = require("../sql-tools");
const database_1 = require("../utils/database");
const main = async () => {
    const command = process.argv[2];
    const path = process.argv[3] || 'src/Migration';
    switch (command) {
        case 'debug': {
            await debug();
            return;
        }
        case 'run': {
            await runMigrations();
            return;
        }
        case 'query': {
            const query = process.argv[3];
            await runQuery(query);
            return;
        }
        case 'create': {
            create(path, [], []);
            return;
        }
        case 'generate': {
            await generate(path);
            return;
        }
        default: {
            console.log(`Usage:
  node dist/bin/migrations.js create <name>
  node dist/bin/migrations.js generate <name>
  node dist/bin/migrations.js run
`);
        }
    }
};
const getDatabaseClient = () => {
    const configRepository = new config_repository_1.ConfigRepository();
    const { database } = configRepository.getEnv();
    return new kysely_1.Kysely((0, database_1.getKyselyConfig)(database.config));
};
const runQuery = async (query) => {
    const db = getDatabaseClient();
    await kysely_1.sql.raw(query).execute(db);
    await db.destroy();
};
const runMigrations = async () => {
    const configRepository = new config_repository_1.ConfigRepository();
    const logger = logging_repository_1.LoggingRepository.create();
    const db = getDatabaseClient();
    const databaseRepository = new database_repository_1.DatabaseRepository(db, logger, configRepository);
    await databaseRepository.runMigrations();
    await db.destroy();
};
const debug = async () => {
    const { up } = await compare();
    const upSql = '-- UP\n' + up.asSql({ comments: true }).join('\n');
    (0, node_fs_1.writeFileSync)('./migrations.sql', upSql + '\n\n');
    console.log('Wrote migrations.sql');
};
const generate = async (path) => {
    const { up, down } = await compare();
    if (up.items.length === 0) {
        console.log('No changes detected');
        return;
    }
    create(path, up.asSql(), down.asSql());
};
const create = (path, up, down) => {
    const timestamp = Date.now();
    const name = (0, node_path_1.basename)(path, (0, node_path_1.extname)(path));
    const filename = `${timestamp}-${name}.ts`;
    const folder = (0, node_path_1.dirname)(path);
    const fullPath = (0, node_path_1.join)(folder, filename);
    (0, node_fs_1.mkdirSync)(folder, { recursive: true });
    (0, node_fs_1.writeFileSync)(fullPath, asMigration({ up, down }));
    console.log(`Wrote ${fullPath}`);
};
const compare = async () => {
    const configRepository = new config_repository_1.ConfigRepository();
    const { database } = configRepository.getEnv();
    const db = (0, postgres_1.default)((0, database_1.asPostgresConnectionConfig)(database.config));
    const source = (0, sql_tools_1.schemaFromCode)({ overrides: true, namingStrategy: 'default' });
    const target = await (0, sql_tools_1.schemaFromDatabase)(db, {});
    console.log(source.warnings.join('\n'));
    const up = (0, sql_tools_1.schemaDiff)(source, target, {
        tables: { ignoreExtra: true },
        functions: { ignoreExtra: false },
        parameters: { ignoreExtra: true },
    });
    const down = (0, sql_tools_1.schemaDiff)(target, source, {
        tables: { ignoreExtra: false, ignoreMissing: true },
        functions: { ignoreExtra: false },
        extensions: { ignoreMissing: true },
        parameters: { ignoreMissing: true },
    });
    return { up, down };
};
const asMigration = ({ up, down }) => {
    const upSql = up.map((sql) => `  await sql\`${sql}\`.execute(db);`).join('\n');
    const downSql = down.map((sql) => `  await sql\`${sql}\`.execute(db);`).join('\n');
    return `import { Kysely, sql } from 'kysely';

export async function up(db: Kysely<any>): Promise<void> {
${upSql}
}

export async function down(db: Kysely<any>): Promise<void> {
${downSql}
}
`;
};
main()
    .then(() => {
    process.exit(0);
})
    .catch((error) => {
    console.error(error);
    console.log('Something went wrong');
    process.exit(1);
});
//# sourceMappingURL=migrations.js.map