"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
const nest_commander_1 = require("nest-commander");
const node_child_process_1 = require("node:child_process");
const node_path_1 = require("node:path");
const node_worker_threads_1 = require("node:worker_threads");
const app_module_1 = require("./app.module");
const enum_1 = require("./enum");
const config_repository_1 = require("./repositories/config.repository");
const immichApp = process.argv[2];
if (immichApp) {
    process.argv.splice(2, 1);
}
let apiProcess;
const onError = (name, error) => {
    console.error(`${name} worker error: ${error}, stack: ${error.stack}`);
};
const onExit = (name, exitCode) => {
    if (exitCode !== 0) {
        console.error(`${name} worker exited with code ${exitCode}`);
        if (apiProcess && name !== enum_1.ImmichWorker.Api) {
            console.error('Killing api process');
            apiProcess.kill('SIGTERM');
            apiProcess = undefined;
        }
    }
    process.exit(exitCode);
};
function bootstrapWorker(name) {
    console.log(`Starting ${name} worker`);
    const basePath = (0, node_path_1.dirname)(__filename);
    const workerFile = (0, node_path_1.join)(basePath, 'workers', `${name}.js`);
    let worker;
    if (name === enum_1.ImmichWorker.Api) {
        worker = (0, node_child_process_1.fork)(workerFile, [], {
            execArgv: process.execArgv.map((arg) => (arg.startsWith('--inspect') ? '--inspect=0.0.0.0:9231' : arg)),
        });
        apiProcess = worker;
    }
    else {
        worker = new node_worker_threads_1.Worker(workerFile);
    }
    worker.on('error', (error) => onError(name, error));
    worker.on('exit', (exitCode) => onExit(name, exitCode));
}
function bootstrap() {
    if (immichApp === 'immich-admin') {
        process.title = 'immich_admin_cli';
        process.env.IMMICH_LOG_LEVEL = enum_1.LogLevel.Warn;
        return nest_commander_1.CommandFactory.run(app_module_1.ImmichAdminModule);
    }
    if (immichApp === 'immich' || immichApp === 'microservices') {
        console.error(`Using "start.sh ${immichApp}" has been deprecated. See https://github.com/immich-app/immich/releases/tag/v1.118.0 for more information.`);
        process.exit(1);
    }
    if (immichApp) {
        console.error(`Unknown command: "${immichApp}"`);
        process.exit(1);
    }
    process.title = 'immich';
    const { workers } = new config_repository_1.ConfigRepository().getEnv();
    for (const worker of workers) {
        bootstrapWorker(worker);
    }
}
void bootstrap();
//# sourceMappingURL=main.js.map