"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AlbumService = void 0;
const common_1 = require("@nestjs/common");
const album_dto_1 = require("../dtos/album.dto");
const asset_ids_response_dto_1 = require("../dtos/asset-ids.response.dto");
const enum_1 = require("../enum");
const base_service_1 = require("./base.service");
const asset_util_1 = require("../utils/asset.util");
const preferences_1 = require("../utils/preferences");
let AlbumService = class AlbumService extends base_service_1.BaseService {
    async getStatistics(auth) {
        const [owned, shared, notShared] = await Promise.all([
            this.albumRepository.getOwned(auth.user.id),
            this.albumRepository.getShared(auth.user.id),
            this.albumRepository.getNotShared(auth.user.id),
        ]);
        return {
            owned: owned.length,
            shared: shared.length,
            notShared: notShared.length,
        };
    }
    async getAll({ user: { id: ownerId } }, { assetId, shared }) {
        await this.albumRepository.updateThumbnails();
        let albums;
        if (assetId) {
            albums = await this.albumRepository.getByAssetId(ownerId, assetId);
        }
        else if (shared === true) {
            albums = await this.albumRepository.getShared(ownerId);
        }
        else if (shared === false) {
            albums = await this.albumRepository.getNotShared(ownerId);
        }
        else {
            albums = await this.albumRepository.getOwned(ownerId);
        }
        const results = await this.albumRepository.getMetadataForIds(albums.map((album) => album.id));
        const albumMetadata = {};
        for (const metadata of results) {
            albumMetadata[metadata.albumId] = metadata;
        }
        return albums.map((album) => ({
            ...(0, album_dto_1.mapAlbumWithoutAssets)(album),
            sharedLinks: undefined,
            startDate: albumMetadata[album.id]?.startDate ?? undefined,
            endDate: albumMetadata[album.id]?.endDate ?? undefined,
            assetCount: albumMetadata[album.id]?.assetCount ?? 0,
            lastModifiedAssetTimestamp: albumMetadata[album.id]?.lastModifiedAssetTimestamp ?? undefined,
        }));
    }
    async get(auth, id, dto) {
        await this.requireAccess({ auth, permission: enum_1.Permission.AlbumRead, ids: [id] });
        await this.albumRepository.updateThumbnails();
        const withAssets = dto.withoutAssets === undefined ? true : !dto.withoutAssets;
        const album = await this.findOrFail(id, { withAssets });
        const [albumMetadataForIds] = await this.albumRepository.getMetadataForIds([album.id]);
        return {
            ...(0, album_dto_1.mapAlbum)(album, withAssets, auth),
            startDate: albumMetadataForIds?.startDate ?? undefined,
            endDate: albumMetadataForIds?.endDate ?? undefined,
            assetCount: albumMetadataForIds?.assetCount ?? 0,
            lastModifiedAssetTimestamp: albumMetadataForIds?.lastModifiedAssetTimestamp ?? undefined,
        };
    }
    async create(auth, dto) {
        const albumUsers = dto.albumUsers || [];
        for (const { userId } of albumUsers) {
            const exists = await this.userRepository.get(userId, {});
            if (!exists) {
                throw new common_1.BadRequestException('User not found');
            }
            if (userId == auth.user.id) {
                throw new common_1.BadRequestException('Cannot share album with owner');
            }
        }
        const allowedAssetIdsSet = await this.checkAccess({
            auth,
            permission: enum_1.Permission.AssetShare,
            ids: dto.assetIds || [],
        });
        const assetIds = [...allowedAssetIdsSet].map((id) => id);
        const userMetadata = await this.userRepository.getMetadata(auth.user.id);
        const album = await this.albumRepository.create({
            ownerId: auth.user.id,
            albumName: dto.albumName,
            description: dto.description,
            albumThumbnailAssetId: assetIds[0] || null,
            order: (0, preferences_1.getPreferences)(userMetadata).albums.defaultAssetOrder,
        }, assetIds, albumUsers);
        for (const { userId } of albumUsers) {
            await this.eventRepository.emit('AlbumInvite', { id: album.id, userId });
        }
        return (0, album_dto_1.mapAlbumWithAssets)(album);
    }
    async update(auth, id, dto) {
        await this.requireAccess({ auth, permission: enum_1.Permission.AlbumUpdate, ids: [id] });
        const album = await this.findOrFail(id, { withAssets: true });
        if (dto.albumThumbnailAssetId) {
            const results = await this.albumRepository.getAssetIds(id, [dto.albumThumbnailAssetId]);
            if (results.size === 0) {
                throw new common_1.BadRequestException('Invalid album thumbnail');
            }
        }
        const updatedAlbum = await this.albumRepository.update(album.id, {
            id: album.id,
            albumName: dto.albumName,
            description: dto.description,
            albumThumbnailAssetId: dto.albumThumbnailAssetId,
            isActivityEnabled: dto.isActivityEnabled,
            order: dto.order,
        });
        return (0, album_dto_1.mapAlbumWithoutAssets)({ ...updatedAlbum, assets: album.assets });
    }
    async delete(auth, id) {
        await this.requireAccess({ auth, permission: enum_1.Permission.AlbumDelete, ids: [id] });
        await this.albumRepository.delete(id);
    }
    async addAssets(auth, id, dto) {
        const album = await this.findOrFail(id, { withAssets: false });
        await this.requireAccess({ auth, permission: enum_1.Permission.AlbumAssetCreate, ids: [id] });
        const results = await (0, asset_util_1.addAssets)(auth, { access: this.accessRepository, bulk: this.albumRepository }, { parentId: id, assetIds: dto.ids });
        const { id: firstNewAssetId } = results.find(({ success }) => success) || {};
        if (firstNewAssetId) {
            await this.albumRepository.update(id, {
                id,
                updatedAt: new Date(),
                albumThumbnailAssetId: album.albumThumbnailAssetId ?? firstNewAssetId,
            });
            const allUsersExceptUs = [...album.albumUsers.map(({ user }) => user.id), album.owner.id].filter((userId) => userId !== auth.user.id);
            for (const recipientId of allUsersExceptUs) {
                await this.eventRepository.emit('AlbumUpdate', { id, recipientId });
            }
        }
        return results;
    }
    async addAssetsToAlbums(auth, dto) {
        const results = {
            success: false,
            error: asset_ids_response_dto_1.BulkIdErrorReason.DUPLICATE,
        };
        const allowedAlbumIds = await this.checkAccess({
            auth,
            permission: enum_1.Permission.AlbumAssetCreate,
            ids: dto.albumIds,
        });
        if (allowedAlbumIds.size === 0) {
            results.error = asset_ids_response_dto_1.BulkIdErrorReason.NO_PERMISSION;
            return results;
        }
        const allowedAssetIds = await this.checkAccess({ auth, permission: enum_1.Permission.AssetShare, ids: dto.assetIds });
        if (allowedAssetIds.size === 0) {
            results.error = asset_ids_response_dto_1.BulkIdErrorReason.NO_PERMISSION;
            return results;
        }
        const albumAssetValues = [];
        const events = [];
        for (const albumId of allowedAlbumIds) {
            const existingAssetIds = await this.albumRepository.getAssetIds(albumId, [...allowedAssetIds]);
            const notPresentAssetIds = [...allowedAssetIds].filter((id) => !existingAssetIds.has(id));
            if (notPresentAssetIds.length === 0) {
                continue;
            }
            const album = await this.findOrFail(albumId, { withAssets: false });
            results.error = undefined;
            results.success = true;
            for (const assetId of notPresentAssetIds) {
                albumAssetValues.push({ albumsId: albumId, assetsId: assetId });
            }
            await this.albumRepository.update(albumId, {
                id: albumId,
                updatedAt: new Date(),
                albumThumbnailAssetId: album.albumThumbnailAssetId ?? notPresentAssetIds[0],
            });
            const allUsersExceptUs = [...album.albumUsers.map(({ user }) => user.id), album.owner.id].filter((userId) => userId !== auth.user.id);
            events.push({ id: albumId, recipients: allUsersExceptUs });
        }
        await this.albumRepository.addAssetIdsToAlbums(albumAssetValues);
        for (const event of events) {
            for (const recipientId of event.recipients) {
                await this.eventRepository.emit('AlbumUpdate', { id: event.id, recipientId });
            }
        }
        return results;
    }
    async removeAssets(auth, id, dto) {
        await this.requireAccess({ auth, permission: enum_1.Permission.AlbumAssetDelete, ids: [id] });
        const album = await this.findOrFail(id, { withAssets: false });
        const results = await (0, asset_util_1.removeAssets)(auth, { access: this.accessRepository, bulk: this.albumRepository }, { parentId: id, assetIds: dto.ids, canAlwaysRemove: enum_1.Permission.AlbumDelete });
        const removedIds = results.filter(({ success }) => success).map(({ id }) => id);
        if (removedIds.length > 0 && album.albumThumbnailAssetId && removedIds.includes(album.albumThumbnailAssetId)) {
            await this.albumRepository.updateThumbnails();
        }
        return results;
    }
    async addUsers(auth, id, { albumUsers }) {
        await this.requireAccess({ auth, permission: enum_1.Permission.AlbumShare, ids: [id] });
        const album = await this.findOrFail(id, { withAssets: false });
        for (const { userId, role } of albumUsers) {
            if (album.ownerId === userId) {
                throw new common_1.BadRequestException('Cannot be shared with owner');
            }
            const exists = album.albumUsers.find(({ user: { id } }) => id === userId);
            if (exists) {
                throw new common_1.BadRequestException('User already added');
            }
            const user = await this.userRepository.get(userId, {});
            if (!user) {
                throw new common_1.BadRequestException('User not found');
            }
            await this.albumUserRepository.create({ usersId: userId, albumsId: id, role });
            await this.eventRepository.emit('AlbumInvite', { id, userId });
        }
        return this.findOrFail(id, { withAssets: true }).then(album_dto_1.mapAlbumWithoutAssets);
    }
    async removeUser(auth, id, userId) {
        if (userId === 'me') {
            userId = auth.user.id;
        }
        const album = await this.findOrFail(id, { withAssets: false });
        if (album.ownerId === userId) {
            throw new common_1.BadRequestException('Cannot remove album owner');
        }
        const exists = album.albumUsers.find(({ user: { id } }) => id === userId);
        if (!exists) {
            throw new common_1.BadRequestException('Album not shared with user');
        }
        if (auth.user.id !== userId) {
            await this.requireAccess({ auth, permission: enum_1.Permission.AlbumShare, ids: [id] });
        }
        await this.albumUserRepository.delete({ albumsId: id, usersId: userId });
    }
    async updateUser(auth, id, userId, dto) {
        await this.requireAccess({ auth, permission: enum_1.Permission.AlbumShare, ids: [id] });
        await this.albumUserRepository.update({ albumsId: id, usersId: userId }, { role: dto.role });
    }
    async findOrFail(id, options) {
        const album = await this.albumRepository.getById(id, options);
        if (!album) {
            throw new common_1.BadRequestException('Album not found');
        }
        return album;
    }
};
exports.AlbumService = AlbumService;
exports.AlbumService = AlbumService = __decorate([
    (0, common_1.Injectable)()
], AlbumService);
//# sourceMappingURL=album.service.js.map