"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.AuthService = void 0;
const common_1 = require("@nestjs/common");
const class_validator_1 = require("class-validator");
const cookie_1 = require("cookie");
const luxon_1 = require("luxon");
const node_path_1 = require("node:path");
const constants_1 = require("../constants");
const storage_core_1 = require("../cores/storage.core");
const auth_dto_1 = require("../dtos/auth.dto");
const user_dto_1 = require("../dtos/user.dto");
const enum_1 = require("../enum");
const base_service_1 = require("./base.service");
const access_1 = require("../utils/access");
const bytes_1 = require("../utils/bytes");
const mime_types_1 = require("../utils/mime-types");
let AuthService = class AuthService extends base_service_1.BaseService {
    async login(dto, details) {
        const config = await this.getConfig({ withCache: false });
        if (!config.passwordLogin.enabled) {
            throw new common_1.UnauthorizedException('Password login has been disabled');
        }
        let user = await this.userRepository.getByEmail(dto.email, { withPassword: true });
        if (user) {
            const isAuthenticated = this.validateSecret(dto.password, user.password);
            if (!isAuthenticated) {
                user = undefined;
            }
        }
        if (!user) {
            this.logger.warn(`Failed login attempt for user ${dto.email} from ip address ${details.clientIp}`);
            throw new common_1.UnauthorizedException('Incorrect email or password');
        }
        return this.createLoginResponse(user, details);
    }
    async logout(auth, authType) {
        if (auth.session) {
            await this.sessionRepository.delete(auth.session.id);
            await this.eventRepository.emit('SessionDelete', { sessionId: auth.session.id });
        }
        return {
            successful: true,
            redirectUri: await this.getLogoutEndpoint(authType),
        };
    }
    async changePassword(auth, dto) {
        const { password, newPassword } = dto;
        const user = await this.userRepository.getForChangePassword(auth.user.id);
        const valid = this.validateSecret(password, user.password);
        if (!valid) {
            throw new common_1.BadRequestException('Wrong password');
        }
        const hashedPassword = await this.cryptoRepository.hashBcrypt(newPassword, constants_1.SALT_ROUNDS);
        const updatedUser = await this.userRepository.update(user.id, { password: hashedPassword });
        return (0, user_dto_1.mapUserAdmin)(updatedUser);
    }
    async setupPinCode(auth, { pinCode }) {
        const user = await this.userRepository.getForPinCode(auth.user.id);
        if (!user) {
            throw new common_1.UnauthorizedException();
        }
        if (user.pinCode) {
            throw new common_1.BadRequestException('User already has a PIN code');
        }
        const hashed = await this.cryptoRepository.hashBcrypt(pinCode, constants_1.SALT_ROUNDS);
        await this.userRepository.update(auth.user.id, { pinCode: hashed });
    }
    async resetPinCode(auth, dto) {
        const user = await this.userRepository.getForPinCode(auth.user.id);
        this.validatePinCode(user, dto);
        await this.userRepository.update(auth.user.id, { pinCode: null });
        await this.sessionRepository.lockAll(auth.user.id);
    }
    async changePinCode(auth, dto) {
        const user = await this.userRepository.getForPinCode(auth.user.id);
        this.validatePinCode(user, dto);
        const hashed = await this.cryptoRepository.hashBcrypt(dto.newPinCode, constants_1.SALT_ROUNDS);
        await this.userRepository.update(auth.user.id, { pinCode: hashed });
    }
    validatePinCode(user, dto) {
        if (!user.pinCode) {
            throw new common_1.BadRequestException('User does not have a PIN code');
        }
        if (dto.password) {
            if (!this.validateSecret(dto.password, user.password)) {
                throw new common_1.BadRequestException('Wrong password');
            }
        }
        else if (dto.pinCode) {
            if (!this.validateSecret(dto.pinCode, user.pinCode)) {
                throw new common_1.BadRequestException('Wrong PIN code');
            }
        }
        else {
            throw new common_1.BadRequestException('Either password or pinCode is required');
        }
    }
    async adminSignUp(dto) {
        const adminUser = await this.userRepository.getAdmin();
        if (adminUser) {
            throw new common_1.BadRequestException('The server already has an admin');
        }
        const admin = await this.createUser({
            isAdmin: true,
            email: dto.email,
            name: dto.name,
            password: dto.password,
            storageLabel: 'admin',
        });
        return (0, user_dto_1.mapUserAdmin)(admin);
    }
    async authenticate({ headers, queryParams, metadata }) {
        const authDto = await this.validate({ headers, queryParams });
        const { adminRoute, sharedLinkRoute, uri } = metadata;
        const requestedPermission = metadata.permission ?? enum_1.Permission.All;
        if (!authDto.user.isAdmin && adminRoute) {
            this.logger.warn(`Denied access to admin only route: ${uri}`);
            throw new common_1.ForbiddenException('Forbidden');
        }
        if (authDto.sharedLink && !sharedLinkRoute) {
            this.logger.warn(`Denied access to non-shared route: ${uri}`);
            throw new common_1.ForbiddenException('Forbidden');
        }
        if (authDto.apiKey &&
            requestedPermission !== false &&
            !(0, access_1.isGranted)({ requested: [requestedPermission], current: authDto.apiKey.permissions })) {
            throw new common_1.ForbiddenException(`Missing required permission: ${requestedPermission}`);
        }
        return authDto;
    }
    async validate({ headers, queryParams }) {
        const shareKey = (headers[enum_1.ImmichHeader.SharedLinkKey] || queryParams[enum_1.ImmichQuery.SharedLinkKey]);
        const shareSlug = (headers[enum_1.ImmichHeader.SharedLinkSlug] || queryParams[enum_1.ImmichQuery.SharedLinkSlug]);
        const session = (headers[enum_1.ImmichHeader.UserToken] ||
            headers[enum_1.ImmichHeader.SessionToken] ||
            queryParams[enum_1.ImmichQuery.SessionKey] ||
            this.getBearerToken(headers) ||
            this.getCookieToken(headers));
        const apiKey = (headers[enum_1.ImmichHeader.ApiKey] || queryParams[enum_1.ImmichQuery.ApiKey]);
        if (shareKey) {
            return this.validateSharedLinkKey(shareKey);
        }
        if (shareSlug) {
            return this.validateSharedLinkSlug(shareSlug);
        }
        if (session) {
            return this.validateSession(session);
        }
        if (apiKey) {
            return this.validateApiKey(apiKey);
        }
        throw new common_1.UnauthorizedException('Authentication required');
    }
    getMobileRedirect(url) {
        return `${constants_1.MOBILE_REDIRECT}?${url.split('?')[1] || ''}`;
    }
    async authorize(dto) {
        const { oauth } = await this.getConfig({ withCache: false });
        if (!oauth.enabled) {
            throw new common_1.BadRequestException('OAuth is not enabled');
        }
        return await this.oauthRepository.authorize(oauth, this.resolveRedirectUri(oauth, dto.redirectUri), dto.state, dto.codeChallenge);
    }
    async callback(dto, headers, loginDetails) {
        const expectedState = dto.state ?? this.getCookieOauthState(headers);
        if (!expectedState?.length) {
            throw new common_1.BadRequestException('OAuth state is missing');
        }
        const codeVerifier = dto.codeVerifier ?? this.getCookieCodeVerifier(headers);
        if (!codeVerifier?.length) {
            throw new common_1.BadRequestException('OAuth code verifier is missing');
        }
        const { oauth } = await this.getConfig({ withCache: false });
        const url = this.resolveRedirectUri(oauth, dto.url);
        const profile = await this.oauthRepository.getProfile(oauth, url, expectedState, codeVerifier);
        const { autoRegister, defaultStorageQuota, storageLabelClaim, storageQuotaClaim, roleClaim } = oauth;
        this.logger.debug(`Logging in with OAuth: ${JSON.stringify(profile)}`);
        let user = await this.userRepository.getByOAuthId(profile.sub);
        if (!user && profile.email) {
            const emailUser = await this.userRepository.getByEmail(profile.email);
            if (emailUser) {
                if (emailUser.oauthId) {
                    throw new common_1.BadRequestException('User already exists, but is linked to another account.');
                }
                user = await this.userRepository.update(emailUser.id, { oauthId: profile.sub });
            }
        }
        if (!user) {
            if (!autoRegister) {
                this.logger.warn(`Unable to register ${profile.sub}/${profile.email || '(no email)'}. To enable set OAuth Auto Register to true in admin settings.`);
                throw new common_1.BadRequestException(`User does not exist and auto registering is disabled.`);
            }
            if (!profile.email) {
                throw new common_1.BadRequestException('OAuth profile does not have an email address');
            }
            this.logger.log(`Registering new user: ${profile.sub}/${profile.email}`);
            const storageLabel = this.getClaim(profile, {
                key: storageLabelClaim,
                default: '',
                isValid: class_validator_1.isString,
            });
            const storageQuota = this.getClaim(profile, {
                key: storageQuotaClaim,
                default: defaultStorageQuota,
                isValid: (value) => Number(value) >= 0,
            });
            const role = this.getClaim(profile, {
                key: roleClaim,
                default: 'user',
                isValid: (value) => (0, class_validator_1.isString)(value) && ['admin', 'user'].includes(value),
            });
            const userName = profile.name ?? `${profile.given_name || ''} ${profile.family_name || ''}`;
            user = await this.createUser({
                name: userName,
                email: profile.email,
                oauthId: profile.sub,
                quotaSizeInBytes: storageQuota === null ? null : storageQuota * bytes_1.HumanReadableSize.GiB,
                storageLabel: storageLabel || null,
                isAdmin: role === 'admin',
            });
        }
        if (!user.profileImagePath && profile.picture) {
            await this.syncProfilePicture(user, profile.picture);
        }
        return this.createLoginResponse(user, loginDetails);
    }
    async syncProfilePicture(user, url) {
        try {
            const oldPath = user.profileImagePath;
            const { contentType, data } = await this.oauthRepository.getProfilePicture(url);
            const extensionWithDot = mime_types_1.mimeTypes.toExtension(contentType || 'image/jpeg') ?? 'jpg';
            const profileImagePath = (0, node_path_1.join)(storage_core_1.StorageCore.getFolderLocation(enum_1.StorageFolder.Profile, user.id), `${this.cryptoRepository.randomUUID()}${extensionWithDot}`);
            this.storageCore.ensureFolders(profileImagePath);
            await this.storageRepository.createFile(profileImagePath, Buffer.from(data));
            await this.userRepository.update(user.id, { profileImagePath, profileChangedAt: new Date() });
            if (oldPath) {
                await this.jobRepository.queue({ name: enum_1.JobName.FileDelete, data: { files: [oldPath] } });
            }
        }
        catch (error) {
            this.logger.warn(`Unable to sync oauth profile picture: ${error}\n${error?.stack}`);
        }
    }
    async link(auth, dto, headers) {
        const expectedState = dto.state ?? this.getCookieOauthState(headers);
        if (!expectedState?.length) {
            throw new common_1.BadRequestException('OAuth state is missing');
        }
        const codeVerifier = dto.codeVerifier ?? this.getCookieCodeVerifier(headers);
        if (!codeVerifier?.length) {
            throw new common_1.BadRequestException('OAuth code verifier is missing');
        }
        const { oauth } = await this.getConfig({ withCache: false });
        const { sub: oauthId } = await this.oauthRepository.getProfile(oauth, dto.url, expectedState, codeVerifier);
        const duplicate = await this.userRepository.getByOAuthId(oauthId);
        if (duplicate && duplicate.id !== auth.user.id) {
            this.logger.warn(`OAuth link account failed: sub is already linked to another user (${duplicate.email}).`);
            throw new common_1.BadRequestException('This OAuth account has already been linked to another user.');
        }
        const user = await this.userRepository.update(auth.user.id, { oauthId });
        return (0, user_dto_1.mapUserAdmin)(user);
    }
    async unlink(auth) {
        const user = await this.userRepository.update(auth.user.id, { oauthId: '' });
        return (0, user_dto_1.mapUserAdmin)(user);
    }
    async getLogoutEndpoint(authType) {
        if (authType !== enum_1.AuthType.OAuth) {
            return constants_1.LOGIN_URL;
        }
        const config = await this.getConfig({ withCache: false });
        if (!config.oauth.enabled) {
            return constants_1.LOGIN_URL;
        }
        return (await this.oauthRepository.getLogoutEndpoint(config.oauth)) || constants_1.LOGIN_URL;
    }
    getBearerToken(headers) {
        const [type, token] = (headers.authorization || '').split(' ');
        if (type.toLowerCase() === 'bearer') {
            return token;
        }
        return null;
    }
    getCookieToken(headers) {
        const cookies = (0, cookie_1.parse)(headers.cookie || '');
        return cookies[enum_1.ImmichCookie.AccessToken] || null;
    }
    getCookieOauthState(headers) {
        const cookies = (0, cookie_1.parse)(headers.cookie || '');
        return cookies[enum_1.ImmichCookie.OAuthState] || null;
    }
    getCookieCodeVerifier(headers) {
        const cookies = (0, cookie_1.parse)(headers.cookie || '');
        return cookies[enum_1.ImmichCookie.OAuthCodeVerifier] || null;
    }
    async validateSharedLinkKey(key) {
        key = Array.isArray(key) ? key[0] : key;
        const bytes = Buffer.from(key, key.length === 100 ? 'hex' : 'base64url');
        const sharedLink = await this.sharedLinkRepository.getByKey(bytes);
        if (!this.isValidSharedLink(sharedLink)) {
            throw new common_1.UnauthorizedException('Invalid share key');
        }
        return { user: sharedLink.user, sharedLink };
    }
    async validateSharedLinkSlug(slug) {
        slug = Array.isArray(slug) ? slug[0] : slug;
        const sharedLink = await this.sharedLinkRepository.getBySlug(slug);
        if (!this.isValidSharedLink(sharedLink)) {
            throw new common_1.UnauthorizedException('Invalid share slug');
        }
        return { user: sharedLink.user, sharedLink };
    }
    isValidSharedLink(sharedLink) {
        return !!sharedLink?.user && (!sharedLink.expiresAt || new Date(sharedLink.expiresAt) > new Date());
    }
    async validateApiKey(key) {
        const hashedKey = this.cryptoRepository.hashSha256(key);
        const apiKey = await this.apiKeyRepository.getKey(hashedKey);
        if (apiKey?.user) {
            return {
                user: apiKey.user,
                apiKey,
            };
        }
        throw new common_1.UnauthorizedException('Invalid API key');
    }
    validateSecret(inputSecret, existingHash) {
        if (!existingHash) {
            return false;
        }
        return this.cryptoRepository.compareBcrypt(inputSecret, existingHash);
    }
    async validateSession(tokenValue) {
        const hashedToken = this.cryptoRepository.hashSha256(tokenValue);
        const session = await this.sessionRepository.getByToken(hashedToken);
        if (session?.user) {
            const now = luxon_1.DateTime.now();
            const updatedAt = luxon_1.DateTime.fromJSDate(session.updatedAt);
            const diff = now.diff(updatedAt, ['hours']);
            if (diff.hours > 1) {
                await this.sessionRepository.update(session.id, { id: session.id, updatedAt: new Date() });
            }
            let hasElevatedPermission = false;
            if (session.pinExpiresAt) {
                const pinExpiresAt = luxon_1.DateTime.fromJSDate(session.pinExpiresAt);
                hasElevatedPermission = pinExpiresAt > now;
                if (hasElevatedPermission && now.plus({ minutes: 5 }) > pinExpiresAt) {
                    await this.sessionRepository.update(session.id, {
                        pinExpiresAt: luxon_1.DateTime.now().plus({ minutes: 5 }).toJSDate(),
                    });
                }
            }
            return {
                user: session.user,
                session: {
                    id: session.id,
                    hasElevatedPermission,
                },
            };
        }
        throw new common_1.UnauthorizedException('Invalid user token');
    }
    async unlockSession(auth, dto) {
        if (!auth.session) {
            throw new common_1.BadRequestException('This endpoint can only be used with a session token');
        }
        const user = await this.userRepository.getForPinCode(auth.user.id);
        this.validatePinCode(user, { pinCode: dto.pinCode });
        await this.sessionRepository.update(auth.session.id, {
            pinExpiresAt: luxon_1.DateTime.now().plus({ minutes: 15 }).toJSDate(),
        });
    }
    async lockSession(auth) {
        if (!auth.session) {
            throw new common_1.BadRequestException('This endpoint can only be used with a session token');
        }
        await this.sessionRepository.update(auth.session.id, { pinExpiresAt: null });
    }
    async createLoginResponse(user, loginDetails) {
        const token = this.cryptoRepository.randomBytesAsText(32);
        const tokenHashed = this.cryptoRepository.hashSha256(token);
        await this.sessionRepository.create({
            token: tokenHashed,
            deviceOS: loginDetails.deviceOS,
            deviceType: loginDetails.deviceType,
            userId: user.id,
        });
        return (0, auth_dto_1.mapLoginResponse)(user, token);
    }
    getClaim(profile, options) {
        const value = profile[options.key];
        return options.isValid(value) ? value : options.default;
    }
    resolveRedirectUri({ mobileRedirectUri, mobileOverrideEnabled }, url) {
        if (mobileOverrideEnabled && mobileRedirectUri) {
            return url.replace(/app\.immich:\/+oauth-callback/, mobileRedirectUri);
        }
        return url;
    }
    async getAuthStatus(auth) {
        const user = await this.userRepository.getForPinCode(auth.user.id);
        if (!user) {
            throw new common_1.UnauthorizedException();
        }
        const session = auth.session ? await this.sessionRepository.get(auth.session.id) : undefined;
        return {
            pinCode: !!user.pinCode,
            password: !!user.password,
            isElevated: !!auth.session?.hasElevatedPermission,
            expiresAt: session?.expiresAt?.toISOString(),
            pinExpiresAt: session?.pinExpiresAt?.toISOString(),
        };
    }
};
exports.AuthService = AuthService;
exports.AuthService = AuthService = __decorate([
    (0, common_1.Injectable)()
], AuthService);
//# sourceMappingURL=auth.service.js.map