"use strict";
var __decorate = (this && this.__decorate) || function (decorators, target, key, desc) {
    var c = arguments.length, r = c < 3 ? target : desc === null ? desc = Object.getOwnPropertyDescriptor(target, key) : desc, d;
    if (typeof Reflect === "object" && typeof Reflect.decorate === "function") r = Reflect.decorate(decorators, target, key, desc);
    else for (var i = decorators.length - 1; i >= 0; i--) if (d = decorators[i]) r = (c < 3 ? d(r) : c > 3 ? d(target, key, r) : d(target, key)) || r;
    return c > 3 && r && Object.defineProperty(target, key, r), r;
};
var __metadata = (this && this.__metadata) || function (k, v) {
    if (typeof Reflect === "object" && typeof Reflect.metadata === "function") return Reflect.metadata(k, v);
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.JobService = void 0;
const common_1 = require("@nestjs/common");
const lodash_1 = require("lodash");
const decorators_1 = require("../decorators");
const asset_response_dto_1 = require("../dtos/asset-response.dto");
const job_dto_1 = require("../dtos/job.dto");
const enum_1 = require("../enum");
const base_service_1 = require("./base.service");
const bytes_1 = require("../utils/bytes");
const misc_1 = require("../utils/misc");
const asJobItem = (dto) => {
    switch (dto.name) {
        case enum_1.ManualJobName.TagCleanup: {
            return { name: enum_1.JobName.TagCleanup };
        }
        case enum_1.ManualJobName.PersonCleanup: {
            return { name: enum_1.JobName.PersonCleanup };
        }
        case enum_1.ManualJobName.UserCleanup: {
            return { name: enum_1.JobName.UserDeleteCheck };
        }
        case enum_1.ManualJobName.MemoryCleanup: {
            return { name: enum_1.JobName.MemoryCleanup };
        }
        case enum_1.ManualJobName.MemoryCreate: {
            return { name: enum_1.JobName.MemoryGenerate };
        }
        case enum_1.ManualJobName.BackupDatabase: {
            return { name: enum_1.JobName.DatabaseBackup };
        }
        default: {
            throw new common_1.BadRequestException('Invalid job name');
        }
    }
};
const asNightlyTasksCron = (config) => {
    const [hours, minutes] = config.nightlyTasks.startTime.split(':').map(Number);
    return `${minutes} ${hours} * * *`;
};
let JobService = class JobService extends base_service_1.BaseService {
    services = [];
    nightlyJobsLock = false;
    async onConfigInit({ newConfig: config }) {
        if (this.worker === enum_1.ImmichWorker.Microservices) {
            this.updateQueueConcurrency(config);
            return;
        }
        this.nightlyJobsLock = await this.databaseRepository.tryLock(enum_1.DatabaseLock.NightlyJobs);
        if (this.nightlyJobsLock) {
            const cronExpression = asNightlyTasksCron(config);
            this.logger.debug(`Scheduling nightly jobs for ${cronExpression}`);
            this.cronRepository.create({
                name: enum_1.CronJob.NightlyJobs,
                expression: cronExpression,
                start: true,
                onTick: () => (0, misc_1.handlePromiseError)(this.handleNightlyJobs(), this.logger),
            });
        }
    }
    onConfigUpdate({ newConfig: config }) {
        if (this.worker === enum_1.ImmichWorker.Microservices) {
            this.updateQueueConcurrency(config);
            return;
        }
        if (this.nightlyJobsLock) {
            const cronExpression = asNightlyTasksCron(config);
            this.logger.debug(`Scheduling nightly jobs for ${cronExpression}`);
            this.cronRepository.update({ name: enum_1.CronJob.NightlyJobs, expression: cronExpression, start: true });
        }
    }
    onBootstrap() {
        this.jobRepository.setup(this.services);
        if (this.worker === enum_1.ImmichWorker.Microservices) {
            this.jobRepository.startWorkers();
        }
    }
    updateQueueConcurrency(config) {
        this.logger.debug(`Updating queue concurrency settings`);
        for (const queueName of Object.values(enum_1.QueueName)) {
            let concurrency = 1;
            if (this.isConcurrentQueue(queueName)) {
                concurrency = config.job[queueName].concurrency;
            }
            this.logger.debug(`Setting ${queueName} concurrency to ${concurrency}`);
            this.jobRepository.setConcurrency(queueName, concurrency);
        }
    }
    setServices(services) {
        this.services = services;
    }
    async create(dto) {
        await this.jobRepository.queue(asJobItem(dto));
    }
    async handleCommand(queueName, dto) {
        this.logger.debug(`Handling command: queue=${queueName},command=${dto.command},force=${dto.force}`);
        switch (dto.command) {
            case enum_1.JobCommand.Start: {
                await this.start(queueName, dto);
                break;
            }
            case enum_1.JobCommand.Pause: {
                await this.jobRepository.pause(queueName);
                break;
            }
            case enum_1.JobCommand.Resume: {
                await this.jobRepository.resume(queueName);
                break;
            }
            case enum_1.JobCommand.Empty: {
                await this.jobRepository.empty(queueName);
                break;
            }
            case enum_1.JobCommand.ClearFailed: {
                const failedJobs = await this.jobRepository.clear(queueName, enum_1.QueueCleanType.Failed);
                this.logger.debug(`Cleared failed jobs: ${failedJobs}`);
                break;
            }
        }
        return this.getJobStatus(queueName);
    }
    async getJobStatus(queueName) {
        const [jobCounts, queueStatus] = await Promise.all([
            this.jobRepository.getJobCounts(queueName),
            this.jobRepository.getQueueStatus(queueName),
        ]);
        return { jobCounts, queueStatus };
    }
    async getAllJobsStatus() {
        const response = new job_dto_1.AllJobStatusResponseDto();
        for (const queueName of Object.values(enum_1.QueueName)) {
            response[queueName] = await this.getJobStatus(queueName);
        }
        return response;
    }
    async start(name, { force }) {
        const { isActive } = await this.jobRepository.getQueueStatus(name);
        if (isActive) {
            throw new common_1.BadRequestException(`Job is already running`);
        }
        this.telemetryRepository.jobs.addToCounter(`immich.queues.${(0, lodash_1.snakeCase)(name)}.started`, 1);
        switch (name) {
            case enum_1.QueueName.VideoConversion: {
                return this.jobRepository.queue({ name: enum_1.JobName.AssetEncodeVideoQueueAll, data: { force } });
            }
            case enum_1.QueueName.StorageTemplateMigration: {
                return this.jobRepository.queue({ name: enum_1.JobName.StorageTemplateMigration });
            }
            case enum_1.QueueName.Migration: {
                return this.jobRepository.queue({ name: enum_1.JobName.FileMigrationQueueAll });
            }
            case enum_1.QueueName.SmartSearch: {
                return this.jobRepository.queue({ name: enum_1.JobName.SmartSearchQueueAll, data: { force } });
            }
            case enum_1.QueueName.DuplicateDetection: {
                return this.jobRepository.queue({ name: enum_1.JobName.AssetDetectDuplicatesQueueAll, data: { force } });
            }
            case enum_1.QueueName.MetadataExtraction: {
                return this.jobRepository.queue({ name: enum_1.JobName.AssetExtractMetadataQueueAll, data: { force } });
            }
            case enum_1.QueueName.Sidecar: {
                return this.jobRepository.queue({ name: enum_1.JobName.SidecarQueueAll, data: { force } });
            }
            case enum_1.QueueName.ThumbnailGeneration: {
                return this.jobRepository.queue({ name: enum_1.JobName.AssetGenerateThumbnailsQueueAll, data: { force } });
            }
            case enum_1.QueueName.FaceDetection: {
                return this.jobRepository.queue({ name: enum_1.JobName.AssetDetectFacesQueueAll, data: { force } });
            }
            case enum_1.QueueName.FacialRecognition: {
                return this.jobRepository.queue({ name: enum_1.JobName.FacialRecognitionQueueAll, data: { force } });
            }
            case enum_1.QueueName.Library: {
                return this.jobRepository.queue({ name: enum_1.JobName.LibraryScanQueueAll, data: { force } });
            }
            case enum_1.QueueName.BackupDatabase: {
                return this.jobRepository.queue({ name: enum_1.JobName.DatabaseBackup, data: { force } });
            }
            default: {
                throw new common_1.BadRequestException(`Invalid job name: ${name}`);
            }
        }
    }
    async onJobStart(...[queueName, job]) {
        const queueMetric = `immich.queues.${(0, lodash_1.snakeCase)(queueName)}.active`;
        this.telemetryRepository.jobs.addToGauge(queueMetric, 1);
        try {
            const status = await this.jobRepository.run(job);
            const jobMetric = `immich.jobs.${(0, lodash_1.snakeCase)(job.name)}.${status}`;
            this.telemetryRepository.jobs.addToCounter(jobMetric, 1);
            if (status === enum_1.JobStatus.Success || status == enum_1.JobStatus.Skipped) {
                await this.onDone(job);
            }
        }
        catch (error) {
            await this.eventRepository.emit('JobFailed', { job, error });
        }
        finally {
            this.telemetryRepository.jobs.addToGauge(queueMetric, -1);
        }
    }
    isConcurrentQueue(name) {
        return ![
            enum_1.QueueName.FacialRecognition,
            enum_1.QueueName.StorageTemplateMigration,
            enum_1.QueueName.DuplicateDetection,
            enum_1.QueueName.BackupDatabase,
        ].includes(name);
    }
    async handleNightlyJobs() {
        const config = await this.getConfig({ withCache: false });
        const jobs = [];
        if (config.nightlyTasks.databaseCleanup) {
            jobs.push({ name: enum_1.JobName.AssetDeleteCheck }, { name: enum_1.JobName.UserDeleteCheck }, { name: enum_1.JobName.PersonCleanup }, { name: enum_1.JobName.MemoryCleanup }, { name: enum_1.JobName.SessionCleanup }, { name: enum_1.JobName.AuditTableCleanup }, { name: enum_1.JobName.AuditLogCleanup });
        }
        if (config.nightlyTasks.generateMemories) {
            jobs.push({ name: enum_1.JobName.MemoryGenerate });
        }
        if (config.nightlyTasks.syncQuotaUsage) {
            jobs.push({ name: enum_1.JobName.UserSyncUsage });
        }
        if (config.nightlyTasks.missingThumbnails) {
            jobs.push({ name: enum_1.JobName.AssetGenerateThumbnailsQueueAll, data: { force: false } });
        }
        if (config.nightlyTasks.clusterNewFaces) {
            jobs.push({ name: enum_1.JobName.FacialRecognitionQueueAll, data: { force: false, nightly: true } });
        }
        await this.jobRepository.queueAll(jobs);
    }
    async onDone(item) {
        switch (item.name) {
            case enum_1.JobName.SidecarCheck: {
                await this.jobRepository.queue({ name: enum_1.JobName.AssetExtractMetadata, data: item.data });
                break;
            }
            case enum_1.JobName.SidecarWrite: {
                await this.jobRepository.queue({
                    name: enum_1.JobName.AssetExtractMetadata,
                    data: { id: item.data.id, source: 'sidecar-write' },
                });
                break;
            }
            case enum_1.JobName.StorageTemplateMigrationSingle: {
                if (item.data.source === 'upload' || item.data.source === 'copy') {
                    await this.jobRepository.queue({ name: enum_1.JobName.AssetGenerateThumbnails, data: item.data });
                }
                break;
            }
            case enum_1.JobName.PersonGenerateThumbnail: {
                const { id } = item.data;
                const person = await this.personRepository.getById(id);
                if (person) {
                    this.eventRepository.clientSend('on_person_thumbnail', person.ownerId, person.id);
                }
                break;
            }
            case enum_1.JobName.AssetGenerateThumbnails: {
                if (!item.data.notify && item.data.source !== 'upload') {
                    break;
                }
                const [asset] = await this.assetRepository.getByIdsWithAllRelationsButStacks([item.data.id]);
                if (!asset) {
                    this.logger.warn(`Could not find asset ${item.data.id} after generating thumbnails`);
                    break;
                }
                const jobs = [
                    { name: enum_1.JobName.SmartSearch, data: item.data },
                    { name: enum_1.JobName.AssetDetectFaces, data: item.data },
                ];
                if (asset.type === enum_1.AssetType.Video) {
                    jobs.push({ name: enum_1.JobName.AssetEncodeVideo, data: item.data });
                }
                await this.jobRepository.queueAll(jobs);
                if (asset.visibility === enum_1.AssetVisibility.Timeline || asset.visibility === enum_1.AssetVisibility.Archive) {
                    this.eventRepository.clientSend('on_upload_success', asset.ownerId, (0, asset_response_dto_1.mapAsset)(asset));
                    if (asset.exifInfo) {
                        const exif = asset.exifInfo;
                        this.eventRepository.clientSend('AssetUploadReadyV1', asset.ownerId, {
                            asset: {
                                id: asset.id,
                                ownerId: asset.ownerId,
                                originalFileName: asset.originalFileName,
                                thumbhash: asset.thumbhash ? (0, bytes_1.hexOrBufferToBase64)(asset.thumbhash) : null,
                                checksum: (0, bytes_1.hexOrBufferToBase64)(asset.checksum),
                                fileCreatedAt: asset.fileCreatedAt,
                                fileModifiedAt: asset.fileModifiedAt,
                                localDateTime: asset.localDateTime,
                                duration: asset.duration,
                                type: asset.type,
                                deletedAt: asset.deletedAt,
                                isFavorite: asset.isFavorite,
                                visibility: asset.visibility,
                                livePhotoVideoId: asset.livePhotoVideoId,
                                stackId: asset.stackId,
                                libraryId: asset.libraryId,
                            },
                            exif: {
                                assetId: exif.assetId,
                                description: exif.description,
                                exifImageWidth: exif.exifImageWidth,
                                exifImageHeight: exif.exifImageHeight,
                                fileSizeInByte: exif.fileSizeInByte,
                                orientation: exif.orientation,
                                dateTimeOriginal: exif.dateTimeOriginal,
                                modifyDate: exif.modifyDate,
                                timeZone: exif.timeZone,
                                latitude: exif.latitude,
                                longitude: exif.longitude,
                                projectionType: exif.projectionType,
                                city: exif.city,
                                state: exif.state,
                                country: exif.country,
                                make: exif.make,
                                model: exif.model,
                                lensModel: exif.lensModel,
                                fNumber: exif.fNumber,
                                focalLength: exif.focalLength,
                                iso: exif.iso,
                                exposureTime: exif.exposureTime,
                                profileDescription: exif.profileDescription,
                                rating: exif.rating,
                                fps: exif.fps,
                            },
                        });
                    }
                }
                break;
            }
            case enum_1.JobName.SmartSearch: {
                if (item.data.source === 'upload') {
                    await this.jobRepository.queue({ name: enum_1.JobName.AssetDetectDuplicates, data: item.data });
                }
                break;
            }
            case enum_1.JobName.UserDelete: {
                this.eventRepository.clientBroadcast('on_user_delete', item.data.id);
                break;
            }
        }
    }
};
exports.JobService = JobService;
__decorate([
    (0, decorators_1.OnEvent)({ name: 'ConfigInit' }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", Promise)
], JobService.prototype, "onConfigInit", null);
__decorate([
    (0, decorators_1.OnEvent)({ name: 'ConfigUpdate', server: true }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [Object]),
    __metadata("design:returntype", void 0)
], JobService.prototype, "onConfigUpdate", null);
__decorate([
    (0, decorators_1.OnEvent)({ name: 'AppBootstrap', priority: enum_1.BootstrapEventPriority.JobService }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", []),
    __metadata("design:returntype", void 0)
], JobService.prototype, "onBootstrap", null);
__decorate([
    (0, decorators_1.OnEvent)({ name: 'JobStart' }),
    __metadata("design:type", Function),
    __metadata("design:paramtypes", [String]),
    __metadata("design:returntype", Promise)
], JobService.prototype, "onJobStart", null);
exports.JobService = JobService = __decorate([
    (0, common_1.Injectable)()
], JobService);
//# sourceMappingURL=job.service.js.map