"use strict";
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
const core_1 = require("@nestjs/core");
const body_parser_1 = require("body-parser");
const compression_1 = __importDefault(require("compression"));
const cookie_parser_1 = __importDefault(require("cookie-parser"));
const node_fs_1 = require("node:fs");
const sirv_1 = __importDefault(require("sirv"));
const app_module_1 = require("../app.module");
const constants_1 = require("../constants");
const websocket_adapter_1 = require("../middleware/websocket.adapter");
const config_repository_1 = require("../repositories/config.repository");
const logging_repository_1 = require("../repositories/logging.repository");
const telemetry_repository_1 = require("../repositories/telemetry.repository");
const api_service_1 = require("../services/api.service");
const misc_1 = require("../utils/misc");
async function bootstrap() {
    process.title = 'immich-api';
    const { telemetry, network } = new config_repository_1.ConfigRepository().getEnv();
    if (telemetry.metrics.size > 0) {
        (0, telemetry_repository_1.bootstrapTelemetry)(telemetry.apiPort);
    }
    const app = await core_1.NestFactory.create(app_module_1.ApiModule, { bufferLogs: true });
    const logger = await app.resolve(logging_repository_1.LoggingRepository);
    const configRepository = app.get(config_repository_1.ConfigRepository);
    const { environment, host, port, resourcePaths } = configRepository.getEnv();
    logger.setContext('Bootstrap');
    app.useLogger(logger);
    app.set('trust proxy', ['loopback', ...network.trustedProxies]);
    app.set('etag', 'strong');
    app.use((0, cookie_parser_1.default)());
    app.use((0, body_parser_1.json)({ limit: '10mb' }));
    if (configRepository.isDev()) {
        app.enableCors();
    }
    app.useWebSocketAdapter(new websocket_adapter_1.WebSocketAdapter(app));
    (0, misc_1.useSwagger)(app, { write: configRepository.isDev() });
    app.setGlobalPrefix('api', { exclude: constants_1.excludePaths });
    if ((0, node_fs_1.existsSync)(resourcePaths.web.root)) {
        app.use((0, sirv_1.default)(resourcePaths.web.root, {
            etag: true,
            gzip: true,
            brotli: true,
            extensions: [],
            setHeaders: (res, pathname) => {
                if (pathname.startsWith(`/_app/immutable`) && res.statusCode === 200) {
                    res.setHeader('cache-control', 'public,max-age=31536000,immutable');
                }
            },
        }));
    }
    app.use(app.get(api_service_1.ApiService).ssr(constants_1.excludePaths));
    app.use((0, compression_1.default)());
    const server = await (host ? app.listen(port, host) : app.listen(port));
    server.requestTimeout = 24 * 60 * 60 * 1000;
    logger.log(`Immich Server is listening on ${await app.getUrl()} [v${constants_1.serverVersion}] [${environment}] `);
}
bootstrap().catch((error) => {
    if (!(0, misc_1.isStartUpError)(error)) {
        console.error(error);
    }
    process.exit(1);
});
//# sourceMappingURL=api.js.map