import { Feature, Point, LineString, Polygon, MultiPoint, MultiLineString, MultiPolygon, FeatureCollection, Geometry } from 'geojson';

/**
 * Unwrap a coordinate from a Point Feature, Geometry or a single coordinate.
 *
 * @function
 * @param {Array<number>|Geometry<Point>|Feature<Point>} coord GeoJSON Point or an Array of numbers
 * @returns {Array<number>} coordinates
 * @example
 * var pt = turf.point([10, 10]);
 *
 * var coord = turf.getCoord(pt);
 * //= [10, 10]
 */
declare function getCoord(coord: Feature<Point> | Point | number[]): number[];
/**
 * Unwrap coordinates from a Feature, Geometry Object or an Array
 *
 * @function
 * @param {Array<any>|Geometry|Feature} coords Feature, Geometry Object or an Array
 * @returns {Array<any>} coordinates
 * @example
 * var poly = turf.polygon([[[119.32, -8.7], [119.55, -8.69], [119.51, -8.54], [119.32, -8.7]]]);
 *
 * var coords = turf.getCoords(poly);
 * //= [[[119.32, -8.7], [119.55, -8.69], [119.51, -8.54], [119.32, -8.7]]]
 */
declare function getCoords<G extends Point | LineString | Polygon | MultiPoint | MultiLineString | MultiPolygon>(coords: any[] | Feature<G> | G): any[];
/**
 * Checks if coordinates contains a number
 *
 * @function
 * @param {Array<any>} coordinates GeoJSON Coordinates
 * @returns {boolean} true if Array contains a number
 */
declare function containsNumber(coordinates: any[]): boolean;
/**
 * Enforce expectations about types of GeoJSON objects for Turf.
 *
 * @function
 * @param {GeoJSON} value any GeoJSON object
 * @param {string} type expected GeoJSON type
 * @param {string} name name of calling function
 * @throws {Error} if value is not the expected type.
 */
declare function geojsonType(value: any, type: string, name: string): void;
/**
 * Enforce expectations about types of {@link Feature} inputs for Turf.
 * Internally this uses {@link geojsonType} to judge geometry types.
 *
 * @function
 * @param {Feature} feature a feature with an expected geometry type
 * @param {string} type expected GeoJSON type
 * @param {string} name name of calling function
 * @throws {Error} error if value is not the expected type.
 */
declare function featureOf(feature: Feature<any>, type: string, name: string): void;
/**
 * Enforce expectations about types of {@link FeatureCollection} inputs for Turf.
 * Internally this uses {@link geojsonType} to judge geometry types.
 *
 * @function
 * @param {FeatureCollection} featureCollection a FeatureCollection for which features will be judged
 * @param {string} type expected GeoJSON type
 * @param {string} name name of calling function
 * @throws {Error} if value is not the expected type.
 */
declare function collectionOf(featureCollection: FeatureCollection<any>, type: string, name: string): void;
/**
 * Get Geometry from Feature or Geometry Object
 *
 * @param {Feature|Geometry} geojson GeoJSON Feature or Geometry Object
 * @returns {Geometry|null} GeoJSON Geometry Object
 * @throws {Error} if geojson is not a Feature or Geometry Object
 * @example
 * var point = {
 *   "type": "Feature",
 *   "properties": {},
 *   "geometry": {
 *     "type": "Point",
 *     "coordinates": [110, 40]
 *   }
 * }
 * var geom = turf.getGeom(point)
 * //={"type": "Point", "coordinates": [110, 40]}
 */
declare function getGeom<G extends Geometry>(geojson: Feature<G> | G): G;
/**
 * Get GeoJSON object's type, Geometry type is prioritize.
 *
 * @param {GeoJSON} geojson GeoJSON object
 * @param {string} [name="geojson"] name of the variable to display in error message (unused)
 * @returns {string} GeoJSON type
 * @example
 * var point = {
 *   "type": "Feature",
 *   "properties": {},
 *   "geometry": {
 *     "type": "Point",
 *     "coordinates": [110, 40]
 *   }
 * }
 * var geom = turf.getType(point)
 * //="Point"
 */
declare function getType(geojson: Feature<any> | FeatureCollection<any> | Geometry, _name?: string): string;

export { collectionOf, containsNumber, featureOf, geojsonType, getCoord, getCoords, getGeom, getType };
