"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CommandFactory = void 0;
const core_1 = require("@nestjs/core");
const cosmiconfig_1 = require("cosmiconfig");
const command_root_module_1 = require("./command-root.module");
const command_runner_module_1 = require("./command-runner.module");
const command_runner_service_1 = require("./command-runner.service");
const completion_factory_1 = require("./completion.factory");
class CommandFactory {
    static async run(rootModule, optionsOrLogger) {
        const app = await this.createWithoutRunning(rootModule, optionsOrLogger);
        try {
            await this.runApplication(app);
        }
        finally {
            await app.close();
        }
    }
    static async runWithoutClosing(rootModule, optionsOrLogger) {
        const app = await this.createWithoutRunning(rootModule, optionsOrLogger);
        await this.runApplication(app);
        return app;
    }
    static async createWithoutRunning(rootModule, optionsOrLogger = false) {
        const options = this.getOptions(optionsOrLogger);
        const imports = [rootModule];
        let pluginsAvailable = false;
        if (options.usePlugins) {
            pluginsAvailable = await this.registerPlugins(options.cliName, imports);
        }
        const commandRunnerModule = this.createCommandModule(imports, Object.assign(Object.assign({}, options), { pluginsAvailable }));
        const app = await core_1.NestFactory.createApplicationContext(commandRunnerModule, options);
        if (options.completion) {
            completion_factory_1.CompletionFactory.registerCompletionCommand(app, options.completion);
        }
        return app;
    }
    static async runApplication(app) {
        const runner = app.get(command_runner_service_1.CommandRunnerService);
        await runner.run();
        return app;
    }
    static createCommandModule(imports, options) {
        return command_runner_module_1.CommandRunnerModule.forModule({ module: command_root_module_1.CommandRootModule, imports }, options);
    }
    static getOptions(optionsOrLogger) {
        let options = {};
        const isOptionsIsFactoryOptionsObject = this.isFactoryOptionsObject(optionsOrLogger);
        options = isOptionsIsFactoryOptionsObject ? optionsOrLogger : options;
        options.logger =
            (isOptionsIsFactoryOptionsObject
                ? optionsOrLogger.logger
                : optionsOrLogger) || false;
        options.errorHandler = options.errorHandler || undefined;
        options.usePlugins = options.usePlugins || false;
        options.cliName = options.cliName || 'nest-commander';
        options.serviceErrorHandler = options.serviceErrorHandler || undefined;
        options.enablePositionalOptions = options.enablePositionalOptions || false;
        options.enablePassThroughOptions =
            options.enablePassThroughOptions || false;
        options.outputConfiguration = options.outputConfiguration || undefined;
        options.completion = options.completion || false;
        options.helpConfiguration = options.helpConfiguration || undefined;
        return options;
    }
    static isFactoryOptionsObject(loggerOrOptions) {
        return !(Array.isArray(loggerOrOptions) ||
            loggerOrOptions === false ||
            !!loggerOrOptions.log);
    }
    static async registerPlugins(cliName, imports) {
        var _a;
        const pluginExplorer = (0, cosmiconfig_1.cosmiconfig)(cliName, {
            searchPlaces: [
                `.${cliName}rc`,
                `.${cliName}rc.json`,
                `.${cliName}rc.yaml`,
                `.${cliName}rc.yml`,
                `${cliName}.json`,
                `${cliName}.yaml`,
                `${cliName}.yml`,
            ],
        });
        const pluginConfig = await pluginExplorer.search();
        if (!pluginConfig) {
            return false;
        }
        for (const pluginPath of (_a = pluginConfig === null || pluginConfig === void 0 ? void 0 : pluginConfig.config.plugins) !== null && _a !== void 0 ? _a : []) {
            const plugin = await import(require.resolve(pluginPath, { paths: [process.cwd()] }));
            imports === null || imports === void 0 ? void 0 : imports.push(plugin.default);
        }
        return true;
    }
}
exports.CommandFactory = CommandFactory;
//# sourceMappingURL=command.factory.js.map