"use strict";
var __defProp = Object.defineProperty;
var __getOwnPropDesc = Object.getOwnPropertyDescriptor;
var __getOwnPropNames = Object.getOwnPropertyNames;
var __hasOwnProp = Object.prototype.hasOwnProperty;
var __name = (target, value) => __defProp(target, "name", { value, configurable: true });
var __export = (target, all) => {
  for (var name in all)
    __defProp(target, name, { get: all[name], enumerable: true });
};
var __copyProps = (to, from, except, desc) => {
  if (from && typeof from === "object" || typeof from === "function") {
    for (let key of __getOwnPropNames(from))
      if (!__hasOwnProp.call(to, key) && key !== except)
        __defProp(to, key, { get: () => from[key], enumerable: !(desc = __getOwnPropDesc(from, key)) || desc.enumerable });
  }
  return to;
};
var __toCommonJS = (mod) => __copyProps(__defProp({}, "__esModule", { value: true }), mod);
var adapter_exports = {};
__export(adapter_exports, {
  NVM3Adapter: () => NVM3Adapter
});
module.exports = __toCommonJS(adapter_exports);
var import_core = require("@zwave-js/core");
var import_shared = require("@zwave-js/shared");
var import_helpers = require("alcalzone-shared/helpers");
var import_consts = require("../../consts.js");
var import_files = require("./files/index.js");
const DEFAULT_FILE_VERSION = "7.0.0";
class NVM3Adapter {
  static {
    __name(this, "NVM3Adapter");
  }
  constructor(nvm) {
    this._nvm = nvm;
  }
  _nvm;
  _initialized = false;
  _protocolInfo;
  _applicationInfo;
  /** A list of pending changes that haven't been written to the NVM yet. `null` indicates a deleted entry. */
  _pendingChanges = /* @__PURE__ */ new Map();
  getFileVersion(fileId) {
    if (fileId === import_files.ProtocolVersionFileID || fileId === import_files.ApplicationVersionFileID || fileId === import_files.ApplicationVersionFile800ID) {
      return DEFAULT_FILE_VERSION;
    }
    const section = (0, import_files.getNVMSectionByFileID)(fileId);
    if (section === "application") {
      return this._applicationInfo?.version ?? DEFAULT_FILE_VERSION;
    } else if (section === "protocol") {
      return this._protocolInfo?.version ?? DEFAULT_FILE_VERSION;
    }
    return DEFAULT_FILE_VERSION;
  }
  async init() {
    if (!this._protocolInfo) {
      const protocolVersionFile = await this._getFile(import_files.ProtocolVersionFileID, true);
      if (protocolVersionFile) {
        const version = `${protocolVersionFile.major}.${protocolVersionFile.minor}.${protocolVersionFile.patch}`;
        this._protocolInfo = {
          version,
          format: protocolVersionFile.format
        };
      }
    }
    if (!this._applicationInfo) {
      const applicationVersionFile700 = await this._getFile(import_files.ApplicationVersionFileID, true);
      const applicationVersionFile800 = await this._getFile(import_files.ApplicationVersionFile800ID, true);
      const applicationVersionFile = applicationVersionFile700 ?? applicationVersionFile800;
      if (applicationVersionFile) {
        const version = `${applicationVersionFile.major}.${applicationVersionFile.minor}.${applicationVersionFile.patch}`;
        this._applicationInfo = {
          version,
          format: applicationVersionFile.format
        };
      }
    }
    this._initialized = true;
  }
  /** Adds a complete file to the list of pending changes */
  setFile(file) {
    const { key, data } = file.serialize();
    this._pendingChanges.set(key, data);
  }
  async hasFile(fileId) {
    if (!this._initialized)
      await this.init();
    if (this._pendingChanges.has(fileId)) {
      return this._pendingChanges.get(fileId) !== null;
    } else {
      return this._nvm.has(fileId);
    }
  }
  async _getFile(fileId, skipInit = false) {
    if (!skipInit && !this._initialized)
      await this.init();
    let data;
    if (this._pendingChanges.has(fileId)) {
      data = this._pendingChanges.get(fileId);
    } else {
      data = await this._nvm.get(fileId);
    }
    if (!data)
      return;
    const fileVersion = this.getFileVersion(fileId);
    return import_files.NVMFile.from(fileId, data, fileVersion);
  }
  async _expectFile(fileId, skipInit = false) {
    const file = await this._getFile(fileId, skipInit);
    if (!file) {
      throw new import_core.ZWaveError(`NVM file ${(0, import_shared.num2hex)(fileId)} not found`, import_core.ZWaveErrorCodes.NVM_ObjectNotFound);
    }
    return file;
  }
  getFile(fileId, required) {
    if (required) {
      return this._expectFile(fileId);
    } else {
      return this._getFile(fileId);
    }
  }
  get(property, required) {
    if (property.domain === "controller") {
      return this.getControllerNVMProperty(property, !!required);
    } else if (property.domain === "lrnode") {
      return this.getLRNodeNVMProperty(property, !!required);
    } else {
      return this.getNodeNVMProperty(property, !!required);
    }
  }
  async getControllerNVMProperty(property, required) {
    const getFile = /* @__PURE__ */ __name((fileId) => {
      if (required) {
        return this._expectFile(fileId);
      } else {
        return this._getFile(fileId);
      }
    }, "getFile");
    switch (property.type) {
      case "protocolVersion": {
        const file = await getFile(import_files.ProtocolVersionFileID);
        if (!file)
          return;
        return `${file.major}.${file.minor}.${file.patch}`;
      }
      case "protocolFileFormat": {
        const file = await getFile(import_files.ProtocolVersionFileID);
        return file?.format;
      }
      case "applicationVersion":
      case "applicationFileFormat": {
        const file700 = await this._getFile(import_files.ApplicationVersionFileID);
        const file800 = await this._getFile(import_files.ApplicationVersionFile800ID);
        const file = file700 ?? file800;
        if (!file) {
          if (required) {
            throw new import_core.ZWaveError("ApplicationVersionFile not found!", import_core.ZWaveErrorCodes.NVM_ObjectNotFound);
          } else {
            return;
          }
        }
        if (property.type === "applicationVersion") {
          return `${file.major}.${file.minor}.${file.patch}`;
        } else if (property.type === "applicationFileFormat") {
          return file?.format;
        }
      }
      case "applicationData": {
        const file = await getFile(import_files.ApplicationDataFileID);
        return file?.applicationData;
      }
      case "applicationName": {
        const file = await getFile(import_files.ApplicationNameFileID);
        return file?.name;
      }
      case "homeId":
      case "nodeId":
      case "lastNodeId":
      case "staticControllerNodeId":
      case "sucLastIndex":
      case "controllerConfiguration":
      case "sucAwarenessPushNeeded":
      case "maxNodeId":
      case "reservedId":
      case "systemState":
      case "lastNodeIdLR":
      case "maxNodeIdLR":
      case "reservedIdLR":
      case "primaryLongRangeChannelId":
      case "dcdcConfig": {
        const file = await getFile(import_files.ControllerInfoFileID);
        return file?.[property.type];
      }
      case "includedInsecurely":
      case "includedSecurelyInsecureCCs":
      case "includedSecurelySecureCCs": {
        const file = await getFile(import_files.ApplicationCCsFileID);
        return file?.[property.type];
      }
      case "rfRegion":
      case "txPower":
      case "measured0dBm":
      case "enablePTI":
      case "maxTXPower":
      case "nodeIdType": {
        const file = await getFile(import_files.ApplicationRFConfigFileID);
        return file?.[property.type];
      }
      case "isListening":
      case "optionalFunctionality":
      case "genericDeviceClass":
      case "specificDeviceClass": {
        const file = await getFile(import_files.ApplicationTypeFileID);
        return file?.[property.type];
      }
      case "preferredRepeaters": {
        const file = await getFile(import_files.ProtocolPreferredRepeatersFileID);
        return file?.nodeIds;
      }
      case "appRouteLock": {
        const file = await getFile(import_files.ProtocolAppRouteLockNodeMaskFileID);
        return file?.nodeIds;
      }
      case "routeSlaveSUC": {
        const file = await getFile(import_files.ProtocolRouteSlaveSUCNodeMaskFileID);
        return file?.nodeIds;
      }
      case "sucPendingUpdate": {
        const file = await getFile(import_files.ProtocolSUCPendingUpdateNodeMaskFileID);
        return file?.nodeIds;
      }
      case "pendingDiscovery": {
        const file = await getFile(import_files.ProtocolPendingDiscoveryNodeMaskFileID);
        return file?.nodeIds;
      }
      case "nodeIds": {
        const file = await getFile(import_files.ProtocolNodeListFileID);
        return file?.nodeIds;
      }
      case "lrNodeIds": {
        const file = await getFile(import_files.ProtocolLRNodeListFileID);
        return file?.nodeIds;
      }
      case "virtualNodeIds": {
        const file = await getFile(import_files.ProtocolVirtualNodeMaskFileID);
        return file?.nodeIds;
      }
      case "sucUpdateEntries": {
        if (this._protocolInfo.format < 5) {
          const file = await getFile(import_files.SUCUpdateEntriesFileIDV0);
          return file?.updateEntries;
        } else {
          const updateEntries = [];
          for (let index = 0; index < import_consts.SUC_MAX_UPDATES; index += import_files.SUC_UPDATES_PER_FILE_V5) {
            const file = await this._getFile((0, import_files.sucUpdateIndexToSUCUpdateEntriesFileIDV5)(index));
            if (!file)
              break;
            updateEntries.push(...file.updateEntries);
          }
          return updateEntries;
        }
      }
      case "learnedHomeId":
      case "commandClasses":
      case "watchdogStarted":
      case "powerLevelNormal":
      case "powerLevelLow":
      case "powerMode":
      case "powerModeExtintEnable":
      case "powerModeWutTimeout":
        return;
      default:
        (0, import_helpers.assertNever)(property.type);
    }
  }
  async getNodeNVMProperty(property, required) {
    const getFile = /* @__PURE__ */ __name((fileId) => {
      if (required) {
        return this._expectFile(fileId);
      } else {
        return this._getFile(fileId);
      }
    }, "getFile");
    switch (property.type) {
      case "info": {
        if (this._protocolInfo.format < 1) {
          const file = await getFile((0, import_files.nodeIdToNodeInfoFileIDV0)(property.nodeId));
          return file?.nodeInfo;
        } else {
          const file = await getFile((0, import_files.nodeIdToNodeInfoFileIDV1)(property.nodeId));
          return file?.nodeInfos.find((info) => info.nodeId === property.nodeId);
        }
      }
      case "routes": {
        const nodeMaskFile = await this.getFile(import_files.ProtocolRouteCacheExistsNodeMaskFileID);
        if (!nodeMaskFile)
          return;
        if (!nodeMaskFile.nodeIdSet.has(property.nodeId))
          return;
        let routeCache;
        if (this._protocolInfo.format < 1) {
          const file = await getFile((0, import_files.nodeIdToRouteCacheFileIDV0)(property.nodeId));
          routeCache = file?.routeCache;
        } else {
          const file = await getFile((0, import_files.nodeIdToRouteCacheFileIDV1)(property.nodeId));
          routeCache = file?.routeCaches.find((route) => route.nodeId === property.nodeId);
        }
        if (!routeCache)
          return;
        return {
          lwr: routeCache.lwr,
          nlwr: routeCache.nlwr
        };
      }
      default:
        (0, import_helpers.assertNever)(property.type);
    }
  }
  async getLRNodeNVMProperty(property, required) {
    const getFile = /* @__PURE__ */ __name((fileId) => {
      if (required) {
        return this._expectFile(fileId);
      } else {
        return this._getFile(fileId);
      }
    }, "getFile");
    switch (property.type) {
      case "info": {
        const file = await getFile((0, import_files.nodeIdToLRNodeInfoFileIDV5)(property.nodeId));
        return file?.nodeInfos.find((info) => info.nodeId === property.nodeId);
      }
      default:
        (0, import_helpers.assertNever)(property.type);
    }
  }
  async set(property, value) {
    if (!this._initialized)
      await this.init();
    if (property.domain === "controller") {
      return this.setControllerNVMProperty(property, value);
    } else if (property.domain === "lrnode") {
      return this.setLRNodeNVMProperty(property, value);
    } else {
      return this.setNodeNVMProperty(property, value);
    }
  }
  async setControllerNVMProperty(property, value) {
    function failFileMissing() {
      throw new import_core.ZWaveError("Cannot set property in NVM for non-existing file", import_core.ZWaveErrorCodes.NVM_ObjectNotFound);
    }
    __name(failFileMissing, "failFileMissing");
    const expectFile = /* @__PURE__ */ __name(async (fileId) => {
      const file = await this._getFile(fileId);
      if (!file)
        failFileMissing();
      return file;
    }, "expectFile");
    const changedFiles = [];
    const deletedFiles = [];
    switch (property.type) {
      case "protocolVersion": {
        const file = await expectFile(import_files.ProtocolVersionFileID);
        const [major, minor, patch] = value.split(".").map((part) => parseInt(part, 10));
        file.major = major;
        file.minor = minor;
        file.patch = patch;
        changedFiles.push(file);
        break;
      }
      case "protocolFileFormat": {
        const file = await expectFile(import_files.ProtocolVersionFileID);
        file.format = value;
        changedFiles.push(file);
        break;
      }
      case "applicationVersion": {
        const file700 = await this._getFile(import_files.ApplicationVersionFileID);
        const file800 = await this._getFile(import_files.ApplicationVersionFile800ID);
        const file = file700 ?? file800;
        if (!file) {
          throw new import_core.ZWaveError("ApplicationVersionFile not found!", import_core.ZWaveErrorCodes.NVM_ObjectNotFound);
        }
        const [major, minor, patch] = value.split(".").map((part) => parseInt(part, 10));
        file.major = major;
        file.minor = minor;
        file.patch = patch;
        changedFiles.push(file);
        break;
      }
      case "applicationFileFormat": {
        const file = await expectFile(import_files.ApplicationVersionFileID);
        file.format = value;
        changedFiles.push(file);
        break;
      }
      case "applicationData": {
        const file = new import_files.ApplicationDataFile({
          applicationData: value,
          fileVersion: this.getFileVersion(import_files.ApplicationDataFileID)
        });
        file.applicationData = value;
        changedFiles.push(file);
        break;
      }
      case "applicationName": {
        const file = new import_files.ApplicationNameFile({
          name: value,
          fileVersion: this.getFileVersion(import_files.ApplicationNameFileID)
        });
        changedFiles.push(file);
        break;
      }
      case "homeId":
      case "nodeId":
      case "lastNodeId":
      case "staticControllerNodeId":
      case "sucLastIndex":
      case "controllerConfiguration":
      case "sucAwarenessPushNeeded":
      case "maxNodeId":
      case "reservedId":
      case "systemState":
      case "lastNodeIdLR":
      case "maxNodeIdLR":
      case "reservedIdLR":
      case "primaryLongRangeChannelId":
      case "dcdcConfig": {
        const file = await expectFile(import_files.ControllerInfoFileID);
        file[property.type] = value;
        changedFiles.push(file);
        break;
      }
      case "includedInsecurely":
      case "includedSecurelyInsecureCCs":
      case "includedSecurelySecureCCs": {
        const file = await expectFile(import_files.ApplicationCCsFileID);
        file[property.type] = value;
        changedFiles.push(file);
        break;
      }
      case "rfRegion":
      case "txPower":
      case "measured0dBm":
      case "enablePTI":
      case "maxTXPower":
      case "nodeIdType": {
        const file = await expectFile(import_files.ApplicationRFConfigFileID);
        file[property.type] = value;
        changedFiles.push(file);
        break;
      }
      case "isListening":
      case "optionalFunctionality":
      case "genericDeviceClass":
      case "specificDeviceClass": {
        const file = await expectFile(import_files.ApplicationTypeFileID);
        file[property.type] = value;
        changedFiles.push(file);
        break;
      }
      case "nodeIds": {
        const file = await this._getFile(import_files.ProtocolNodeListFileID) ?? new import_files.ProtocolNodeListFile({
          nodeIds: [],
          fileVersion: this.getFileVersion(import_files.ProtocolNodeListFileID)
        });
        file.nodeIds = value;
        changedFiles.push(file);
        break;
      }
      case "lrNodeIds": {
        const file = await this._getFile(import_files.ProtocolLRNodeListFileID) ?? new import_files.ProtocolLRNodeListFile({
          nodeIds: [],
          fileVersion: this.getFileVersion(import_files.ProtocolLRNodeListFileID)
        });
        file.nodeIds = value;
        changedFiles.push(file);
        break;
      }
      case "virtualNodeIds": {
        const file = await this._getFile(import_files.ProtocolVirtualNodeMaskFileID) ?? new import_files.ProtocolVirtualNodeMaskFile({
          nodeIds: [],
          fileVersion: this.getFileVersion(import_files.ProtocolVirtualNodeMaskFileID)
        });
        file.nodeIds = value;
        changedFiles.push(file);
        break;
      }
      case "preferredRepeaters": {
        const file = new import_files.ProtocolPreferredRepeatersFile({
          nodeIds: value,
          fileVersion: this.getFileVersion(import_files.ProtocolPreferredRepeatersFileID)
        });
        changedFiles.push(file);
        break;
      }
      case "appRouteLock": {
        const file = new import_files.ProtocolAppRouteLockNodeMaskFile({
          nodeIds: value,
          fileVersion: this.getFileVersion(import_files.ProtocolAppRouteLockNodeMaskFileID)
        });
        changedFiles.push(file);
        break;
      }
      case "routeSlaveSUC": {
        const file = new import_files.ProtocolRouteSlaveSUCNodeMaskFile({
          nodeIds: value,
          fileVersion: this.getFileVersion(import_files.ProtocolRouteSlaveSUCNodeMaskFileID)
        });
        changedFiles.push(file);
        break;
      }
      case "sucPendingUpdate": {
        const file = new import_files.ProtocolSUCPendingUpdateNodeMaskFile({
          nodeIds: value,
          fileVersion: this.getFileVersion(import_files.ProtocolSUCPendingUpdateNodeMaskFileID)
        });
        changedFiles.push(file);
        break;
      }
      case "pendingDiscovery": {
        const file = new import_files.ProtocolPendingDiscoveryNodeMaskFile({
          nodeIds: value,
          fileVersion: this.getFileVersion(import_files.ProtocolPendingDiscoveryNodeMaskFileID)
        });
        changedFiles.push(file);
        break;
      }
      case "sucUpdateEntries": {
        if (this._protocolInfo.format < 5) {
          const file = new import_files.SUCUpdateEntriesFileV0({
            updateEntries: value,
            fileVersion: this.getFileVersion(import_files.SUCUpdateEntriesFileIDV0)
          });
          changedFiles.push(file);
          break;
        } else {
          for (let index = 0; index < import_consts.SUC_MAX_UPDATES; index += import_files.SUC_UPDATES_PER_FILE_V5) {
            const fileId = (0, import_files.sucUpdateIndexToSUCUpdateEntriesFileIDV5)(index);
            const fileExists = await this.hasFile(fileId);
            const fileVersion = this.getFileVersion(fileId);
            const slice = value.slice(index, index + import_files.SUC_UPDATES_PER_FILE_V5);
            if (slice.length > 0) {
              const file = new import_files.SUCUpdateEntriesFileV5({
                updateEntries: slice,
                fileId,
                fileVersion
              });
              changedFiles.push(file);
            } else if (fileExists) {
              deletedFiles.push(fileId);
            }
          }
        }
        break;
      }
      case "learnedHomeId":
      case "commandClasses":
      case "watchdogStarted":
      case "powerLevelNormal":
      case "powerLevelLow":
      case "powerMode":
      case "powerModeExtintEnable":
      case "powerModeWutTimeout":
        return;
      default:
        (0, import_helpers.assertNever)(property.type);
    }
    for (const file of changedFiles) {
      const { key, data } = file.serialize();
      this._pendingChanges.set(key, data);
    }
    for (const file of deletedFiles) {
      this._pendingChanges.set(file, null);
    }
  }
  async setLRNodeNVMProperty(property, value) {
    const changedFiles = [];
    const deletedFiles = [];
    switch (property.type) {
      case "info": {
        const fileId = (0, import_files.nodeIdToLRNodeInfoFileIDV5)(property.nodeId);
        let file = await this._getFile(fileId);
        if (value) {
          file ??= new import_files.LRNodeInfoFileV5({
            nodeInfos: [],
            fileVersion: this.getFileVersion(fileId)
          });
          const existingIndex = file.nodeInfos.findIndex((info) => info.nodeId === property.nodeId);
          if (existingIndex !== -1) {
            file.nodeInfos[existingIndex] = value;
          } else {
            file.nodeInfos.push(value);
          }
          changedFiles.push(file);
        } else if (file) {
          const existingIndex = file.nodeInfos.findIndex((info) => info.nodeId === property.nodeId);
          if (existingIndex !== -1) {
            file.nodeInfos.splice(existingIndex, 1);
            if (file.nodeInfos.length === 0) {
              deletedFiles.push(fileId);
            } else {
              changedFiles.push(file);
            }
          }
        }
        break;
      }
      default:
        (0, import_helpers.assertNever)(property.type);
    }
    for (const file of changedFiles) {
      const { key, data } = file.serialize();
      this._pendingChanges.set(key, data);
    }
    for (const file of deletedFiles) {
      this._pendingChanges.set(file, null);
    }
  }
  async setNodeNVMProperty(property, value) {
    const changedFiles = [];
    const deletedFiles = [];
    switch (property.type) {
      case "info": {
        if (this._protocolInfo.format < 1) {
          const fileId = (0, import_files.nodeIdToNodeInfoFileIDV0)(property.nodeId);
          let file = await this._getFile(fileId);
          if (value) {
            file ??= new import_files.NodeInfoFileV0({
              nodeInfo: void 0,
              fileVersion: this.getFileVersion(fileId)
            });
            file.nodeInfo = value;
            changedFiles.push(file);
          } else {
            deletedFiles.push(fileId);
          }
        } else {
          const fileId = (0, import_files.nodeIdToNodeInfoFileIDV1)(property.nodeId);
          let file = await this._getFile(fileId);
          if (value) {
            file ??= new import_files.NodeInfoFileV1({
              nodeInfos: [],
              fileVersion: this.getFileVersion(fileId)
            });
            const existingIndex = file.nodeInfos.findIndex((info) => info.nodeId === property.nodeId);
            if (existingIndex !== -1) {
              file.nodeInfos[existingIndex] = value;
            } else {
              file.nodeInfos.push(value);
            }
            changedFiles.push(file);
          } else if (file) {
            const existingIndex = file.nodeInfos.findIndex((info) => info.nodeId === property.nodeId);
            if (existingIndex !== -1) {
              file.nodeInfos.splice(existingIndex, 1);
              if (file.nodeInfos.length === 0) {
                deletedFiles.push(fileId);
              } else {
                changedFiles.push(file);
              }
            }
          }
        }
        break;
      }
      case "routes": {
        if (this._protocolInfo.format < 1) {
          const fileId = (0, import_files.nodeIdToRouteCacheFileIDV0)(property.nodeId);
          let file = await this._getFile(fileId);
          if (value) {
            file ??= new import_files.RouteCacheFileV0({
              routeCache: void 0,
              fileVersion: this.getFileVersion(fileId)
            });
            file.routeCache = {
              nodeId: property.nodeId,
              lwr: value.lwr,
              nlwr: value.nlwr
            };
            changedFiles.push(file);
          } else if (file) {
            deletedFiles.push(fileId);
          }
        } else {
          const fileId = (0, import_files.nodeIdToRouteCacheFileIDV1)(property.nodeId);
          const file = await this._getFile(fileId) ?? new import_files.RouteCacheFileV1({
            routeCaches: [],
            fileVersion: this.getFileVersion(fileId)
          });
          const existingIndex = file.routeCaches.findIndex((route) => route.nodeId === property.nodeId);
          const newRoute = {
            nodeId: property.nodeId,
            lwr: value.lwr,
            nlwr: value.nlwr
          };
          if (existingIndex !== -1) {
            file.routeCaches[existingIndex] = newRoute;
          } else {
            file.routeCaches.push(newRoute);
          }
          changedFiles.push(file);
        }
        const nodeMaskFile = await this._getFile(import_files.ProtocolRouteCacheExistsNodeMaskFileID) ?? new import_files.ProtocolRouteCacheExistsNodeMaskFile({
          nodeIds: [],
          fileVersion: this.getFileVersion(import_files.ProtocolRouteCacheExistsNodeMaskFileID)
        });
        if (!value && nodeMaskFile.nodeIdSet.has(property.nodeId)) {
          nodeMaskFile.nodeIdSet.delete(property.nodeId);
          changedFiles.push(nodeMaskFile);
        } else if (value && !nodeMaskFile.nodeIdSet.has(property.nodeId)) {
          nodeMaskFile.nodeIdSet.add(property.nodeId);
          changedFiles.push(nodeMaskFile);
        }
        break;
      }
      default:
        (0, import_helpers.assertNever)(property.type);
    }
    for (const file of changedFiles) {
      const { key, data } = file.serialize();
      this._pendingChanges.set(key, data);
    }
    for (const file of deletedFiles) {
      this._pendingChanges.set(file, null);
    }
  }
  async delete(property) {
    if (property.domain === "controller") {
      switch (property.type) {
        case "protocolVersion":
        case "protocolFileFormat": {
          this._pendingChanges.set(import_files.ProtocolVersionFileID, null);
          return;
        }
        case "applicationVersion":
        case "applicationFileFormat": {
          if (await this.hasFile(import_files.ApplicationVersionFileID)) {
            this._pendingChanges.set(import_files.ApplicationVersionFileID, null);
          }
          if (await this.hasFile(import_files.ApplicationVersionFile800ID)) {
            this._pendingChanges.set(import_files.ApplicationVersionFile800ID, null);
          }
          return;
        }
        case "applicationData": {
          this._pendingChanges.set(import_files.ApplicationDataFileID, null);
          return;
        }
        case "applicationName": {
          this._pendingChanges.set(import_files.ApplicationNameFileID, null);
          return;
        }
        case "homeId":
        case "nodeId":
        case "lastNodeId":
        case "staticControllerNodeId":
        case "sucLastIndex":
        case "controllerConfiguration":
        case "sucAwarenessPushNeeded":
        case "maxNodeId":
        case "reservedId":
        case "systemState":
        case "lastNodeIdLR":
        case "maxNodeIdLR":
        case "reservedIdLR":
        case "primaryLongRangeChannelId":
        case "dcdcConfig": {
          this._pendingChanges.set(import_files.ControllerInfoFileID, null);
          return;
        }
        case "includedInsecurely":
        case "includedSecurelyInsecureCCs":
        case "includedSecurelySecureCCs": {
          this._pendingChanges.set(import_files.ApplicationCCsFileID, null);
          return;
        }
        case "rfRegion":
        case "txPower":
        case "measured0dBm":
        case "enablePTI":
        case "maxTXPower":
        case "nodeIdType": {
          this._pendingChanges.set(import_files.ApplicationRFConfigFileID, null);
          return;
        }
        case "isListening":
        case "optionalFunctionality":
        case "genericDeviceClass":
        case "specificDeviceClass": {
          this._pendingChanges.set(import_files.ApplicationTypeFileID, null);
          return;
        }
        case "nodeIds": {
          this._pendingChanges.set(import_files.ProtocolNodeListFileID, null);
          return;
        }
        case "lrNodeIds": {
          this._pendingChanges.set(import_files.ProtocolLRNodeListFileID, null);
          return;
        }
        case "virtualNodeIds": {
          this._pendingChanges.set(import_files.ProtocolVirtualNodeMaskFileID, null);
          return;
        }
        case "preferredRepeaters": {
          this._pendingChanges.set(import_files.ProtocolPreferredRepeatersFileID, null);
          return;
        }
        case "appRouteLock": {
          this._pendingChanges.set(import_files.ProtocolAppRouteLockNodeMaskFileID, null);
          return;
        }
        case "routeSlaveSUC": {
          this._pendingChanges.set(import_files.ProtocolRouteSlaveSUCNodeMaskFileID, null);
          return;
        }
        case "sucPendingUpdate": {
          this._pendingChanges.set(import_files.ProtocolSUCPendingUpdateNodeMaskFileID, null);
          return;
        }
        case "pendingDiscovery": {
          this._pendingChanges.set(import_files.ProtocolPendingDiscoveryNodeMaskFileID, null);
          return;
        }
        case "sucUpdateEntries": {
          if (this._protocolInfo.format < 5) {
            this._pendingChanges.set(import_files.SUCUpdateEntriesFileIDV0, null);
          } else {
            for (let id = import_files.SUCUpdateEntriesFileV5IDBase; id <= import_files.SUCUpdateEntriesFileV5IDMax; id++) {
              if (await this.hasFile(id)) {
                this._pendingChanges.set(id, null);
              }
            }
          }
          return;
        }
        case "learnedHomeId":
        case "commandClasses":
        case "watchdogStarted":
        case "powerLevelNormal":
        case "powerLevelLow":
        case "powerMode":
        case "powerModeExtintEnable":
        case "powerModeWutTimeout":
          return;
        default:
          (0, import_helpers.assertNever)(property);
      }
    } else if (property.domain === "lrnode") {
      return this.setLRNodeNVMProperty(property, void 0);
    } else if (property.domain === "node") {
      return this.setNodeNVMProperty(property, void 0);
    }
  }
  hasPendingChanges() {
    return this._pendingChanges.size > 0;
  }
  async commit() {
    await this._nvm.setMany([...this._pendingChanges]);
  }
}
// Annotate the CommonJS export names for ESM import in node:
0 && (module.exports = {
  NVM3Adapter
});
//# sourceMappingURL=adapter.js.map
