// This is basically a duplex transform stream wrapper around a
// readable and a writable stream, both compatible with the Web Streams API.
// This allows supporting different low-level connections to a Z-Wave controller,
// whether it's a serial port, a network connection, or something else.
//
// 0 ┌─────────────────┐ ┌─────────────────┐ ┌──
// 1 <--               <--   PassThrough   <-- write
// 1 │    any stream   │ │ ZWaveSerialPort │ │
// 0 -->               -->     Parsers     --> read
// 1 └─────────────────┘ └─────────────────┘ └──
import { noop } from "@zwave-js/shared";
import { SerialLogger } from "../log/Logger.js";
import { ZnifferParser } from "../parsers/ZnifferParser.js";
/** Re-usable stream factory to create new serial streams */
export class ZnifferSerialStreamFactory {
    constructor(binding, loggers) {
        this.binding = binding;
        this.logger = new SerialLogger(loggers);
    }
    binding;
    logger;
    async createStream() {
        // Set up streams for the underlying resource
        const { source, sink } = await this.binding();
        return new ZnifferSerialStream(source, sink, this.logger);
    }
}
/** Single-use serial stream. Has to be re-created after being closed. */
export class ZnifferSerialStream {
    constructor(source, sink, logger) {
        this.logger = logger;
        this.#abort = new AbortController();
        // Expose the underlying sink as the writable side of this stream.
        // We use an identity stream in the middle to pipe through, so we
        // can properly abort the stream
        const { readable: input, writable } = new TransformStream();
        this.writable = writable;
        const sinkStream = new WritableStream(sink);
        void input
            .pipeTo(sinkStream, { signal: this.#abort.signal })
            .catch(noop);
        // Pipe the underlying source through the parser to the readable side
        const { readable, writable: output } = new TransformStream();
        this.readable = readable;
        const parser = new ZnifferParser(logger);
        const sourceStream = new ReadableStream(source);
        void sourceStream
            .pipeThrough(parser, { signal: this.#abort.signal })
            .pipeTo(output, { signal: this.#abort.signal })
            .catch((_e) => {
            this._isOpen = false;
        });
    }
    logger;
    // Public interface to let consumers read from and write to this stream
    readable;
    writable;
    // Signal to close the underlying stream
    #abort;
    async close() {
        this._isOpen = false;
        // Close the underlying stream
        this._writer?.releaseLock();
        this.#abort.abort();
        return Promise.resolve();
        // // Wait for streams to finish
        // await this.#pipePromise;
    }
    _isOpen = true;
    get isOpen() {
        return this._isOpen;
    }
    _writer;
    async writeAsync(data) {
        if (!this.isOpen) {
            throw new Error("The serial port is not open!");
        }
        this.logger.data("outbound", data);
        // Keep a writer instance to avoid locking and unlocking the
        // writable stream for every write, as this can cause errors
        // when writing in quick succession.
        this._writer ??= this.writable.getWriter();
        await this._writer.write(data);
    }
}
//# sourceMappingURL=ZnifferSerialStream.js.map