import { type AssociationAddress, type AssociationCheckResult, type AssociationGroup, type FirmwareUpdateOptions, type FirmwareUpdateResult, KEXFailType } from "@zwave-js/cc";
import { type ControllerCapabilities, ControllerRole, ControllerStatus, LongRangeChannel, type MaybeNotKnown, type MaybeUnknown, NodeIDType, NodeType, RFRegion, type RFRegionInfo, type RSSI, type Route, RouteKind, type SerialApiInitData, type UnknownZWaveChipType, ValueDB, type ZWaveApiVersion, type ZWaveDataRate, ZWaveLibraryTypes } from "@zwave-js/core";
import { type MigrateNVMOptions, type NVMAdapter } from "@zwave-js/nvmedit";
import { FunctionType } from "@zwave-js/serial";
import { ExtendedNVMOperationsCommand, type NVMId, SerialAPISetupCommand, type SerialAPISetup_GetPowerlevelResponse } from "@zwave-js/serial/serialapi";
import { type ReadonlyObjectKeyMap, type ReadonlyThrowingMap, TypedEventTarget } from "@zwave-js/shared";
import type { StatisticsEventCallbacks } from "../driver/Statistics.js";
import { ZWaveNode } from "../node/Node.js";
import { VirtualNode } from "../node/VirtualNode.js";
import { type LifelineRoutes } from "../node/_Types.js";
import { type ControllerStatistics, ControllerStatisticsHost } from "./ControllerStatistics.js";
import { ZWaveFeature } from "./Features.js";
import { type ExclusionOptions, type FoundNode, type InclusionOptions, type InclusionResult, InclusionState, InclusionStrategy, type JoinNetworkOptions, JoinNetworkResult, LeaveNetworkResult, type PlannedProvisioningEntry, RemoveNodeReason, type ReplaceNodeOptions, type SmartStartProvisioningEntry } from "./Inclusion.js";
import { type ControllerProprietary } from "./Proprietary.js";
import type { FirmwareUpdateInfo, GetFirmwareUpdatesOptions, RebuildRoutesOptions, RebuildRoutesStatus, SDKVersion } from "./_Types.js";
interface ControllerEventCallbacks extends StatisticsEventCallbacks<ControllerStatistics> {
    "inclusion failed": () => void;
    "exclusion failed": () => void;
    "inclusion started": (strategy: InclusionStrategy) => void;
    "exclusion started": () => void;
    "inclusion stopped": () => void;
    "exclusion stopped": () => void;
    "inclusion state changed": (state: InclusionState) => void;
    "node found": (node: FoundNode) => void;
    "node added": (node: ZWaveNode, result: InclusionResult) => void;
    "node removed": (node: ZWaveNode, reason: RemoveNodeReason) => void;
    "network found": (homeId: number, ownNodeId: number) => void;
    "joining network failed": () => void;
    "network joined": () => void;
    "network left": () => void;
    "leaving network failed": () => void;
    "rebuild routes progress": (progress: ReadonlyMap<number, RebuildRoutesStatus>) => void;
    "rebuild routes done": (result: ReadonlyMap<number, RebuildRoutesStatus>) => void;
    identify: (node: ZWaveNode) => void;
    "status changed": (status: ControllerStatus) => void;
}
export type ControllerEvents = Extract<keyof ControllerEventCallbacks, string>;
export interface ZWaveController extends ControllerStatisticsHost {
}
export declare class ZWaveController extends TypedEventTarget<ControllerEventCallbacks> {
    private readonly driver;
    private _type;
    get type(): MaybeNotKnown<ZWaveLibraryTypes>;
    private _protocolVersion;
    get protocolVersion(): MaybeNotKnown<string>;
    private _sdkVersion;
    get sdkVersion(): MaybeNotKnown<string>;
    private _zwaveApiVersion;
    get zwaveApiVersion(): MaybeNotKnown<ZWaveApiVersion>;
    private _zwaveChipType;
    get zwaveChipType(): MaybeNotKnown<string | UnknownZWaveChipType>;
    private _homeId;
    /** A 32bit number identifying the current network */
    get homeId(): MaybeNotKnown<number>;
    private _ownNodeId;
    /** The ID of the controller in the current network */
    get ownNodeId(): MaybeNotKnown<number>;
    private _dsk;
    /**
     * The device specific key (DSK) of the controller in binary format.
     */
    getDSK(): Promise<Uint8Array>;
    /** @deprecated Use {@link role} instead */
    get isPrimary(): MaybeNotKnown<boolean>;
    private _isPrimary;
    private _isSecondary;
    private _isUsingHomeIdFromOtherNetwork;
    /** @deprecated Use {@link role} instead */
    get isUsingHomeIdFromOtherNetwork(): MaybeNotKnown<boolean>;
    private _isSISPresent;
    get isSISPresent(): MaybeNotKnown<boolean>;
    private _wasRealPrimary;
    /** @deprecated Use {@link role} instead */
    get wasRealPrimary(): MaybeNotKnown<boolean>;
    private _isSIS;
    get isSIS(): MaybeNotKnown<boolean>;
    private _isSUC;
    get isSUC(): MaybeNotKnown<boolean>;
    private _noNodesIncluded;
    private _nodeType;
    get nodeType(): MaybeNotKnown<NodeType>;
    /** Checks if the SDK version is greater than the given one */
    sdkVersionGt(version: SDKVersion): MaybeNotKnown<boolean>;
    /** Checks if the SDK version is greater than or equal to the given one */
    sdkVersionGte(version: SDKVersion): MaybeNotKnown<boolean>;
    /** Checks if the SDK version is lower than the given one */
    sdkVersionLt(version: SDKVersion): MaybeNotKnown<boolean>;
    /** Checks if the SDK version is lower than or equal to the given one */
    sdkVersionLte(version: SDKVersion): MaybeNotKnown<boolean>;
    private _manufacturerId;
    get manufacturerId(): MaybeNotKnown<number>;
    private _productType;
    get productType(): MaybeNotKnown<number>;
    private _productId;
    get productId(): MaybeNotKnown<number>;
    private _firmwareVersion;
    get firmwareVersion(): MaybeNotKnown<string>;
    private _supportedFunctionTypes;
    get supportedFunctionTypes(): MaybeNotKnown<readonly FunctionType[]>;
    private _status;
    /**
     * Which status the controller is believed to be in
     */
    get status(): ControllerStatus;
    /**
     * Checks if a given Z-Wave function type is supported by this controller.
     * Returns `NOT_KNOWN`/`undefined` if this information isn't known yet.
     */
    isFunctionSupported(functionType: FunctionType): MaybeNotKnown<boolean>;
    private _supportedSerialAPISetupCommands;
    get supportedSerialAPISetupCommands(): readonly SerialAPISetupCommand[] | undefined;
    /**
     * Checks if a given Serial API setup command is supported by this controller.
     * Returns `NOT_KNOWN`/`undefined` if this information isn't known yet.
     */
    isSerialAPISetupCommandSupported(command: SerialAPISetupCommand): MaybeNotKnown<boolean>;
    /**
     * Tests if the controller supports a certain feature.
     * Returns `undefined` if this information isn't known yet.
     */
    supportsFeature(feature: ZWaveFeature): MaybeNotKnown<boolean>;
    /** Throws if the controller does not support a certain feature */
    private assertFeature;
    private _sucNodeId;
    get sucNodeId(): MaybeNotKnown<number>;
    private _supportsTimers;
    get supportsTimers(): MaybeNotKnown<boolean>;
    private _supportedRegions;
    /** Which RF regions are supported by the controller, including information about them */
    get supportedRegions(): MaybeNotKnown<ReadonlyMap<RFRegion, Readonly<RFRegionInfo>>>;
    private _rfRegion;
    /** Which RF region the controller is currently set to, or `undefined` if it could not be determined (yet). This value is cached and can be changed through {@link setRFRegion}. */
    get rfRegion(): MaybeNotKnown<RFRegion>;
    private _txPower;
    /** The transmit power used for Z-Wave mesh, or `undefined` if it could not be determined (yet). This value is cached and can be changed through {@link setPowerlevel}. */
    get txPower(): MaybeNotKnown<number>;
    private _powerlevelCalibration;
    /** The calibration value for the transmit power, or `undefined` if it could not be determined (yet). This value is cached and can be changed through {@link setPowerlevel}. */
    get powerlevelCalibration(): MaybeNotKnown<number>;
    private _supportsLongRange;
    /** Whether the controller supports the Z-Wave Long Range protocol */
    get supportsLongRange(): MaybeNotKnown<boolean>;
    private _maxLongRangePowerlevel;
    /** The maximum powerlevel to use for Z-Wave Long Range, or `undefined` if it could not be determined (yet). This value is cached and can be changed through {@link setMaxLongRangePowerlevel}. */
    get maxLongRangePowerlevel(): MaybeNotKnown<number>;
    private _longRangeChannel;
    /** The channel to use for Z-Wave Long Range, or `undefined` if it could not be determined (yet). This value is cached and can be changed through {@link setLongRangeChannel}. */
    get longRangeChannel(): MaybeNotKnown<LongRangeChannel>;
    private _supportsLongRangeAutoChannelSelection;
    /** Whether automatic LR channel selection is supported, or `undefined` if it could not be determined (yet). */
    get supportsLongRangeAutoChannelSelection(): MaybeNotKnown<boolean>;
    private _maxPayloadSize;
    /** The maximum payload size that can be transmitted with a Z-Wave explorer frame */
    get maxPayloadSize(): MaybeNotKnown<number>;
    private _maxPayloadSizeLR;
    /** The maximum payload size that can be transmitted with a Z-Wave Long Range frame */
    get maxPayloadSizeLR(): MaybeNotKnown<number>;
    private _nodes;
    /** A dictionary of the nodes connected to this controller */
    get nodes(): ReadonlyThrowingMap<number, ZWaveNode>;
    private _nodeIdType;
    /** Whether the controller is configured to use 8 or 16 bit node IDs */
    get nodeIdType(): NodeIDType;
    /** Returns the node with the given DSK */
    getNodeByDSK(dsk: Uint8Array | string): ZWaveNode | undefined;
    /** Returns the controller node's value DB */
    get valueDB(): ValueDB;
    private _powerlevel;
    /** The role of the controller on the network */
    get role(): MaybeNotKnown<ControllerRole>;
    /** Returns whether learn mode may be enabled on this controller */
    get isLearnModePermitted(): boolean;
    /** Returns whether the routes are currently being rebuilt for one or more nodes. */
    get isRebuildingRoutes(): boolean;
    /**
     * Returns a reference to the (virtual) broadcast node, which allows sending commands to all nodes.
     * This automatically groups nodes by security class, ignores nodes that cannot be controlled via multicast/broadcast, and will fall back to multicast(s) if necessary.
     */
    getBroadcastNode(): VirtualNode;
    /**
     * Returns a reference to the (virtual) broadcast node for Z-Wave Long Range, which allows sending commands to all LR nodes.
     * This automatically groups nodes by security class, ignores nodes that cannot be controlled via multicast/broadcast, and will fall back to multicast(s) if necessary.
     */
    getBroadcastNodeLR(): VirtualNode;
    /**
     * Creates a virtual node that can be used to send one or more multicast commands to several nodes.
     * This automatically groups nodes by security class and ignores nodes that cannot be controlled via multicast.
     */
    getMulticastGroup(nodeIDs: number[]): VirtualNode;
    private set provisioningList(value);
    /** Adds the given entry (DSK and security classes) to the controller's SmartStart provisioning list or replaces an existing entry */
    provisionSmartStartNode(entry: PlannedProvisioningEntry): void;
    /**
     * Removes the given DSK or node ID from the controller's SmartStart provisioning list.
     *
     * **Note:** If this entry corresponds to an included node, it will **NOT** be excluded
     */
    unprovisionSmartStartNode(dskOrNodeId: string | number): void;
    private getProvisioningEntryInternal;
    /**
     * Returns the entry for the given DSK or node ID from the controller's SmartStart provisioning list.
     */
    getProvisioningEntry(dskOrNodeId: string | number): Readonly<SmartStartProvisioningEntry> | undefined;
    /**
     * Returns all entries from the controller's SmartStart provisioning list.
     */
    getProvisioningEntries(): SmartStartProvisioningEntry[];
    /** Returns whether the SmartStart provisioning list contains active entries that have not been included yet */
    hasPlannedProvisioningEntries(): boolean;
    private isLongRangeCapable;
    /** Tries to determine the LR capable replacement of the given region. If none is found, the given region is returned. */
    private tryGetLRCapableRegion;
    private createValueDBForNode;
    /**
     * Gets the list of long range nodes from the controller.
     */
    getLongRangeNodes(): Promise<readonly number[]>;
    /**
     * Starts the hardware watchdog on supporting 700+ series controllers.
     * Returns whether the operation was successful.
     */
    startWatchdog(): Promise<boolean>;
    /**
     * Stops the hardware watchdog on supporting controllers.
     * Returns whether the operation was successful.
     */
    stopWatchdog(): Promise<boolean>;
    private _inclusionState;
    get inclusionState(): InclusionState;
    private _smartStartEnabled;
    /**
     * Starts the inclusion process of new nodes.
     * Resolves to true when the process was started, and false if the inclusion was already active.
     *
     * @param options Defines the inclusion strategy to use.
     */
    beginInclusion(options?: InclusionOptions): Promise<boolean>;
    /**
     * Returns the task to handle the complete classic inclusion process
     */
    private getBeginClassicInclusionTask;
    /**
     * Returns the task to handle the complete classic inclusion process
     */
    private getBeginSmartStartInclusionTask;
    private performInclusion;
    /**
     * Finishes an inclusion process. This must only be called after the ProtocolDone status is received.
     * Returns the ID of the newly added node.
     */
    private finishInclusion;
    /**
     * Stops an active inclusion process. Resolves to true when the controller leaves inclusion mode,
     * and false if the inclusion was not active.
     */
    stopInclusion(): Promise<boolean>;
    /**
     * Stops an active inclusion process, but does not remove the corresponding task.
     * This should only be used from within an inclusion task, or other methods that
     * handle task removal
     */
    private stopInclusionInternal;
    /**
     * Puts the controller into listening mode for Smart Start inclusion.
     * Whenever a node on the provisioning list announces itself, it will automatically be added.
     *
     * Resolves to `true` when the listening mode is started or was active, and `false` if it is scheduled for later activation.
     */
    private enableSmartStart;
    /**
     * Disables the listening mode for Smart Start inclusion.
     *
     * Resolves to `true` when the listening mode is stopped, and `false` if was not active.
     */
    private disableSmartStart;
    private pauseSmartStart;
    /**
     * Starts the exclusion process of new nodes.
     * Resolves to true when the process was started, and false if an inclusion or exclusion process was already active.
     *
     * @param options Influences the exclusion process and what happens with the Smart Start provisioning list.
     */
    beginExclusion(options?: ExclusionOptions): Promise<boolean>;
    /**
     * Stops an active exclusion process. Resolves to true when the controller leaves exclusion mode,
     * and false if the exclusion was not active.
     */
    stopExclusion(): Promise<boolean>;
    /**
     * Stops an active exclusion process, but does not remove the corresponding task.
     * This should only be used from within an exclusion task, or other methods that
     * handle task removal
     */
    private stopExclusionInternal;
    /**
     * Returns the task to handle the complete exclusion process
     */
    private getExclusionTask;
    private performExclusion;
    private _proxyInclusionMachine;
    private _proxyInclusionInitiateTimeout;
    private updateProxyInclusionMachine;
    /**
     * Handles bootstrapping the security keys for a node that was included by an inclusion controller
     */
    private proxyBootstrap;
    private secureBootstrapS0;
    private _bootstrappingS2NodeId;
    private cancelBootstrapS2Promise;
    cancelSecureBootstrapS2(reason: KEXFailType): void;
    private secureBootstrapS2;
    private _rebuildRoutesProgress;
    /**
     * If routes are currently being rebuilt for the entire network, this returns the current progress.
     * The information is the same as in the `"rebuild routes progress"` event.
     */
    get rebuildRoutesProgress(): ReadonlyMap<number, RebuildRoutesStatus> | undefined;
    /**
     * Starts the process of rebuilding routes for all alive nodes in the network,
     * requesting updated neighbor lists and assigning fresh routes to
     * association targets.
     *
     * Returns `true` if the process was started, otherwise `false`. Also returns
     * `false` if the process was already active.
     */
    beginRebuildingRoutes(options?: RebuildRoutesOptions): boolean;
    private rebuildRoutesInternal;
    private getRebuildRoutesTask;
    /**
     * Stops the route rebuilding process. Resolves false if the process was not active, true otherwise.
     */
    stopRebuildingRoutes(): boolean;
    /**
     * Rebuilds routes for a single alive node in the network,
     * updating the neighbor list and assigning fresh routes to
     * association targets.
     *
     * Returns `true` if the process succeeded, `false` otherwise.
     */
    rebuildNodeRoutes(nodeId: number): Promise<boolean>;
    private rebuildNodeRoutesInternal;
    private getRebuildNodeRoutesTask;
    /** Configures the given Node to be SUC/SIS or not */
    configureSUC(nodeId: number, enableSUC: boolean, enableSIS: boolean): Promise<boolean>;
    /**
     * Instructs the controller to assign static routes from the given end node to the SUC.
     * This will assign up to 4 routes, depending on the network topology (that the controller knows about).
     */
    assignSUCReturnRoutes(nodeId: number): Promise<boolean>;
    /**
     * Returns which custom static routes are currently assigned from the given end node to the SUC.
     *
     * **Note:** This only considers routes that were assigned using {@link assignCustomSUCReturnRoutes}.
     * If another controller has assigned routes in the meantime, this information may be out of date.
     */
    getCustomSUCReturnRoutesCached(nodeId: number): Route[];
    private setCustomSUCReturnRoutesCached;
    /**
     * Assigns static routes from the given end node to the SUC. Unlike {@link assignSUCReturnRoutes}, this method assigns
     * the given routes instead of having the controller calculate them. At most 4 routes can be assigned. If less are
     * specified, the remaining routes are cleared.
     *
     * To mark a route as a priority route, pass it as the optional `priorityRoute` parameter. At most 3 routes of the
     * `routes` array will then be used as fallback routes.
     *
     * **Note:** Calling {@link assignSUCReturnRoutes} or {@link deleteSUCReturnRoutes} will override the custom routes.
     *
     * Returns `true` when the process was successful, or `false` if at least one step failed.
     */
    assignCustomSUCReturnRoutes(nodeId: number, routes: Route[], priorityRoute?: Route): Promise<boolean>;
    /**
     * Instructs the controller to assign static routes from the given end node to the SUC.
     * This will assign up to 4 routes, depending on the network topology (that the controller knows about).
     */
    deleteSUCReturnRoutes(nodeId: number): Promise<boolean>;
    /**
     * Returns which custom static routes are currently assigned between the given end nodes.
     *
     * **Note:** This only considers routes that were assigned using {@link assignCustomReturnRoutes}.
     * If another controller has assigned routes in the meantime, this information may be out of date.
     */
    getCustomReturnRoutesCached(nodeId: number, destinationNodeId: number): Route[];
    private setCustomReturnRoutesCached;
    private clearCustomReturnRoutesCached;
    /**
     * Instructs the controller to assign static routes between the two given end nodes.
     * This will assign up to 4 routes, depending on the network topology (that the controller knows about).
     */
    assignReturnRoutes(nodeId: number, destinationNodeId: number): Promise<boolean>;
    /**
     * Assigns static routes between the two given end nodes. Unlike {@link assignReturnRoutes}, this method assigns
     * the given routes instead of having the controller calculate them. At most 4 routes can be assigned. If less are
     * specified, the remaining routes are cleared.
     *
     * **Note:** Calling {@link assignReturnRoutes} or {@link deleteReturnRoutes} will override the custom routes.
     */
    assignCustomReturnRoutes(nodeId: number, destinationNodeId: number, routes: Route[], priorityRoute?: Route): Promise<boolean>;
    /**
     * Instructs the controller to delete all static routes between the given node and all
     * other end nodes, including the priority return routes.
     */
    deleteReturnRoutes(nodeId: number): Promise<boolean>;
    /**
     * Assigns a priority route between two end nodes. This route will always be used for the first transmission attempt.
     * @param nodeId The ID of the source node of the route
     * @param destinationNodeId The ID of the destination node of the route
     * @param repeaters The IDs of the nodes that should be used as repeaters, or an empty array for direct connection
     * @param routeSpeed The transmission speed to use for the route
     */
    assignPriorityReturnRoute(nodeId: number, destinationNodeId: number, repeaters: number[], routeSpeed: ZWaveDataRate): Promise<boolean>;
    private hasPriorityReturnRouteCached;
    private setPriorityReturnRouteCached;
    private clearPriorityReturnRoutesCached;
    /**
     * Returns which priority route is currently assigned between the given end nodes.
     *
     * **Note:** This is using cached information, since there's no way to query priority routes from a node.
     * If another controller has assigned routes in the meantime, this information may be out of date.
     */
    getPriorityReturnRouteCached(nodeId: number, destinationNodeId: number): MaybeUnknown<Route> | undefined;
    /**
     * For the given node, returns all end node destinations and the priority routes to them.
     *
     * **Note:** This is using cached information, since there's no way to query priority routes from a node.
     * If another controller has assigned routes in the meantime, this information may be out of date.
     */
    getPriorityReturnRoutesCached(nodeId: number): Record<number, Route>;
    /**
     * Assigns a priority route from an end node to the SUC. This route will always be used for the first transmission attempt.
     * @param nodeId The ID of the end node for which to assign the route
     * @param repeaters The IDs of the nodes that should be used as repeaters, or an empty array for direct connection
     * @param routeSpeed The transmission speed to use for the route
     */
    assignPrioritySUCReturnRoute(nodeId: number, repeaters: number[], routeSpeed: ZWaveDataRate): Promise<boolean>;
    private setPrioritySUCReturnRouteCached;
    /**
     * Returns which priority route is currently assigned from the given end node to the SUC.
     *
     * **Note:** This is using cached information, since there's no way to query priority routes from a node.
     * If another controller has assigned routes in the meantime, this information may be out of date.
     */
    getPrioritySUCReturnRouteCached(nodeId: number): Route | undefined;
    private handleRouteAssignmentTransmitReport;
    /**
     * Sets the priority route which will always be used for the first transmission attempt from the controller to the given node.
     * @param destinationNodeId The ID of the node that should be reached via the priority route
     * @param repeaters The IDs of the nodes that should be used as repeaters, or an empty array for direct connection
     * @param routeSpeed The transmission speed to use for the route
     */
    setPriorityRoute(destinationNodeId: number, repeaters: number[], routeSpeed: ZWaveDataRate): Promise<boolean>;
    /**
     * Removes the priority route used for the first transmission attempt from the controller to the given node.
     * @param destinationNodeId The ID of the node that should be reached via the priority route
     */
    removePriorityRoute(destinationNodeId: number): Promise<boolean>;
    /**
     * Returns the priority route which is currently set for a node.
     * If none is set, either the LWR or the NLWR is returned.
     * If no route is known yet, this returns `undefined`.
     *
     * @param destinationNodeId The ID of the node for which the priority route should be returned
     */
    getPriorityRoute(destinationNodeId: number): Promise<{
        routeKind: RouteKind.LWR | RouteKind.NLWR | RouteKind.Application;
        repeaters: number[];
        routeSpeed: ZWaveDataRate;
    } | undefined>;
    /**
     * Returns a dictionary of all association groups of this node or endpoint and their information.
     * If no endpoint is given, the associations of the root device (endpoint 0) are returned.
     * This only works AFTER the interview process
     */
    getAssociationGroups(source: AssociationAddress): ReadonlyMap<number, AssociationGroup>;
    /**
     * Returns all association groups that exist on a node and all its endpoints.
     * The returned map uses the endpoint index as keys and its values are maps of group IDs to their definition
     */
    getAllAssociationGroups(nodeId: number): ReadonlyMap<number, ReadonlyMap<number, AssociationGroup>>;
    /**
     * Returns all associations (Multi Channel or normal) that are configured on the root device or an endpoint of a node.
     * If no endpoint is given, the associations of the root device (endpoint 0) are returned.
     */
    getAssociations(source: AssociationAddress): ReadonlyMap<number, readonly AssociationAddress[]>;
    /**
     * Returns all associations (Multi Channel or normal) that are configured on a node and all its endpoints.
     * The returned map uses the source node+endpoint as keys and its values are a map of association group IDs to target node+endpoint.
     */
    getAllAssociations(nodeId: number): ReadonlyObjectKeyMap<AssociationAddress, ReadonlyMap<number, readonly AssociationAddress[]>>;
    /**
     * Checks if a given association is allowed.
     */
    checkAssociation(source: AssociationAddress, group: number, destination: AssociationAddress): AssociationCheckResult;
    /**
     * Adds associations to a node or endpoint.
     *
     * **Note:** This method will throw if:
     * * the source node, endpoint or association group does not exist,
     * * the source node is a ZWLR node and the destination is not the SIS
     * * the destination node is a ZWLR node
     * * the association is not allowed for other reasons. In this case, the error's
     * `context` property will contain an array with all forbidden destinations, each with an added `checkResult` property
     * which contains the reason why the association is forbidden:
     *     ```ts
     *     {
     *         checkResult: AssociationCheckResult;
     *         nodeId: number;
     *         endpoint?: number | undefined;
     *     }[]
     *     ```
     */
    addAssociations(source: AssociationAddress, group: number, destinations: AssociationAddress[]): Promise<void>;
    /**
     * Removes the given associations from a node or endpoint
     */
    removeAssociations(source: AssociationAddress, group: number, destinations: AssociationAddress[]): Promise<void>;
    /**
     * Removes a node from all other nodes' associations
     * WARNING: It is not recommended to await this method
     */
    removeNodeFromAllAssociations(nodeId: number): Promise<void>;
    /**
     * Tests if a node is marked as failed in the controller's memory
     * @param nodeId The id of the node in question
     */
    isFailedNode(nodeId: number): Promise<boolean>;
    /**
     * Removes a failed node from the controller's memory. If the process fails, this will throw an exception with the details why.
     * @param nodeId The id of the node to remove
     */
    removeFailedNode(nodeId: number): Promise<void>;
    private getRemoveFailedNodeTask;
    /**
     * Replace a failed node from the controller's memory. If the process fails, this will throw an exception with the details why.
     * @param nodeId The id of the node to replace
     * @param options Defines the inclusion strategy to use for the replacement node
     */
    replaceFailedNode(nodeId: number, options?: ReplaceNodeOptions): Promise<boolean>;
    /**
     * Returns the task to handle the complete exclusion process
     */
    private getReplaceFailedNodeTask;
    private replaceFailedNodeTask;
    /** Configure the RF region at the Z-Wave API Module */
    setRFRegion(region: RFRegion): Promise<boolean>;
    /** Configure the RF region at the Z-Wave API Module */
    private setRFRegionInternal;
    /** Request the current RF region configured at the Z-Wave API Module */
    getRFRegion(): Promise<RFRegion>;
    /**
     * Query the supported regions of the Z-Wave API Module
     *
     * **Note:** Applications should prefer using {@link getSupportedRFRegions} instead
     */
    querySupportedRFRegions(): Promise<RFRegion[]>;
    /**
     * Query the supported regions of the Z-Wave API Module
     *
     * **Note:** Applications should prefer reading the cached value from {@link supportedRFRegions} instead
     */
    queryRFRegionInfo(region: RFRegion): Promise<{
        region: RFRegion;
        supportsZWave: boolean;
        supportsLongRange: boolean;
        includesRegion?: RFRegion;
    }>;
    /**
     * Returns the RF regions supported by this controller, or `undefined` if the information is not known yet.
     *
     * @param filterSubsets Whether to exclude regions that are subsets of other regions,
     * for example `USA` which is a subset of `USA (Long Range)`
     */
    getSupportedRFRegions(filterSubsets?: boolean): MaybeNotKnown<readonly RFRegion[]>;
    private applyLegalPowerlevelLimits;
    /** Configure the Powerlevel setting of the Z-Wave API */
    setPowerlevel(powerlevel: number, measured0dBm: number): Promise<boolean>;
    /** Request the Powerlevel setting of the Z-Wave API */
    getPowerlevel(): Promise<Pick<SerialAPISetup_GetPowerlevelResponse, "powerlevel" | "measured0dBm">>;
    /** Configure the maximum TX powerlevel for Z-Wave Long Range */
    setMaxLongRangePowerlevel(limit: number): Promise<boolean>;
    /** Request the maximum TX powerlevel setting for Z-Wave Long Range */
    getMaxLongRangePowerlevel(): Promise<number>;
    private applyDesiredPowerlevelMesh;
    private applyDesiredPowerlevelLR;
    /**
     * Configure channel to use for Z-Wave Long Range.
     */
    setLongRangeChannel(channel: LongRangeChannel.A | LongRangeChannel.B | LongRangeChannel.Auto): Promise<boolean>;
    /** Request the channel setting and capabilities for Z-Wave Long Range */
    getLongRangeChannel(): Promise<{
        channel: LongRangeChannel;
        supportsAutoChannelSelection: boolean;
    }>;
    trySetNodeIDType(nodeIdType: NodeIDType): Promise<boolean>;
    /**
     * Instructs a node to (re-)discover its neighbors.
     *
     * **WARNING:** On some controllers, this can cause new SUC return routes to be assigned.
     *
     * @returns `true` if the update was successful and the new neighbors can be retrieved using
     * {@link getNodeNeighbors}. `false` if the update failed.
     */
    discoverNodeNeighbors(nodeId: number): Promise<boolean>;
    /**
     * Returns the known list of neighbors for a node.
     *
     * Throws when the node is a Long Range node.
     */
    getNodeNeighbors(nodeId: number, onlyRepeaters?: boolean): Promise<readonly number[]>;
    /**
     * Returns the known routes the controller will use to communicate with the nodes.
     *
     * This information is dynamically built using TX status reports and may not be accurate at all times.
     * Also, it may not be available immediately after startup or at all if the controller doesn't support this feature.
     *
     * **Note:** To keep information returned by this method updated, use the information contained in each node's `"statistics"` event.
     */
    getKnownLifelineRoutes(): ReadonlyMap<number, LifelineRoutes>;
    /** Request additional information about the controller/Z-Wave chip */
    getSerialApiInitData(): Promise<SerialApiInitData>;
    /** Determines the controller's network role/capabilities */
    getControllerCapabilities(): Promise<ControllerCapabilities>;
    /** Turns the Z-Wave radio on or off */
    toggleRF(enabled: boolean): Promise<boolean>;
    private _nvm;
    /** Provides access to the controller's non-volatile memory */
    get nvm(): NVMAdapter;
    /**
     * **Z-Wave 500 series only**
     *
     * Return the value of the NEWIMAGE marker in the NVM, which is used to signal that a new firmware image is present
     */
    private firmwareUpdateNVMGetNewImage;
    /**
     * **Z-Wave 500 series only**
     *
     * Calculates the CRC-16 for the specified block of data in the NVM
     */
    private firmwareUpdateNVMUpdateCRC16;
    /**
     * **Z-Wave 500 series only**
     *
     * Returns information of the controller's external NVM
     */
    getNVMId(): Promise<NVMId>;
    /**
     * **Z-Wave 500 series only**
     *
     * Reads a byte from the external NVM at the given offset
     */
    externalNVMReadByte(offset: number): Promise<number>;
    /**
     * **Z-Wave 500 series only**
     *
     * Writes a byte to the external NVM at the given offset
     * **WARNING:** This function can write in the full NVM address space and is not offset to start at the application area.
     * Take care not to accidentally overwrite the protocol NVM area!
     *
     * @returns `true` when writing succeeded, `false` otherwise
     */
    externalNVMWriteByte(offset: number, data: number): Promise<boolean>;
    /**
     * **Z-Wave 500 series only**
     *
     * Reads a buffer from the external NVM at the given offset
     */
    externalNVMReadBuffer(offset: number, length: number): Promise<Uint8Array>;
    /**
     * **Z-Wave 700+ series only**
     *
     * Reads a buffer from the external NVM at the given offset
     *
     * **Note:** Prefer {@link externalNVMReadBufferExt} if supported, as that command supports larger NVMs than 64 KiB.
     */
    externalNVMReadBuffer700(offset: number, length: number): Promise<{
        buffer: Uint8Array;
        endOfFile: boolean;
    }>;
    /**
     * **Z-Wave 700+ series only**
     *
     * Reads a buffer from the external NVM at the given offset
     *
     * **Note:** If supported, this command should be preferred over {@link externalNVMReadBuffer700} as it supports larger NVMs than 64 KiB.
     */
    externalNVMReadBufferExt(offset: number, length: number): Promise<{
        buffer: Uint8Array;
        endOfFile: boolean;
    }>;
    /**
     * **Z-Wave 500 series only**
     *
     * Writes a buffer to the external NVM at the given offset
     * **WARNING:** This function can write in the full NVM address space and is not offset to start at the application area.
     * Take care not to accidentally overwrite the protocol NVM area!
     *
     * @returns `true` when writing succeeded, `false` otherwise
     */
    externalNVMWriteBuffer(offset: number, buffer: Uint8Array): Promise<boolean>;
    /**
     * **Z-Wave 700+ series only**
     *
     * Writes a buffer to the external NVM at the given offset
     *
     * **Note:** Prefer {@link externalNVMWriteBufferExt} if supported, as that command supports larger NVMs than 64 KiB.
     *
     * **WARNING:** This function can write in the full NVM address space and is not offset to start at the application area.
     * Take care not to accidentally overwrite the protocol NVM area!
     */
    externalNVMWriteBuffer700(offset: number, buffer: Uint8Array): Promise<{
        endOfFile: boolean;
    }>;
    /**
     * **Z-Wave 700+ series only**
     *
     * Writes a buffer to the external NVM at the given offset
     *
     * **Note:** If supported, this command should be preferred over {@link externalNVMWriteBuffer700} as it supports larger NVMs than 64 KiB.
     *
     * **WARNING:** This function can write in the full NVM address space and is not offset to start at the application area.
     * Take care not to accidentally overwrite the protocol NVM area!
     */
    externalNVMWriteBufferExt(offset: number, buffer: Uint8Array): Promise<{
        endOfFile: boolean;
    }>;
    /**
     * **Z-Wave 700+ series only**
     *
     * Opens the controller's external NVM for reading/writing and returns the NVM size
     *
     * **Note:** Prefer {@link externalNVMOpenExt} if supported, as that command supports larger NVMs than 64 KiB.
     */
    externalNVMOpen(): Promise<number>;
    /**
     * **Z-Wave 700+ series only**
     *
     * Opens the controller's external NVM for reading/writing and returns the NVM size and supported operations.
     *
     * **Note:** If supported, this command should be preferred over {@link externalNVMOpen} as it supports larger NVMs than 64 KiB.
     */
    externalNVMOpenExt(): Promise<{
        size: number;
        supportedOperations: ExtendedNVMOperationsCommand[];
    }>;
    /**
     * **Z-Wave 700+ series only**
     *
     * Closes the controller's external NVM
     *
     * **Note:** Prefer {@link externalNVMCloseExt} if supported, as that command supports larger NVMs than 64 KiB.
     */
    externalNVMClose(): Promise<void>;
    /**
     * **Z-Wave 700+ series only**
     *
     * Closes the controller's external NVM
     *
     * **Note:** If supported, this command should be preferred over {@link externalNVMClose} as it supports larger NVMs than 64 KiB.
     */
    externalNVMCloseExt(): Promise<void>;
    /**
     * Creates a backup of the NVM and returns the raw data as a Buffer. The Z-Wave radio is turned off/on automatically.
     * @param onProgress Can be used to monitor the progress of the operation, which may take several seconds up to a few minutes depending on the NVM size
     * @returns The raw NVM buffer
     */
    backupNVMRaw(onProgress?: (bytesRead: number, total: number) => void): Promise<Uint8Array>;
    private backupNVMRaw500;
    private backupNVMRaw700;
    /**
     * Restores an NVM backup that was created with `backupNVMRaw`. The Z-Wave radio is turned off/on automatically.
     * If the given buffer is in a different NVM format, it is converted automatically. If a conversion is required but not supported, the operation will be aborted.
     *
     * **WARNING:** If both the source and target NVM use an an unsupported format, they will NOT be checked for compatibility!
     *
     * **WARNING:** A failure during this process may brick your controller. Use at your own risk!
     *
     * @param nvmData The NVM backup to be restored
     * @param convertProgress Can be used to monitor the progress of the NVM conversion, which may take several seconds up to a few minutes depending on the NVM size
     * @param restoreProgress Can be used to monitor the progress of the restore operation, which may take several seconds up to a few minutes depending on the NVM size
     * @param migrateOptions Influence which data should be preserved during a migration
     */
    restoreNVM(nvmData: Uint8Array, convertProgress?: (bytesRead: number, total: number) => void, restoreProgress?: (bytesWritten: number, total: number) => void, migrateOptions?: MigrateNVMOptions): Promise<void>;
    /**
     * Restores an NVM backup that was created with `backupNVMRaw`. The Z-Wave radio is turned off/on automatically.
     *
     * **WARNING:** The given buffer is NOT checked for compatibility with the current stick. To have Z-Wave JS do that, use the {@link restoreNVM} method instead.
     *
     * **WARNING:** A failure during this process may brick your controller. Use at your own risk!
     * @param nvmData The raw NVM backup to be restored
     * @param onProgress Can be used to monitor the progress of the operation, which may take several seconds up to a few minutes depending on the NVM size
     */
    restoreNVMRaw(nvmData: Uint8Array, onProgress?: (bytesWritten: number, total: number) => void): Promise<void>;
    private restoreNVMRaw500;
    private restoreNVMRaw700;
    /**
     * Request the most recent background RSSI levels detected by the controller.
     *
     * **Note:** This only returns useful values if something was transmitted recently.
     */
    getBackgroundRSSI(): Promise<{
        rssiChannel0: RSSI;
        rssiChannel1: RSSI;
        rssiChannel2?: RSSI;
        rssiChannel3?: RSSI;
    }>;
    /**
     * Returns whether an OTA firmware update is in progress for any node.
     */
    isAnyOTAFirmwareUpdateInProgress(): boolean;
    /**
     * Retrieves the available firmware updates for the given node from the Z-Wave JS firmware update service.
     *
     * **Note:** Sleeping nodes need to be woken up for this to work. This method will throw when called for a sleeping node
     * which did not wake up within a minute.
     *
     * **Note:** This requires an API key to be set in the driver options, or passed .
     */
    getAvailableFirmwareUpdates(nodeId: number, options?: GetFirmwareUpdatesOptions): Promise<FirmwareUpdateInfo[]>;
    /** Ensures that the device ID used to request a firmware update matches the device the firmware update is for */
    private ensureFirmwareDeviceIdMatches;
    /**
     * Downloads the desired firmware update(s) from the Z-Wave JS firmware update service and updates the firmware of the given node.
     * @param updateInfo The desired entry from the updates array that was returned by {@link getAvailableFirmwareUpdates}.
     * Before applying the update, Z-Wave JS will check whether the device IDs, firmware version and region match.
     *
     * The return value indicates whether the update was successful.
     * **WARNING:** This method will throw instead of returning `false` if invalid arguments are passed or downloading files or starting an update fails.
     */
    firmwareUpdateOTA(nodeId: number, updateInfo: FirmwareUpdateInfo, options?: FirmwareUpdateOptions): Promise<FirmwareUpdateResult>;
    private _currentLearnMode;
    private _joinNetworkOptions;
    beginJoiningNetwork(options?: JoinNetworkOptions): Promise<JoinNetworkResult>;
    stopJoiningNetwork(): Promise<boolean>;
    beginLeavingNetwork(): Promise<LeaveNetworkResult>;
    stopLeavingNetwork(): Promise<boolean>;
    /**
     * Is called when a RemoveNode request is received from the controller.
     * Handles and controls the exclusion process.
     */
    private handleLearnModeCallback;
    private expectSecurityBootstrapS0;
    private expectSecurityBootstrapS2;
    private afterJoiningNetwork;
    private _proprietary;
    /** Provides access to hardware-specific Serial API functionality */
    get proprietary(): ControllerProprietary;
    destroy(): void;
}
export {};
//# sourceMappingURL=Controller.d.ts.map