export interface JsonlDBOptions<V> {
    /**
     * Whether errors reading the db file (e.g. invalid JSON) should silently be ignored.
     * **Warning:** This may result in inconsistent data!
     */
    ignoreReadErrors?: boolean;
    /**
     * An optional reviver function (similar to JSON.parse) to transform parsed values before they are accessible in the database.
     * If this function is defined, it must always return a value.
     */
    reviver?: (key: string, value: any) => V;
    /**
     * An optional serializer function (similar to JSON.serialize) to transform values before they are written to the database file.
     * If this function is defined, it must always return a value.
     */
    serializer?: (key: string, value: V) => any;
    /** Whether timestamps should be recorded when setting values. Default: false */
    enableTimestamps?: boolean;
    /**
     * Configure when the DB should be automatically compressed.
     * If multiple conditions are configured, the DB is compressed when any of them are fulfilled
     */
    autoCompress?: Partial<{
        /**
         * Compress when uncompressedSize >= size * sizeFactor. Default: +Infinity
         */
        sizeFactor: number;
        /**
         * Configure the minimum size necessary for auto-compression based on size. Default: 0
         */
        sizeFactorMinimumSize: number;
        /**
         * Compress after a certain time has passed. Default: never
         */
        intervalMs: number;
        /**
         * Configure the minimum count of changes for auto-compression based on time. Default: 1
         */
        intervalMinChanges: number;
        /** Compress when closing the DB. Default: false */
        onClose: boolean;
        /** Compress after opening the DB. Default: false */
        onOpen: boolean;
    }>;
    /**
     * Can be used to throttle write accesses to the filesystem. By default,
     * every change is immediately written to the FS
     */
    throttleFS?: {
        /**
         * Minimum wait time between two consecutive write accesses. Default: 0
         */
        intervalMs: number;
        /**
         * Maximum commands to be buffered before forcing a write access. Default: +Infinity
         */
        maxBufferedCommands?: number;
    };
    /** Configure settings related to the lockfile */
    lockfile?: Partial<{
        /**
         * Override in which directory the lockfile is created.
         * Defaults to the directory in which the DB file is located.
         */
        directory?: string;
        /** Duration after which the lock is considered stale. Minimum: 5000, Default: 10000 */
        staleMs?: number;
        /** The interval in which the lockfile's `mtime` will be updated. Range: [1000...staleMs/2]. Default: staleMs/2  */
        updateMs?: number;
        /**
         * How often to retry acquiring a lock before giving up. The retries progressively wait longer with an exponential backoff strategy.
         * Range: [0...10]. Default: 0
         */
        retries?: number;
        /** The start interval used for retries. Default: updateMs/2 */
        retryMinTimeoutMs?: number;
    }>;
    /**
     * @deprecated Use lockfile.directory instead.
     *
     * Override in which directory the lockfile is created.
     * Defaults to the directory in which the DB file is located.
     */
    lockfileDirectory?: string;
}
export declare class JsonlDB<V = unknown> {
    constructor(filename: string, options?: JsonlDBOptions<V>);
    private validateOptions;
    readonly filename: string;
    readonly dumpFilename: string;
    readonly backupFilename: string;
    private readonly lockfileName;
    private options;
    private _db;
    private _timestamps;
    forEach: Map<string, V>["forEach"];
    get: Map<string, V>["get"];
    has: Map<string, V>["has"];
    [Symbol.iterator]: () => IterableIterator<[string, V]>;
    entries: Map<string, V>["entries"];
    keys: Map<string, V>["keys"];
    values: Map<string, V>["values"];
    getTimestamp(key: string): number | undefined;
    get size(): number;
    private _uncompressedSize;
    /** Returns the line count of the appendonly file, excluding empty lines */
    get uncompressedSize(): number;
    private _changesSinceLastCompress;
    private _compressBySizeThreshold;
    private _compressBySizeNeeded;
    private _compressIntervalMinChangesExceeded;
    private _writeIntervalElapsed;
    private _journalFlushable;
    private updateCompressBySizeThreshold;
    private triggerJournalFlush;
    private _isOpen;
    get isOpen(): boolean;
    private _persistencePromise;
    private _persistenceTasks;
    private _persistenceTaskSignal;
    private _journal;
    private _handle;
    private drainJournal;
    private pushPersistenceTask;
    private _openPromise;
    open(): Promise<void>;
    /**
     * Makes sure that there are no remains of a previous broken compress attempt and restores
     * a DB backup if it exists.
     */
    private tryRecoverDBFiles;
    /** Reads a line and extracts the key without doing a full-blown JSON.parse() */
    private parseKey;
    /** Parses a line and updates the internal DB correspondingly */
    private parseLine;
    clear(): void;
    delete(key: string): boolean;
    set(key: string, value: V, updateTimestamp?: boolean): this;
    importJSON(json: Record<string, V>): void;
    toJSON(): Record<string, V>;
    private entryToLine;
    private makeLazyClear;
    private makeLazyDelete;
    private makeLazyWrite;
    /**
     * Saves a compressed copy of the DB into the given path.
     *
     * **WARNING:** This MUST be called from {@link persistenceThread}!
     * @param targetFilename Where the compressed copy should be written. Default: `<filename>.dump`
     * @param drainJournal Whether the journal should be drained when writing the compressed copy or simply cloned.
     */
    private dumpInternal;
    /**
     * Saves a compressed copy of the DB into the given path.
     * @param targetFilename Where the compressed copy should be written. Default: `<filename>.dump`
     */
    dump(targetFilename?: string): Promise<void>;
    private exceededMaxBufferedCommands;
    private needToCompressBySize;
    private needToCompressByTime;
    private persistenceThread;
    /** Writes the given journal to the given file descriptor. Returns the new file descriptor if the file was re-opened during the process */
    private writeJournalToFile;
    /**
     * Compresses the db by dumping it and overwriting the aof file.
     *
     * **WARNING:** This MUST be called from {@link persistenceThread}!
     */
    private doCompress;
    /** Compresses the db by dumping it and overwriting the aof file. */
    compress(): Promise<void>;
    /** Compresses the db by dumping it and overwriting the aof file. */
    private compressInternal;
    /** Closes the DB and waits for all data to be written */
    close(): Promise<void>;
}
//# sourceMappingURL=db.d.ts.map