"use strict";
var __defProp = Object.defineProperty;
var __getOwnPropDesc = Object.getOwnPropertyDescriptor;
var __getOwnPropNames = Object.getOwnPropertyNames;
var __hasOwnProp = Object.prototype.hasOwnProperty;
var __name = (target, value) => __defProp(target, "name", { value, configurable: true });
var __export = (target, all) => {
  for (var name in all)
    __defProp(target, name, { get: all[name], enumerable: true });
};
var __copyProps = (to, from, except, desc) => {
  if (from && typeof from === "object" || typeof from === "function") {
    for (let key of __getOwnPropNames(from))
      if (!__hasOwnProp.call(to, key) && key !== except)
        __defProp(to, key, { get: () => from[key], enumerable: !(desc = __getOwnPropDesc(from, key)) || desc.enumerable });
  }
  return to;
};
var __toCommonJS = (mod) => __copyProps(__defProp({}, "__esModule", { value: true }), mod);
var ValueDB_exports = {};
__export(ValueDB_exports, {
  ValueDB: () => ValueDB,
  assertValueID: () => assertValueID,
  dbKeyToValueIdFast: () => dbKeyToValueIdFast,
  indexDBsByNode: () => indexDBsByNode,
  isValueID: () => isValueID,
  normalizeValueID: () => normalizeValueID,
  valueEquals: () => valueEquals,
  valueIdToString: () => valueIdToString
});
module.exports = __toCommonJS(ValueDB_exports);
var import_shared = require("@zwave-js/shared");
var import_typeguards = require("alcalzone-shared/typeguards");
var import_ZWaveError = require("../error/ZWaveError.js");
function isValueID(param) {
  if (typeof param.commandClass !== "number")
    return false;
  if (typeof param.property !== "number" && typeof param.property !== "string") {
    return false;
  }
  if (param.propertyKey != void 0 && typeof param.propertyKey !== "number" && typeof param.propertyKey !== "string") {
    return false;
  }
  if (param.endpoint != void 0 && typeof param.endpoint !== "number") {
    return false;
  }
  return true;
}
__name(isValueID, "isValueID");
function assertValueID(param) {
  if (!isValueID(param)) {
    throw new import_ZWaveError.ZWaveError(`Invalid ValueID passed!`, import_ZWaveError.ZWaveErrorCodes.Argument_Invalid);
  }
}
__name(assertValueID, "assertValueID");
function normalizeValueID(valueID) {
  assertValueID(valueID);
  const { commandClass, endpoint, property, propertyKey } = valueID;
  const normalized = {
    commandClass,
    endpoint: endpoint ?? 0,
    property
  };
  if (propertyKey != void 0)
    normalized.propertyKey = propertyKey;
  return normalized;
}
__name(normalizeValueID, "normalizeValueID");
function valueIdToString(valueID) {
  return JSON.stringify(normalizeValueID(valueID));
}
__name(valueIdToString, "valueIdToString");
function valueEquals(a, b) {
  switch (typeof a) {
    case "bigint":
    case "boolean":
    case "number":
    case "string":
    case "undefined":
      return a === b;
    case "function":
    case "symbol":
      return false;
  }
  if (a === null)
    return b === null;
  if ((0, import_shared.isUint8Array)(a)) {
    return (0, import_shared.isUint8Array)(b) && (0, import_shared.areUint8ArraysEqual)(a, b);
  }
  if ((0, import_typeguards.isArray)(a)) {
    return (0, import_typeguards.isArray)(b) && a.length === b.length && a.every((v, i) => valueEquals(v, b[i]));
  }
  if ((0, import_typeguards.isObject)(a)) {
    if (!(0, import_typeguards.isObject)(b))
      return false;
    const allKeys = /* @__PURE__ */ new Set([...Object.keys(a), ...Object.keys(b)]);
    return [...allKeys].every((k) => valueEquals(a[k], b[k]));
  }
  return false;
}
__name(valueEquals, "valueEquals");
class ValueDB extends import_shared.TypedEventTarget {
  static {
    __name(this, "ValueDB");
  }
  // This is a wrapper around the driver's on-disk value and metadata key value stores
  /**
   * @param nodeId The ID of the node this Value DB belongs to
   * @param valueDB The DB instance which stores values
   * @param metadataDB The DB instance which stores metadata
   * @param ownKeys An optional pre-created index of this ValueDB's own keys
   */
  constructor(nodeId, valueDB, metadataDB, ownKeys) {
    super();
    this.nodeId = nodeId;
    this._db = valueDB;
    this._metadata = metadataDB;
    this._index = ownKeys ?? this.buildIndex();
  }
  nodeId;
  _db;
  _metadata;
  _index;
  buildIndex() {
    const ret = /* @__PURE__ */ new Set();
    for (const key of this._db.keys()) {
      if (compareDBKeyFast(key, this.nodeId))
        ret.add(key);
    }
    for (const key of this._metadata.keys()) {
      if (!ret.has(key) && compareDBKeyFast(key, this.nodeId)) {
        ret.add(key);
      }
    }
    return ret;
  }
  valueIdToDBKey(valueID) {
    return JSON.stringify({
      nodeId: this.nodeId,
      ...normalizeValueID(valueID)
    });
  }
  dbKeyToValueId(key) {
    try {
      return dbKeyToValueIdFast(key);
    } catch {
    }
    try {
      return JSON.parse(key);
    } catch {
    }
  }
  /**
   * Stores a value for a given value id
   */
  setValue(valueId, value, options = {}) {
    let dbKey;
    try {
      dbKey = this.valueIdToDBKey(valueId);
    } catch (e) {
      if ((0, import_ZWaveError.isZWaveError)(e) && e.code === import_ZWaveError.ZWaveErrorCodes.Argument_Invalid && options.noThrow === true) {
        return;
      }
      throw e;
    }
    if (options.stateful !== false) {
      const cbArg = {
        ...valueId,
        newValue: value
      };
      let event;
      if (this._db.has(dbKey)) {
        event = "value updated";
        cbArg.prevValue = this._db.get(dbKey);
        if (options.source) {
          cbArg.source = options.source;
        }
      } else {
        event = "value added";
      }
      this._index.add(dbKey);
      this._db.set(dbKey, value, options.updateTimestamp !== false);
      if (valueId.commandClass >= 0 && options.noEvent !== true) {
        this.emit(event, cbArg);
      }
    } else if (valueId.commandClass >= 0) {
      this.emit("value notification", {
        ...valueId,
        value
      });
    }
  }
  /**
   * Removes a value for a given value id
   */
  removeValue(valueId, options = {}) {
    const dbKey = this.valueIdToDBKey(valueId);
    if (!this._metadata.has(dbKey)) {
      this._index.delete(dbKey);
    }
    if (this._db.has(dbKey)) {
      const prevValue = this._db.get(dbKey);
      this._db.delete(dbKey);
      if (valueId.commandClass >= 0 && options.noEvent !== true) {
        const cbArg = {
          ...valueId,
          prevValue
        };
        this.emit("value removed", cbArg);
      }
      return true;
    }
    return false;
  }
  /**
   * Retrieves a value for a given value id
   */
  getValue(valueId) {
    const key = this.valueIdToDBKey(valueId);
    return this._db.get(key);
  }
  /**
   * Checks if a value for a given value id exists in this ValueDB
   */
  hasValue(valueId) {
    const key = this.valueIdToDBKey(valueId);
    return this._db.has(key);
  }
  /** Returns all values whose id matches the given predicate */
  findValues(predicate) {
    const ret = [];
    for (const key of this._index) {
      if (!this._db.has(key))
        continue;
      const vid = this.dbKeyToValueId(key);
      if (!vid) {
        this.dropBrokenEntry(key);
        continue;
      }
      const { nodeId, ...valueId } = vid;
      if (predicate(valueId)) {
        ret.push({ ...valueId, value: this._db.get(key) });
      }
    }
    return ret;
  }
  /** Returns all values that are stored for a given CC */
  getValues(forCC) {
    const ret = [];
    for (const key of this._index) {
      if (compareDBKeyFast(key, this.nodeId, { commandClass: forCC }) && this._db.has(key)) {
        const vid = this.dbKeyToValueId(key);
        if (!vid) {
          this.dropBrokenEntry(key);
          continue;
        }
        const { nodeId, ...valueId } = vid;
        const value = this._db.get(key);
        ret.push({ ...valueId, value });
      }
    }
    return ret;
  }
  /**
   * Returns when the given value id was last updated
   */
  getTimestamp(valueId) {
    const key = this.valueIdToDBKey(valueId);
    return this._db.getTimestamp(key);
  }
  /** Clears all values from the value DB */
  clear(options = {}) {
    for (const key of this._index) {
      const vid = this.dbKeyToValueId(key);
      if (!vid)
        continue;
      const { nodeId, ...valueId } = vid;
      if (this._db.has(key)) {
        const prevValue = this._db.get(key);
        this._db.delete(key);
        if (valueId.commandClass >= 0 && options.noEvent !== true) {
          const cbArg = {
            ...valueId,
            prevValue
          };
          this.emit("value removed", cbArg);
        }
      }
      if (this._metadata.has(key)) {
        this._metadata.delete(key);
        if (valueId.commandClass >= 0 && options.noEvent !== true) {
          const cbArg = {
            ...valueId,
            metadata: void 0
          };
          this.emit("metadata updated", cbArg);
        }
      }
    }
    this._index.clear();
  }
  dropBrokenEntry(key) {
    this._db.delete(key);
    this._metadata.delete(key);
    this._index.delete(key);
  }
  /**
   * Stores metadata for a given value id
   */
  setMetadata(valueId, metadata, options = {}) {
    let dbKey;
    try {
      dbKey = this.valueIdToDBKey(valueId);
    } catch (e) {
      if ((0, import_ZWaveError.isZWaveError)(e) && e.code === import_ZWaveError.ZWaveErrorCodes.Argument_Invalid && options.noThrow === true) {
        return;
      }
      throw e;
    }
    if (metadata) {
      this._index.add(dbKey);
      this._metadata.set(dbKey, metadata);
    } else {
      if (!this._db.has(dbKey)) {
        this._index.delete(dbKey);
      }
      this._metadata.delete(dbKey);
    }
    const cbArg = {
      ...valueId,
      metadata
    };
    if (valueId.commandClass >= 0 && options.noEvent !== true) {
      this.emit("metadata updated", cbArg);
    }
  }
  /**
   * Checks if metadata for a given value id exists in this ValueDB
   */
  hasMetadata(valueId) {
    const key = this.valueIdToDBKey(valueId);
    return this._metadata.has(key);
  }
  /**
   * Retrieves metadata for a given value id
   */
  getMetadata(valueId) {
    const key = this.valueIdToDBKey(valueId);
    return this._metadata.get(key);
  }
  /** Returns all metadata that is stored for a given CC */
  getAllMetadata(forCC) {
    const ret = [];
    for (const key of this._index) {
      if (compareDBKeyFast(key, this.nodeId, { commandClass: forCC }) && this._metadata.has(key)) {
        const vid = this.dbKeyToValueId(key);
        if (!vid) {
          this.dropBrokenEntry(key);
          continue;
        }
        const { nodeId, ...valueId } = vid;
        const metadata = this._metadata.get(key);
        ret.push({ ...valueId, metadata });
      }
    }
    return ret;
  }
  /** Returns all values whose id matches the given predicate */
  findMetadata(predicate) {
    const ret = [];
    for (const key of this._index) {
      if (!this._metadata.has(key))
        continue;
      const vid = this.dbKeyToValueId(key);
      if (!vid) {
        this.dropBrokenEntry(key);
        continue;
      }
      const { nodeId, ...valueId } = vid;
      if (predicate(valueId)) {
        ret.push({ ...valueId, metadata: this._metadata.get(key) });
      }
    }
    return ret;
  }
}
function dbKeyToValueIdFast(key) {
  let start = 10;
  if (key.charCodeAt(start - 1) !== 58) {
    console.error(key.slice(start - 1));
    throw new Error("Invalid input format!");
  }
  let end = start + 1;
  const len = key.length;
  while (end < len && key.charCodeAt(end) !== 44)
    end++;
  const nodeId = parseInt(key.slice(start, end));
  start = end + 16;
  if (key.charCodeAt(start - 1) !== 58) {
    throw new Error("Invalid input format!");
  }
  end = start + 1;
  while (end < len && key.charCodeAt(end) !== 44)
    end++;
  const commandClass = parseInt(key.slice(start, end));
  start = end + 12;
  if (key.charCodeAt(start - 1) !== 58) {
    throw new Error("Invalid input format!");
  }
  end = start + 1;
  while (end < len && key.charCodeAt(end) !== 44)
    end++;
  const endpoint = parseInt(key.slice(start, end));
  start = end + 12;
  if (key.charCodeAt(start - 1) !== 58) {
    throw new Error("Invalid input format!");
  }
  let property;
  if (key.charCodeAt(start) === 34) {
    start++;
    end = start + 1;
    while (end < len && key.charCodeAt(end) !== 34)
      end++;
    property = key.slice(start, end);
    end++;
  } else {
    end = start + 1;
    while (end < len && key.charCodeAt(end) !== 44 && key.charCodeAt(end) !== 125) {
      end++;
    }
    property = parseInt(key.slice(start, end));
  }
  if (key.charCodeAt(end) !== 125) {
    let propertyKey;
    start = end + 15;
    if (key.charCodeAt(start - 1) !== 58) {
      throw new Error("Invalid input format!");
    }
    if (key.charCodeAt(start) === 34) {
      start++;
      end = start + 1;
      while (end < len && key.charCodeAt(end) !== 34)
        end++;
      propertyKey = key.slice(start, end);
      end++;
    } else {
      end = start + 1;
      while (end < len && key.charCodeAt(end) !== 44 && key.charCodeAt(end) !== 125) {
        end++;
      }
      propertyKey = parseInt(key.slice(start, end));
    }
    return {
      nodeId,
      commandClass,
      endpoint,
      property,
      propertyKey
    };
  } else {
    return {
      nodeId,
      commandClass,
      endpoint,
      property
    };
  }
}
__name(dbKeyToValueIdFast, "dbKeyToValueIdFast");
function compareDBKeyFast(key, nodeId, valueId) {
  if (-1 === key.indexOf(`{"nodeId":${nodeId},`))
    return false;
  if (!valueId)
    return true;
  if ("commandClass" in valueId) {
    if (-1 === key.indexOf(`,"commandClass":${valueId.commandClass},`)) {
      return false;
    }
  }
  if ("endpoint" in valueId) {
    if (-1 === key.indexOf(`,"endpoint":${valueId.endpoint},`)) {
      return false;
    }
  }
  if (typeof valueId.property === "string") {
    if (-1 === key.indexOf(`,"property":"${valueId.property}"`)) {
      return false;
    }
  } else if (typeof valueId.property === "number") {
    if (-1 === key.indexOf(`,"property":${valueId.property}`))
      return false;
  }
  if (typeof valueId.propertyKey === "string") {
    if (-1 === key.indexOf(`,"propertyKey":"${valueId.propertyKey}"`)) {
      return false;
    }
  } else if (typeof valueId.propertyKey === "number") {
    if (-1 === key.indexOf(`,"propertyKey":${valueId.propertyKey}`)) {
      return false;
    }
  }
  return true;
}
__name(compareDBKeyFast, "compareDBKeyFast");
function indexDBsByNode(databases) {
  const indexes = /* @__PURE__ */ new Map();
  for (const db of databases) {
    for (const key of db.keys()) {
      const nodeId = extractNodeIdFromDBKeyFast(key);
      if (nodeId == void 0)
        continue;
      if (!indexes.has(nodeId)) {
        indexes.set(nodeId, /* @__PURE__ */ new Set());
      }
      indexes.get(nodeId).add(key);
    }
  }
  return indexes;
}
__name(indexDBsByNode, "indexDBsByNode");
function extractNodeIdFromDBKeyFast(key) {
  const start = 10;
  if (key.charCodeAt(start - 1) !== 58) {
    return void 0;
  }
  let end = start + 1;
  const len = key.length;
  while (end < len && key.charCodeAt(end) !== 44)
    end++;
  return parseInt(key.slice(start, end));
}
__name(extractNodeIdFromDBKeyFast, "extractNodeIdFromDBKeyFast");
// Annotate the CommonJS export names for ESM import in node:
0 && (module.exports = {
  ValueDB,
  assertValueID,
  dbKeyToValueIdFast,
  indexDBsByNode,
  isValueID,
  normalizeValueID,
  valueEquals,
  valueIdToString
});
//# sourceMappingURL=ValueDB.js.map
