"use strict";
var __defProp = Object.defineProperty;
var __getOwnPropDesc = Object.getOwnPropertyDescriptor;
var __getOwnPropNames = Object.getOwnPropertyNames;
var __hasOwnProp = Object.prototype.hasOwnProperty;
var __name = (target, value) => __defProp(target, "name", { value, configurable: true });
var __export = (target, all) => {
  for (var name in all)
    __defProp(target, name, { get: all[name], enumerable: true });
};
var __copyProps = (to, from, except, desc) => {
  if (from && typeof from === "object" || typeof from === "function") {
    for (let key of __getOwnPropNames(from))
      if (!__hasOwnProp.call(to, key) && key !== except)
        __defProp(to, key, { get: () => from[key], enumerable: !(desc = __getOwnPropDesc(from, key)) || desc.enumerable });
  }
  return to;
};
var __toCommonJS = (mod) => __copyProps(__defProp({}, "__esModule", { value: true }), mod);
var NVM500_exports = {};
__export(NVM500_exports, {
  NVM500: () => NVM500
});
module.exports = __toCommonJS(NVM500_exports);
var import_core = require("@zwave-js/core");
var import_shared = require("@zwave-js/shared");
var import_definitions = require("./common/definitions.js");
var import_routeCache = require("./common/routeCache.js");
var import_sucUpdateEntry = require("./common/sucUpdateEntry.js");
var import_utils = require("./common/utils.js");
var import_EntryParsers = require("./nvm500/EntryParsers.js");
var import_impls = require("./nvm500/impls/index.js");
var import_shared2 = require("./nvm500/shared.js");
class NVM500 {
  static {
    __name(this, "NVM500");
  }
  constructor(io) {
    this._io = io;
  }
  _io;
  _access = import_definitions.NVMAccess.None;
  _info;
  get info() {
    return this._info;
  }
  async ensureReadable() {
    if (this._access === import_definitions.NVMAccess.Read || this._access === import_definitions.NVMAccess.ReadWrite) {
      return;
    }
    if (this._access === import_definitions.NVMAccess.Write) {
      await this._io.close();
    }
    this._access = await this._io.open(import_definitions.NVMAccess.Read);
  }
  async ensureWritable() {
    if (this._access === import_definitions.NVMAccess.Write || this._access === import_definitions.NVMAccess.ReadWrite) {
      return;
    }
    if (this._access === import_definitions.NVMAccess.Read) {
      await this._io.close();
    }
    this._access = await this._io.open(import_definitions.NVMAccess.Write);
  }
  async init() {
    await this.ensureReadable();
    for (const impl of import_impls.nvm500Impls) {
      try {
        const info = await this.resolveLayout(impl);
        if (await this.isLayoutValid(info, impl.protocolVersions)) {
          this._info = info;
        }
        break;
      } catch {
        continue;
      }
    }
    if (!this._info) {
      throw new import_core.ZWaveError("Did not find a matching NVM 500 parser implementation! Make sure that the NVM data belongs to a controller with Z-Wave SDK 6.61 or higher.", import_core.ZWaveErrorCodes.NVM_NotSupported);
    }
    return this._info;
  }
  async resolveLayout(impl) {
    const resolvedLayout = /* @__PURE__ */ new Map();
    let nvmDescriptor;
    const moduleDescriptors = /* @__PURE__ */ new Map();
    let offset = 0;
    let moduleStart = -1;
    let moduleSize = -1;
    const nvmEnd = await (0, import_utils.nvmReadUInt16BE)(this._io, 0);
    for (const entry of impl.layout) {
      const size = entry.size ?? import_shared2.NVMEntrySizes[entry.type];
      if (entry.type === import_shared2.NVMEntryType.NVMModuleSize) {
        if (moduleStart !== -1) {
          offset = moduleStart + moduleSize;
        }
        moduleStart = offset;
        moduleSize = await (0, import_utils.nvmReadUInt16BE)(this._io, offset);
      } else if (entry.type === import_shared2.NVMEntryType.NVMModuleDescriptor) {
        offset = moduleStart + moduleSize - size;
      }
      if (entry.offset != void 0 && entry.offset !== offset) {
        throw new import_core.ZWaveError(`${entry.name} is at wrong location in NVM buffer!`, import_core.ZWaveErrorCodes.NVM_InvalidFormat);
      }
      const resolvedEntry = {
        ...entry,
        offset,
        size
      };
      if (entry.type === import_shared2.NVMEntryType.NVMDescriptor) {
        const entryData = await this.readRawEntry(resolvedEntry);
        nvmDescriptor = (0, import_EntryParsers.parseNVMDescriptor)(entryData[0]);
      } else if (entry.type === import_shared2.NVMEntryType.NVMModuleDescriptor) {
        const entryData = await this.readRawEntry(resolvedEntry);
        const descriptor = (0, import_EntryParsers.parseNVMModuleDescriptor)(entryData[0]);
        if (descriptor.size !== moduleSize) {
          throw new import_core.ZWaveError("NVM module descriptor size does not match module size!", import_core.ZWaveErrorCodes.NVM_InvalidFormat);
        }
        moduleDescriptors.set(entry.name, descriptor);
      }
      resolvedLayout.set(entry.name, resolvedEntry);
      offset += size * entry.count;
      if (offset >= nvmEnd)
        break;
    }
    if (!nvmDescriptor) {
      throw new import_core.ZWaveError("NVM descriptor not found in NVM!", import_core.ZWaveErrorCodes.NVM_InvalidFormat);
    }
    return {
      layout: resolvedLayout,
      library: impl.library,
      moduleDescriptors,
      nvmDescriptor
    };
  }
  async isLayoutValid(info, protocolVersions) {
    const eeoffset_magic_entry = info.layout.get("EEOFFSET_MAGIC_far");
    if (!eeoffset_magic_entry)
      return false;
    const eeoffset_magic = (await this.readEntry(eeoffset_magic_entry))[0];
    const configuration_valid_0_entry = info.layout.get("NVM_CONFIGURATION_VALID_far");
    if (!configuration_valid_0_entry)
      return false;
    const configuration_valid_0 = (await this.readEntry(configuration_valid_0_entry))[0];
    const configuration_valid_1_entry = info.layout.get("NVM_CONFIGURATION_REALLYVALID_far");
    if (!configuration_valid_1_entry)
      return false;
    const configuration_valid_1 = (await this.readEntry(configuration_valid_1_entry))[0];
    const routecache_valid_entry = info.layout.get("EX_NVM_ROUTECACHE_MAGIC_far");
    if (!routecache_valid_entry)
      return false;
    const routecache_valid = (await this.readEntry(routecache_valid_entry))[0];
    const endMarker_entry = info.layout.get("nvmModuleSizeEndMarker");
    if (!endMarker_entry)
      return false;
    const endMarker = (await this.readEntry(endMarker_entry))[0];
    return eeoffset_magic === import_shared2.MAGIC_VALUE && configuration_valid_0 === import_shared2.CONFIGURATION_VALID_0 && configuration_valid_1 === import_shared2.CONFIGURATION_VALID_1 && routecache_valid === import_shared2.ROUTECACHE_VALID && protocolVersions.includes(info.nvmDescriptor.protocolVersion) && endMarker === 0;
  }
  async has(property) {
    this._info ??= await this.init();
    return this._info.layout.has(property);
  }
  async readSingleRawEntry(entry, index) {
    if (index >= entry.count) {
      throw new import_core.ZWaveError(`Index out of range. Tried to read entry ${index} of ${entry.count}.`, import_core.ZWaveErrorCodes.Argument_Invalid);
    }
    return (0, import_utils.nvmReadBuffer)(this._io, entry.offset + index * entry.size, entry.size);
  }
  async readRawEntry(entry) {
    const ret = [];
    const nvmData = await (0, import_utils.nvmReadBuffer)(this._io, entry.offset, entry.count * entry.size);
    for (let i = 0; i < entry.count; i++) {
      ret.push(nvmData.subarray(i * entry.size, (i + 1) * entry.size));
    }
    return ret;
  }
  parseEntry(type, data) {
    switch (type) {
      case import_shared2.NVMEntryType.Byte:
        return data.readUInt8(0);
      case import_shared2.NVMEntryType.Word:
      case import_shared2.NVMEntryType.NVMModuleSize:
        return data.readUInt16BE(0);
      case import_shared2.NVMEntryType.DWord:
        return data.readUInt32BE(0);
      case import_shared2.NVMEntryType.NodeInfo:
        if (data.every((byte) => byte === 0)) {
          return void 0;
        }
        return (0, import_EntryParsers.parseNVM500NodeInfo)(data, 0);
      case import_shared2.NVMEntryType.NodeMask:
        return (0, import_core.parseBitMask)(data);
      case import_shared2.NVMEntryType.SUCUpdateEntry:
        if (data.every((byte) => byte === 0)) {
          return void 0;
        }
        return (0, import_sucUpdateEntry.parseSUCUpdateEntry)(data, 0);
      case import_shared2.NVMEntryType.Route:
        if (data.every((byte) => byte === 0)) {
          return void 0;
        }
        return (0, import_routeCache.parseRoute)(data, 0);
      case import_shared2.NVMEntryType.NVMModuleDescriptor: {
        return (0, import_EntryParsers.parseNVMModuleDescriptor)(data);
      }
      case import_shared2.NVMEntryType.NVMDescriptor:
        return (0, import_EntryParsers.parseNVMDescriptor)(data);
      default:
        return data;
    }
  }
  async readEntry(entry) {
    const data = await this.readRawEntry(entry);
    return data.map((buffer) => this.parseEntry(entry.type, import_shared.Bytes.view(buffer)));
  }
  async readSingleEntry(entry, index) {
    const data = await this.readSingleRawEntry(entry, index);
    return this.parseEntry(entry.type, import_shared.Bytes.view(data));
  }
  async get(property) {
    this._info ??= await this.init();
    await this.ensureReadable();
    const entry = this._info.layout.get(property);
    if (!entry)
      return void 0;
    return this.readEntry(entry);
  }
  async getSingle(property, index) {
    this._info ??= await this.init();
    await this.ensureReadable();
    const entry = this._info.layout.get(property);
    if (!entry)
      return void 0;
    return this.readSingleEntry(entry, index);
  }
  encodeEntry(type, data, entrySize) {
    const size = entrySize ?? import_shared2.NVMEntrySizes[type];
    switch (type) {
      case import_shared2.NVMEntryType.Byte:
        return import_shared.Bytes.from([data]);
      case import_shared2.NVMEntryType.Word:
      case import_shared2.NVMEntryType.NVMModuleSize: {
        const ret = new import_shared.Bytes(2);
        ret.writeUInt16BE(data, 0);
        return ret;
      }
      case import_shared2.NVMEntryType.DWord: {
        const ret = new import_shared.Bytes(4);
        ret.writeUInt32BE(data, 0);
        return ret;
      }
      case import_shared2.NVMEntryType.NodeInfo:
        return data ? (0, import_EntryParsers.encodeNVM500NodeInfo)(data) : new import_shared.Bytes(size).fill(0);
      case import_shared2.NVMEntryType.NodeMask: {
        const ret = new import_shared.Bytes(size).fill(0);
        if (data) {
          ret.set((0, import_core.encodeBitMask)(data, import_core.MAX_NODES, 1), 0);
        }
        return ret;
      }
      case import_shared2.NVMEntryType.SUCUpdateEntry:
        return (0, import_sucUpdateEntry.encodeSUCUpdateEntry)(data);
      case import_shared2.NVMEntryType.Route:
        return (0, import_routeCache.encodeRoute)(data);
      case import_shared2.NVMEntryType.NVMModuleDescriptor:
        return (0, import_EntryParsers.encodeNVMModuleDescriptor)(data);
      case import_shared2.NVMEntryType.NVMDescriptor:
        return (0, import_EntryParsers.encodeNVMDescriptor)(data);
      case import_shared2.NVMEntryType.Buffer:
        return data;
    }
  }
  async writeSingleRawEntry(entry, index, data) {
    if (index >= entry.count) {
      throw new import_core.ZWaveError(`Index out of range. Tried to write entry ${index} of ${entry.count}.`, import_core.ZWaveErrorCodes.Argument_Invalid);
    }
    return (0, import_utils.nvmWriteBuffer)(this._io, entry.offset + index * entry.size, data);
  }
  async writeRawEntry(entry, data) {
    await (0, import_utils.nvmWriteBuffer)(this._io, entry.offset, import_shared.Bytes.concat(data));
  }
  async writeEntry(entry, data) {
    const buffers = data.map((d) => this.encodeEntry(entry.type, d, entry.size));
    await this.writeRawEntry(entry, buffers);
  }
  async writeSingleEntry(entry, index, data) {
    const buffer = this.encodeEntry(entry.type, data, entry.size);
    await this.writeSingleRawEntry(entry, index, buffer);
  }
  async set(property, value) {
    this._info ??= await this.init();
    await this.ensureWritable();
    const entry = this._info.layout.get(property);
    if (!entry)
      return;
    await this.writeEntry(entry, value);
  }
  async setSingle(property, index, value) {
    this._info ??= await this.init();
    await this.ensureWritable();
    const entry = this._info.layout.get(property);
    if (!entry)
      return void 0;
    await this.writeSingleEntry(entry, index, value);
  }
  async fill(key, value) {
    this._info ??= await this.init();
    await this.ensureWritable();
    const entry = this._info.layout.get(key);
    if (!entry)
      return;
    const size = entry.size ?? import_shared2.NVMEntrySizes[entry.type];
    const data = [];
    for (let i = 1; i <= entry.count; i++) {
      switch (entry.type) {
        case import_shared2.NVMEntryType.Byte:
        case import_shared2.NVMEntryType.Word:
        case import_shared2.NVMEntryType.DWord:
          data.push(value);
          break;
        case import_shared2.NVMEntryType.Buffer:
          data.push(new Uint8Array(size).fill(value));
          break;
        case import_shared2.NVMEntryType.NodeMask:
          data.push(Array.from({ length: size }).fill(value));
          break;
        case import_shared2.NVMEntryType.NodeInfo:
        case import_shared2.NVMEntryType.Route:
          data.push(void 0);
          break;
        default:
          throw new Error(`Cannot fill entry of type ${import_shared2.NVMEntryType[entry.type]}`);
      }
    }
    await this.writeEntry(entry, data);
  }
  // eslint-disable-next-line @typescript-eslint/require-await
  async delete(_property) {
    throw new Error("Deleting entries is not supported for 500 series NVMs");
  }
  async erase(options) {
    await (0, import_utils.nvmWriteBuffer)(this._io, 0, new Uint8Array(options.nvmSize).fill(255));
    const layoutEntries = Array.from(options.layout.values());
    const moduleSizeEntries = layoutEntries.filter((entry) => entry.type === import_shared2.NVMEntryType.NVMModuleSize);
    const moduleDescriptorEntries = layoutEntries.filter((entry) => entry.type === import_shared2.NVMEntryType.NVMModuleDescriptor);
    const moduleDescriptors = /* @__PURE__ */ new Map();
    for (let i = 0; i < moduleSizeEntries.length; i++) {
      const sizeEntry = moduleSizeEntries[i];
      const descriptorEntry = moduleDescriptorEntries[i];
      const size = descriptorEntry.offset + descriptorEntry.size - sizeEntry.offset;
      await this.writeEntry(sizeEntry, [size]);
      const moduleType = descriptorEntry.name === "nvmZWlibraryDescriptor" ? import_shared2.NVMModuleType.ZW_LIBRARY : descriptorEntry.name === "nvmApplicationDescriptor" ? import_shared2.NVMModuleType.APPLICATION : descriptorEntry.name === "nvmHostApplicationDescriptor" ? import_shared2.NVMModuleType.HOST_APPLICATION : descriptorEntry.name === "nvmDescriptorDescriptor" ? import_shared2.NVMModuleType.NVM_DESCRIPTOR : 0;
      const moduleDescriptor = {
        size,
        type: moduleType,
        version: descriptorEntry.name === "nvmZWlibraryDescriptor" ? options.nvmDescriptor.protocolVersion : options.nvmDescriptor.firmwareVersion
      };
      moduleDescriptors.set(descriptorEntry.name, moduleDescriptor);
      await this.writeEntry(descriptorEntry, [moduleDescriptor]);
    }
    this._info = {
      ...options,
      moduleDescriptors
    };
    await this.set("nvmTotalEnd", [options.nvmSize - 1]);
    await this.set("NVM_CONFIGURATION_VALID_far", [import_shared2.CONFIGURATION_VALID_0]);
    await this.set("NVM_CONFIGURATION_REALLYVALID_far", [
      import_shared2.CONFIGURATION_VALID_1
    ]);
    await this.set("EEOFFSET_MAGIC_far", [import_shared2.MAGIC_VALUE]);
    await this.set("EX_NVM_ROUTECACHE_MAGIC_far", [import_shared2.ROUTECACHE_VALID]);
    await this.set("nvmModuleSizeEndMarker", [0]);
    await this.set("nvmDescriptor", [options.nvmDescriptor]);
    await this.fill("NVM_INTERNAL_RESERVED_1_far", 0);
    await this.fill("NVM_INTERNAL_RESERVED_2_far", 255);
    await this.fill("NVM_INTERNAL_RESERVED_3_far", 0);
    await this.fill("NVM_RTC_TIMERS_far", 0);
    await this.fill("EX_NVM_SUC_ACTIVE_START_far", 0);
    await this.fill("EX_NVM_ZENSOR_TABLE_START_far", 0);
    await this.fill("NVM_SECURITY0_KEY_far", 0);
    await this.fill("EX_NVM_SUC_CONTROLLER_LIST_START_far", 254);
    await this.fill("EX_NVM_NODE_TABLE_START_far", 0);
    await this.fill("EX_NVM_ROUTING_TABLE_START_far", 0);
    await this.fill("EX_NVM_ROUTECACHE_START_far", 0);
    await this.fill("EX_NVM_ROUTECACHE_NLWR_SR_START_far", 0);
  }
}
// Annotate the CommonJS export names for ESM import in node:
0 && (module.exports = {
  NVM500
});
//# sourceMappingURL=NVM500.js.map
