import { ZWaveError, ZWaveErrorCodes } from "@zwave-js/core";
import { Bytes } from "@zwave-js/shared";
import net from "node:net";
import { DisconnectRequest } from "../esphome/ConnectionMessages.js";
import { DeviceInfoRequest, DeviceInfoResponse, } from "../esphome/DeviceInfoMessages.js";
import { ESPHomeMessage, ESPHomeMessageRaw, } from "../esphome/ESPHomeMessage.js";
import { HelloRequest, HelloResponse } from "../esphome/HelloMessages.js";
import { ESPHomeZWaveProxyRequestType, ZWaveProxyFrame, ZWaveProxyRequest, } from "../esphome/ZWaveProxyMessages.js";
var ConnectionState;
(function (ConnectionState) {
    ConnectionState["Disconnected"] = "disconnected";
    ConnectionState["Connecting"] = "connecting";
    ConnectionState["HelloSent"] = "hello_sent";
    ConnectionState["HelloReceived"] = "hello_received";
    ConnectionState["DeviceInfoSent"] = "device_info_sent";
    ConnectionState["DeviceInfoReceived"] = "device_info_received";
    ConnectionState["Ready"] = "ready";
})(ConnectionState || (ConnectionState = {}));
export function createESPHomeFactory(options) {
    return async function () {
        const socket = new net.Socket();
        const host = options.host;
        const port = options.port ?? 6053;
        const timeout = 5000;
        let connectionState = ConnectionState.Disconnected;
        let deviceInfo;
        let sourceController;
        // Buffer for incoming frame reassembly
        let frameBuffer = new Bytes();
        function removeListeners() {
            socket.removeAllListeners("close");
            socket.removeAllListeners("error");
            socket.removeAllListeners("connect");
            socket.removeAllListeners("timeout");
            socket.removeAllListeners("data");
        }
        function sendMessage(message) {
            const frame = message.serialize();
            return new Promise((resolve, reject) => {
                socket.write(frame, (err) => {
                    if (err)
                        reject(err);
                    else
                        resolve();
                });
            });
        }
        async function performHandshake() {
            // Send HelloRequest
            connectionState = ConnectionState.HelloSent;
            const helloRequest = new HelloRequest({
                clientInfo: "zwave-js",
                apiVersionMajor: 1,
                apiVersionMinor: 0,
            });
            await sendMessage(helloRequest);
            // Wait for HelloResponse (handled in data event)
            await waitForState(ConnectionState.HelloReceived, timeout);
            // Send DeviceInfoRequest to check Z-Wave support
            connectionState = ConnectionState.DeviceInfoSent;
            const deviceInfoRequest = new DeviceInfoRequest();
            await sendMessage(deviceInfoRequest);
            // Wait for DeviceInfoResponse
            await waitForState(ConnectionState.DeviceInfoReceived, timeout);
            // Check if device supports Z-Wave proxy
            if (!deviceInfo?.hasZWaveProxySupport) {
                throw new ZWaveError("ESPHome device does not support Z-Wave proxy functionality", ZWaveErrorCodes.Driver_SerialPortClosed);
            }
            // Subscribe to Z-Wave traffic
            const subscribeRequest = new ZWaveProxyRequest({
                type: ESPHomeZWaveProxyRequestType.Subscribe,
            });
            await sendMessage(subscribeRequest);
            // Connection is ready - no service discovery needed
            connectionState = ConnectionState.Ready;
        }
        function waitForState(targetState, timeoutMs) {
            return new Promise((resolve, reject) => {
                if (connectionState === targetState) {
                    resolve();
                    return;
                }
                const startTime = Date.now();
                const checkInterval = setInterval(() => {
                    if (connectionState === targetState) {
                        clearInterval(checkInterval);
                        resolve();
                    }
                    else if (Date.now() - startTime > timeoutMs) {
                        clearInterval(checkInterval);
                        reject(new ZWaveError(`Timeout waiting for connection state ${targetState}`, ZWaveErrorCodes.Driver_SerialPortClosed));
                    }
                }, 10);
            });
        }
        function processIncomingData(data) {
            try {
                frameBuffer = Bytes.concat([frameBuffer, data]);
                // Try to extract complete frames
                while (frameBuffer.length > 0) {
                    try {
                        // Check if we have enough data for the basic header
                        if (frameBuffer.length < 3) {
                            break; // Need more data
                        }
                        // Parse the raw message from buffer
                        const rawMessage = ESPHomeMessageRaw.parse(frameBuffer);
                        // Calculate frame length
                        const frameLength = 1 // indicator
                            + getVarIntLength(rawMessage.payload.length) // payload size
                            + getVarIntLength(rawMessage.messageType) // message type
                            + rawMessage.payload.length; // payload
                        // Parse into specific message types and process
                        const message = ESPHomeMessage.parse(frameBuffer);
                        processIncomingMessage(message);
                        // Remove the processed frame from the buffer
                        frameBuffer = frameBuffer.subarray(frameLength);
                    }
                    catch (error) {
                        // If we can't decode a complete frame yet, wait for more data
                        if (error instanceof ZWaveError
                            && error.code
                                === ZWaveErrorCodes.PacketFormat_Truncated) {
                            break;
                        }
                        // For other errors, reset the buffer and continue
                        frameBuffer = new Bytes();
                        break;
                    }
                }
            }
            catch {
                // Reset buffer on any parsing error
                frameBuffer = new Bytes();
            }
        }
        function processIncomingMessage(message) {
            if (message instanceof HelloResponse) {
                if (connectionState === ConnectionState.HelloSent) {
                    connectionState = ConnectionState.HelloReceived;
                }
            }
            else if (message instanceof DeviceInfoResponse) {
                if (connectionState === ConnectionState.DeviceInfoSent) {
                    deviceInfo = message;
                    connectionState = ConnectionState.DeviceInfoReceived;
                }
            }
            else if (message instanceof ZWaveProxyFrame) {
                // Handle Z-Wave proxy frames returned from the device
                // This message may include full payloads or simple ACK/NAK/CAN responses
                // If we have a source controller, enqueue any remaining data for Z-Wave processing
                // (This would be for any data that's not ESPHome protocol messages)
                if (sourceController
                    && connectionState === ConnectionState.Ready) {
                    // Enqueue frame to handle Z-Wave data as needed
                    sourceController.enqueue(message.data);
                }
            }
        }
        function open() {
            return new Promise((resolve, reject) => {
                const onClose = () => {
                    removeListeners();
                    socket.destroy();
                    reject(new ZWaveError(`ESPHome connection closed unexpectedly!`, ZWaveErrorCodes.Driver_SerialPortClosed));
                };
                const onError = (err) => {
                    removeListeners();
                    socket.destroy();
                    reject(err);
                };
                const onTimeout = () => {
                    removeListeners();
                    socket.destroy();
                    reject(new ZWaveError(`Connection timed out after ${timeout}ms`, ZWaveErrorCodes.Driver_SerialPortClosed));
                };
                const onConnect = async () => {
                    removeListeners();
                    connectionState = ConnectionState.Connecting;
                    // During testing, values below 1000 caused the keep alive functionality to silently fail
                    socket.setKeepAlive(true, 1000);
                    // Prevent communication delays
                    socket.setNoDelay();
                    // FIXME: We should set the SO_RCVBUF to 2 MB or so
                    // like aioesphome does, but Node.js does not expose
                    // a way to do that natively.
                    // https://github.com/derhuerst/node-sockopt might help.
                    // Set up data listener before performing handshake
                    socket.on("data", processIncomingData);
                    try {
                        await performHandshake();
                        resolve();
                    }
                    catch (error) {
                        // Clean up the socket on handshake failure
                        socket.destroy();
                        reject(error instanceof Error
                            ? error
                            : new Error(String(error)));
                    }
                };
                socket.setTimeout(timeout);
                socket.once("close", onClose);
                socket.once("error", onError);
                socket.once("timeout", onTimeout);
                socket.once("connect", onConnect);
                socket.connect(port, host);
            });
        }
        async function close() {
            try {
                // Send disconnect request if connected
                if (connectionState !== ConnectionState.Disconnected) {
                    const disconnectRequest = new DisconnectRequest();
                    await sendMessage(disconnectRequest);
                }
            }
            catch {
                // Ignore errors during disconnect
            }
            return new Promise((resolve) => {
                removeListeners();
                connectionState = ConnectionState.Disconnected;
                if (socket.destroyed) {
                    resolve();
                }
                else {
                    socket.once("close", () => resolve()).destroy();
                }
            });
        }
        await open();
        let isOpen = true;
        const sink = {
            async write(data, controller) {
                if (!isOpen || connectionState !== ConnectionState.Ready) {
                    controller.error(new Error("ESPHome connection is not ready!"));
                    return;
                }
                if (!deviceInfo?.hasZWaveProxySupport) {
                    controller.error(new Error("Z-Wave proxy support not available!"));
                    return;
                }
                try {
                    // Create Z-Wave proxy write request with Bytes data
                    const writeRequest = new ZWaveProxyFrame({
                        data: new Bytes(data),
                    });
                    // Send the Z-Wave proxy write request
                    await sendMessage(writeRequest);
                }
                catch (error) {
                    controller.error(error);
                }
            },
            close() {
                return close();
            },
            abort(_reason) {
                return close();
            },
        };
        const source = {
            start(controller) {
                // Store the controller so we can enqueue data when needed
                sourceController = controller;
                // Handle ESPHome connection events
                socket.on("close", () => {
                    isOpen = false;
                    connectionState = ConnectionState.Disconnected;
                    controller.error(new ZWaveError(`ESPHome connection closed unexpectedly!`, ZWaveErrorCodes.Driver_SerialPortClosed));
                });
                socket.on("error", (_e) => {
                    isOpen = false;
                    connectionState = ConnectionState.Disconnected;
                    controller.error(new ZWaveError(`ESPHome connection error!`, ZWaveErrorCodes.Driver_SerialPortClosed));
                });
            },
            cancel() {
                sourceController = undefined;
                socket.removeAllListeners();
            },
        };
        return { source, sink };
    };
}
/**
 * Helper function to calculate VarInt length
 */
function getVarIntLength(value) {
    let length = 1;
    while (value >= 0x80) {
        value >>>= 7;
        length++;
    }
    return length;
}
//# sourceMappingURL=ESPHomeSocket.js.map