import { type CCId, type SecurityManagers } from "@zwave-js/core";
import { Message, type MessageEncodingContext, type MessageParsingContext, type ZWaveSerialStream } from "@zwave-js/serial";
import type { MockPort } from "@zwave-js/serial/mock";
import { type BytesView } from "@zwave-js/shared";
import { type MockControllerCapabilities } from "./MockControllerCapabilities.js";
import type { MockNode, NodePendingInclusion } from "./MockNode.js";
import { type LazyMockZWaveFrame, type MockZWaveAckFrame, type MockZWaveFrame, type MockZWaveRequestFrame } from "./MockZWaveFrame.js";
export interface MockControllerOptions {
    mockPort: MockPort;
    serial: ZWaveSerialStream;
    ownNodeId?: number;
    homeId?: number;
    capabilities?: Partial<MockControllerCapabilities>;
    securityKeys?: {
        S2_AccessControl?: BytesView;
        S2_Authenticated?: BytesView;
        S2_Unauthenticated?: BytesView;
        S0_Legacy?: BytesView;
    };
}
/** A mock Z-Wave controller which interacts with {@link MockNode}s and can be controlled via a {@link MockSerialPort} */
export declare class MockController {
    static create(options: MockControllerOptions): Promise<MockController>;
    private constructor();
    private _options;
    homeId: number;
    ownNodeId: number;
    securityManagers: SecurityManagers;
    private setupSecurityManagers;
    encodingContext: MessageEncodingContext;
    parsingContext: MessageParsingContext;
    readonly mockPort: MockPort;
    readonly serial: ZWaveSerialStream;
    private expectedHostACKs;
    private expectedHostMessages;
    private expectedNodeFrames;
    private behaviors;
    /** Shared medium for sending messages back and forth */
    private air;
    /** Records the messages received from the host to perform assertions on them */
    private _receivedHostMessages;
    get receivedHostMessages(): readonly Readonly<Message>[];
    private _nodes;
    get nodes(): ReadonlyMap<number, MockNode>;
    addNode(node: MockNode): void;
    removeNode(node: MockNode): void;
    readonly capabilities: MockControllerCapabilities;
    /** Can be used by behaviors to store controller related state */
    readonly state: Map<string, unknown>;
    /** Node info for the node that is pending inclusion. Set this before starting inclusion to simulate a node joining. */
    nodePendingInclusion: NodePendingInclusion | undefined;
    /** Controls whether the controller automatically ACKs messages from the host before handling them */
    autoAckHostMessages: boolean;
    /** Controls whether the controller automatically ACKs node frames before handling them */
    autoAckNodeFrames: boolean;
    /** Allows reproducing issues with the 7.19.x firmware where the high nibble of the ACK after soft-reset is corrupted */
    corruptACK: boolean;
    private handleSerialData;
    /** Gets called when parsed/chunked data is received from the serial port */
    private serialOnData;
    /**
     * Waits until the host sends an ACK or a timeout has elapsed.
     *
     * @param timeout The number of milliseconds to wait. If the timeout elapses, the returned promise will be rejected
     */
    expectHostACK(timeout: number, errorMessage?: string): Promise<void>;
    /**
     * Waits until the host sends a message matching the given predicate or a timeout has elapsed.
     *
     * @param predicate A predicate function to test incoming messages
     * @param options Optional configuration
     * @param options.timeout The number of milliseconds to wait. If the timeout elapses, the returned promise will be rejected. Default: 5000ms
     * @param options.preventDefault If true, the default behavior will not be executed after the expectation is fulfilled. Default: false
     */
    expectHostMessage(predicate: (msg: Message) => boolean, options?: {
        timeout?: number;
        preventDefault?: boolean;
        errorMessage?: string;
    }): Promise<Message>;
    /**
     * Waits until the node sends a message matching the given predicate or a timeout has elapsed.
     *
     * @param node The node to expect a frame from
     * @param predicate A predicate function to test incoming frames
     * @param options Optional configuration
     * @param options.timeout The number of milliseconds to wait. If the timeout elapses, the returned promise will be rejected. Default: 5000ms
     * @param options.preventDefault If true, the default behavior will not be executed after the expectation is fulfilled. Default: false
     */
    expectNodeFrame<T extends MockZWaveFrame = MockZWaveFrame>(node: MockNode, predicate: (msg: MockZWaveFrame) => msg is T, options?: {
        timeout?: number;
        preventDefault?: boolean;
        errorMessage?: string;
    }): Promise<T>;
    /**
     * Waits until the node sends a message matching the given predicate or a timeout has elapsed.
     *
     * @param node The node to expect a CC from
     * @param predicate A predicate function to test incoming CCs
     * @param options Optional configuration
     * @param options.timeout The number of milliseconds to wait. If the timeout elapses, the returned promise will be rejected. Default: 5000ms
     * @param options.preventDefault If true, the default behavior will not be executed after the expectation is fulfilled. Default: false
     */
    expectNodeCC<T extends CCId = CCId>(node: MockNode, predicate: (cc: CCId) => cc is T, options?: {
        timeout?: number;
        preventDefault?: boolean;
        errorMessage?: string;
    }): Promise<T>;
    /**
     * Waits until the controller sends an ACK frame or a timeout has elapsed.
     *
     * @param timeout The number of milliseconds to wait. If the timeout elapses, the returned promise will be rejected
     */
    expectNodeACK(node: MockNode, timeout: number, errorMessage?: string): Promise<MockZWaveAckFrame>;
    /** Sends a message header (ACK/NAK/CAN) to the host/driver */
    private sendHeaderToHost;
    /** Sends a raw buffer to the host/driver and expect an ACK */
    sendMessageToHost(msg: Message, fromNode?: MockNode): Promise<void>;
    /** Sends a raw buffer to the host/driver and expect an ACK */
    sendToHost(data: BytesView): Promise<void>;
    /**
     * Sends an ACK frame to the host
     */
    ackHostMessage(): void;
    /** Gets called when a {@link MockZWaveFrame} is received from a {@link MockNode} */
    onNodeFrame(node: MockNode, frame: MockZWaveFrame): Promise<void>;
    /**
     * Sends an ACK frame to a {@link MockNode}
     */
    ackNodeRequestFrame(node: MockNode, frame?: MockZWaveRequestFrame): Promise<void>;
    /**
     * Sends a {@link MockZWaveFrame} to a {@link MockNode}
     */
    sendToNode(node: MockNode, frame: LazyMockZWaveFrame): Promise<MockZWaveAckFrame | undefined>;
    defineBehavior(...behaviors: MockControllerBehavior[]): void;
    /** Asserts that a message matching the given predicate was received from the host */
    assertReceivedHostMessage(predicate: (msg: Message) => boolean, options?: {
        errorMessage?: string;
    }): void;
    /** Forgets all recorded messages received from the host */
    clearReceivedHostMessages(): void;
    execute(): Promise<void>;
    destroy(): void;
}
export interface MockControllerBehavior {
    /**
     * Can be used to inspect raw data received from the host before it is processed by the serial parser and the mock controller.
     * Return `true` to indicate that the data has been handled and should not be processed further.
     */
    onHostData?: (controller: MockController, data: BytesView) => Promise<boolean | undefined> | boolean | undefined;
    /** Gets called when a message from the host is received. Return `true` to indicate that the message has been handled. */
    onHostMessage?: (controller: MockController, msg: Message) => Promise<boolean | undefined> | boolean | undefined;
    /** Gets called when a message from a node is received. Return `true` to indicate that the message has been handled. */
    onNodeFrame?: (controller: MockController, node: MockNode, frame: MockZWaveFrame) => Promise<boolean | undefined> | boolean | undefined;
}
//# sourceMappingURL=MockController.d.ts.map