//#region src/compile.d.ts

type TemplateFunction = (this: Eta$1, data?: object, options?: Partial<Options>) => string;
/**
 * Takes a template string and returns a template function that can be called with (data, config)
 *
 * @param str - The template string
 * @param config - A custom configuration object (optional)
 */
declare function compile(this: Eta$1, str: string, options?: Partial<Options>): TemplateFunction;
//#endregion
//#region src/compile-string.d.ts
/**
 * Compiles a template string to a function string. Most often users just use `compile()`, which calls `compileToString` and creates a new function using the result
 */
declare function compileToString(this: Eta$1, str: string, options?: Partial<Options>): string;
/**
 * Loops through the AST generated by `parse` and transform each item into JS calls
 *
 * **Example**
 *
 * ```js
 * let templateAST = ['Hi ', { val: 'it.name', t: 'i' }]
 * compileBody.call(Eta, templateAST)
 * // => "__eta.res+='Hi '\n__eta.res+=__eta.e(it.name)\n"
 * ```
 */
declare function compileBody(this: Eta$1, buff: Array<AstObject>): string;
//#endregion
//#region src/err.d.ts
declare class EtaError extends Error {
  constructor(message: string);
}
declare class EtaParseError extends EtaError {
  constructor(message: string);
}
declare class EtaRuntimeError extends EtaError {
  constructor(message: string);
}
declare class EtaFileResolutionError extends EtaError {
  constructor(message: string);
}
declare class EtaNameResolutionError extends EtaError {
  constructor(message: string);
}
/**
 * Throws an EtaError with a nicely formatted error and message showing where in the template the error occurred.
 */

declare function RuntimeErr(originalError: Error, str: string, lineNo: number, path: string): never;
//#endregion
//#region src/render.d.ts
declare function render<T extends object>(this: Eta$1, template: string | TemplateFunction,
// template name or template function
data: T, meta?: {
  filepath: string;
}): string;
declare function renderAsync<T extends object>(this: Eta$1, template: string | TemplateFunction,
// template name or template function
data: T, meta?: {
  filepath: string;
}): Promise<string>;
declare function renderString<T extends object>(this: Eta$1, template: string, data: T): string;
declare function renderStringAsync<T extends object>(this: Eta$1, template: string, data: T): Promise<string>;
//#endregion
//#region src/storage.d.ts
/**
 * Handles storage and accessing of values
 *
 * In this case, we use it to store compiled template functions
 * Indexed by their `name` or `filename`
 */
declare class Cacher<T> {
  private cache;
  constructor(cache: Record<string, T>);
  define(key: string, val: T): void;
  get(key: string): T;
  remove(key: string): void;
  reset(): void;
  load(cacheObj: Record<string, T>): void;
}
//#endregion
//#region src/internal.d.ts
declare class Eta$1 {
  constructor(customConfig?: Partial<EtaConfig>);
  config: EtaConfig;
  RuntimeErr: typeof RuntimeErr;
  compile: typeof compile;
  compileToString: typeof compileToString;
  compileBody: typeof compileBody;
  parse: typeof parse;
  render: typeof render;
  renderAsync: typeof renderAsync;
  renderString: typeof renderString;
  renderStringAsync: typeof renderStringAsync;
  filepathCache: Record<string, string>;
  templatesSync: Cacher<TemplateFunction>;
  templatesAsync: Cacher<TemplateFunction>;
  resolvePath: null | ((this: Eta$1, template: string, options?: Partial<Options>) => string);
  readFile: null | ((this: Eta$1, path: string) => string);
  configure(customConfig: Partial<EtaConfig>): void;
  withConfig(customConfig: Partial<EtaConfig>): this & {
    config: EtaConfig;
  };
  loadTemplate(name: string, template: string | TemplateFunction,
  // template string or template function
  options?: {
    async: boolean;
  }): void;
}
//#endregion
//#region src/parse.d.ts
type TagType = "r" | "e" | "i" | "";
interface TemplateObject {
  t: TagType;
  val: string;
  lineNo?: number;
}
type AstObject = string | TemplateObject;
declare function parse(this: Eta$1, str: string): Array<AstObject>;
//#endregion
//#region src/config.d.ts
type trimConfig = "nl" | "slurp" | false;
interface Options {
  /** Compile to async function */
  async?: boolean;
  /** Absolute path to template file */
  filepath?: string;
}
interface EtaConfig {
  /** Whether or not to automatically XML-escape interpolations. Default true */
  autoEscape: boolean;
  /** Apply a filter function defined on the class to every interpolation or raw interpolation */
  autoFilter: boolean;
  /** Configure automatic whitespace trimming. Default `[false, 'nl']` */
  autoTrim: trimConfig | [trimConfig, trimConfig];
  /** Whether or not to cache templates if `name` or `filename` is passed */
  cache: boolean;
  /** Holds cache of resolved filepaths. Set to `false` to disable. */
  cacheFilepaths: boolean;
  /** Whether to pretty-format error messages (introduces runtime penalties) */
  debug: boolean;
  /** Function to XML-sanitize interpolations */
  escapeFunction: (str: unknown) => string;
  /** Function applied to all interpolations when autoFilter is true */
  filterFunction: (val: unknown) => string;
  /** Raw JS code inserted in the template function. Useful for declaring global variables for user templates */
  functionHeader: string;
  /** Parsing options */
  parse: {
    /** Which prefix to use for evaluation. Default `""`, does not support `"-"` or `"_"` */
    exec: string;
    /** Which prefix to use for interpolation. Default `"="`, does not support `"-"` or `"_"` */
    interpolate: string;
    /** Which prefix to use for raw interpolation. Default `"~"`, does not support `"-"` or `"_"` */
    raw: string;
  };
  /** Array of plugins */
  plugins: Array<{
    processFnString?: (fnString: string, env?: EtaConfig) => string;
    processAST?: (ast: AstObject[], env?: EtaConfig) => AstObject[];
    processTemplate?: (fnString: string, env?: EtaConfig) => string;
  }>;
  /** Remove all safe-to-remove whitespace */
  rmWhitespace: boolean;
  /** Delimiters: by default `['<%', '%>']` */
  tags: [string, string];
  /** Make data available on the global object instead of varName */
  useWith: boolean;
  /** Name of the data object. Default `it` */
  varName: string;
  /** Directory that contains templates */
  views?: string;
  /** Control template file extension defaults. Default `.eta` */
  defaultExtension?: string;
}
/** Eta's base (global) configuration */
//#endregion
//#region src/file-handling.d.ts
declare function readFile(this: Eta$1, path: string): string;
declare function resolvePath(this: Eta$1, templatePath: string, options?: Partial<Options>): string;
//#endregion
//#region src/index.d.ts
declare class Eta extends Eta$1 {
  readFile: typeof readFile;
  resolvePath: typeof resolvePath;
}
//#endregion
export { Eta, type EtaConfig, EtaError, EtaFileResolutionError, EtaNameResolutionError, EtaParseError, EtaRuntimeError, type Options };
//# sourceMappingURL=index.d.ts.map