import type { CommandClass } from "@zwave-js/cc";
import { type BeamingInfo, MPDUHeaderType, type MessageOrCCLogEntry, NODE_ID_BROADCAST, NODE_ID_BROADCAST_LR, Protocols, type RSSI, type ZnifferProtocolDataRate, ZnifferRegion } from "@zwave-js/core";
import { type ZnifferDataMessage, type ZnifferFrameInfo } from "@zwave-js/serial";
import { type AllOrNone, Bytes, type BytesView } from "@zwave-js/shared";
import { ExplorerFrameCommand, LongRangeFrameType, ZWaveFrameType } from "./_Types.js";
export interface MPDUOptions {
    data: Bytes;
    frameInfo: ZnifferFrameInfo;
}
export interface MPDU {
    frameInfo: ZnifferFrameInfo;
    homeId: number;
    sourceNodeId: number;
    ackRequested: boolean;
    headerType: MPDUHeaderType;
    sequenceNumber: number;
    payload: Bytes;
}
export declare function parseMPDU(frame: ZnifferDataMessage): ZWaveMPDU | LongRangeMPDU;
export declare class LongRangeMPDU implements MPDU {
    constructor(options: MPDUOptions);
    readonly frameInfo: ZnifferFrameInfo;
    readonly homeId: number;
    readonly sourceNodeId: number;
    readonly destinationNodeId: number;
    readonly ackRequested: boolean;
    readonly headerType: MPDUHeaderType;
    readonly sequenceNumber: number;
    readonly noiseFloor: RSSI;
    readonly txPower: number;
    payload: Bytes;
    static from(msg: ZnifferDataMessage): LongRangeMPDU;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class SinglecastLongRangeMPDU extends LongRangeMPDU {
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class AckLongRangeMPDU extends LongRangeMPDU {
    constructor(options: MPDUOptions);
    readonly incomingRSSI: RSSI;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class ZWaveMPDU implements MPDU {
    constructor(options: MPDUOptions);
    readonly frameInfo: ZnifferFrameInfo;
    readonly homeId: number;
    readonly sourceNodeId: number;
    readonly routed: boolean;
    readonly ackRequested: boolean;
    readonly lowPower: boolean;
    readonly speedModified: boolean;
    readonly headerType: MPDUHeaderType;
    readonly beamingInfo: BeamingInfo;
    readonly sequenceNumber: number;
    protected readonly destinationBuffer: Bytes;
    payload: Bytes;
    static from(msg: ZnifferDataMessage): ZWaveMPDU;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class SinglecastZWaveMPDU extends ZWaveMPDU {
    constructor(options: MPDUOptions);
    readonly destinationNodeId: number;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class AckZWaveMPDU extends ZWaveMPDU {
    constructor(options: MPDUOptions);
    readonly destinationNodeId: number;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class RoutedZWaveMPDU extends ZWaveMPDU {
    constructor(options: MPDUOptions);
    readonly destinationNodeId: number;
    readonly direction: "outbound" | "inbound";
    readonly routedAck: boolean;
    readonly routedError: boolean;
    readonly failedHop?: number;
    readonly hop: number;
    readonly repeaters: readonly number[];
    readonly destinationWakeup?: boolean;
    readonly destinationWakeupType?: "250ms" | "1000ms";
    readonly repeaterRSSI?: readonly RSSI[];
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class MulticastZWaveMPDU extends ZWaveMPDU {
    constructor(options: MPDUOptions);
    readonly destinationNodeIds: readonly number[];
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class ExplorerZWaveMPDU extends ZWaveMPDU {
    constructor(options: MPDUOptions);
    readonly destinationNodeId: number;
    readonly version: number;
    readonly command: ExplorerFrameCommand;
    readonly stop: boolean;
    readonly sourceRouted: boolean;
    readonly direction: "outbound" | "inbound";
    readonly randomTXInterval: number;
    readonly ttl: number;
    readonly repeaters: readonly number[];
}
export declare class NormalExplorerZWaveMPDU extends ExplorerZWaveMPDU {
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class InclusionRequestExplorerZWaveMPDU extends ExplorerZWaveMPDU {
    constructor(options: MPDUOptions);
    /** The home ID of the repeating node */
    readonly networkHomeId: number;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class SearchResultExplorerZWaveMPDU extends ExplorerZWaveMPDU {
    constructor(options: MPDUOptions);
    /** The node ID that sent the explorer frame that's being answered here */
    readonly searchingNodeId: number;
    /** The sequence number of the original explorer frame */
    readonly frameHandle: number;
    readonly resultTTL: number;
    readonly resultRepeaters: readonly number[];
    toLogEntry(): MessageOrCCLogEntry;
}
export declare function parseBeamFrame(frame: ZnifferDataMessage): ZWaveBeamStart | LongRangeBeamStart | BeamStop;
export declare class ZWaveBeamStart {
    constructor(options: MPDUOptions);
    readonly frameInfo: ZnifferFrameInfo;
    readonly homeIdHash?: number;
    readonly destinationNodeId: number;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class LongRangeBeamStart {
    constructor(options: MPDUOptions);
    readonly frameInfo: ZnifferFrameInfo;
    readonly homeIdHash: number;
    readonly destinationNodeId: number;
    readonly txPower: number;
    toLogEntry(): MessageOrCCLogEntry;
}
export declare class BeamStop {
    constructor(options: MPDUOptions);
    readonly frameInfo: ZnifferFrameInfo;
    toLogEntry(): MessageOrCCLogEntry;
}
/** An application-oriented representation of a Z-Wave frame that was captured by the Zniffer */
export type ZWaveFrame = {
    protocol: Protocols.ZWave;
    channel: number;
    region: number;
    rssiRaw: number;
    rssi?: RSSI;
    protocolDataRate: ZnifferProtocolDataRate;
    speedModified: boolean;
    sequenceNumber: number;
    homeId: number;
    sourceNodeId: number;
} & (({
    type: ZWaveFrameType.Singlecast;
    destinationNodeId: number;
    ackRequested: boolean;
    payload: BytesView | CommandClass;
} & AllOrNone<{
    direction: "outbound" | "inbound";
    hop: number;
    repeaters: number[];
    repeaterRSSI?: RSSI[];
} & ({
    routedAck: false;
    routedError: false;
    failedHop?: undefined;
} | {
    routedAck: true;
    routedError: false;
    failedHop?: undefined;
} | {
    routedAck: false;
    routedError: true;
    failedHop: number;
})>) | {
    type: ZWaveFrameType.Broadcast;
    destinationNodeId: typeof NODE_ID_BROADCAST;
    ackRequested: boolean;
    payload: BytesView | CommandClass;
} | {
    type: ZWaveFrameType.Multicast;
    destinationNodeIds: number[];
    payload: BytesView | CommandClass;
} | {
    type: ZWaveFrameType.AckDirect;
    destinationNodeId: number;
} | (({
    type: ZWaveFrameType.ExplorerNormal;
    payload: BytesView | CommandClass;
} | {
    type: ZWaveFrameType.ExplorerSearchResult;
    searchingNodeId: number;
    frameHandle: number;
    resultTTL: number;
    resultRepeaters: readonly number[];
} | {
    type: ZWaveFrameType.ExplorerInclusionRequest;
    networkHomeId: number;
    payload: BytesView | CommandClass;
}) & {
    destinationNodeId: number;
    ackRequested: boolean;
    direction: "outbound" | "inbound";
    repeaters: number[];
    ttl: number;
}));
export type LongRangeFrame = {
    protocol: Protocols.ZWaveLongRange;
    channel: number;
    region: ZnifferRegion;
    protocolDataRate: ZnifferProtocolDataRate;
    rssiRaw: number;
    rssi?: RSSI;
    noiseFloor: RSSI;
    txPower: number;
    sequenceNumber: number;
    homeId: number;
    sourceNodeId: number;
    destinationNodeId: number;
} & ({
    type: LongRangeFrameType.Singlecast;
    ackRequested: boolean;
    payload: BytesView | CommandClass;
} | {
    type: LongRangeFrameType.Broadcast;
    destinationNodeId: typeof NODE_ID_BROADCAST_LR;
    ackRequested: boolean;
    payload: BytesView | CommandClass;
} | {
    type: LongRangeFrameType.Ack;
    incomingRSSI: RSSI;
    payload: BytesView;
});
export type BeamFrame = {
    channel: number;
} & ({
    protocol: Protocols.ZWave;
    type: ZWaveFrameType.BeamStart;
    protocolDataRate: ZnifferProtocolDataRate;
    rssiRaw: number;
    rssi?: RSSI;
    region: ZnifferRegion;
    homeIdHash?: number;
    destinationNodeId: number;
} | {
    protocol: Protocols.ZWaveLongRange;
    type: LongRangeFrameType.BeamStart;
    protocolDataRate: ZnifferProtocolDataRate;
    rssiRaw: number;
    rssi?: RSSI;
    region: ZnifferRegion;
    txPower: number;
    homeIdHash: number;
    destinationNodeId: number;
} | {
    protocol: Protocols.ZWave;
    type: ZWaveFrameType.BeamStop;
} | {
    protocol: Protocols.ZWaveLongRange;
    type: LongRangeFrameType.BeamStop;
});
export type Frame = ZWaveFrame | LongRangeFrame | BeamFrame;
export type CorruptedFrame = {
    channel: number;
    region: number;
    rssiRaw: number;
    rssi?: RSSI;
    protocolDataRate: ZnifferProtocolDataRate;
    payload: BytesView;
};
export declare function mpduToFrame(mpdu: MPDU, payloadCC?: CommandClass): Frame;
export declare function mpduToZWaveFrame(mpdu: ZWaveMPDU, payloadCC?: CommandClass): ZWaveFrame;
export declare function mpduToLongRangeFrame(mpdu: LongRangeMPDU, payloadCC?: CommandClass): LongRangeFrame;
export declare function beamToFrame(beam: ZWaveBeamStart | LongRangeBeamStart | BeamStop): Frame;
export declare function znifferDataMessageToCorruptedFrame(msg: ZnifferDataMessage, rssi?: RSSI): CorruptedFrame;
//# sourceMappingURL=MPDU.d.ts.map