import { MockPort } from "@zwave-js/serial/mock";
import { createDeferredPromise } from "alcalzone-shared/deferred-promise";
import { tmpdir } from "node:os";
import path from "pathe";
import { Driver } from "./Driver.js";
/** Creates a real driver instance with a mocked serial port to enable end to end tests */
export function createAndStartDriverWithMockPort(options = {}) {
    const { ...driverOptions } = options;
    return new Promise((resolve, reject) => {
        // eslint-disable-next-line prefer-const
        let driver;
        const mockPort = new MockPort();
        const bindingFactory = mockPort.factory();
        // let mockPort: MockPortBinding;
        // MockBinding.reset();
        // MockBinding.createPort(portAddress, {
        // 	record: true,
        // 	readyData: new Uint8Array(),
        // });
        // This will be called when the driver has opened the serial port
        const onSerialPortOpen = (serial) => {
            // // Extract the mock serial port
            // mockPort = MockBinding.getInstance(portAddress)!;
            // if (!mockPort) reject(new Error("Mock serial port is not open!"));
            // And return the info to the calling code, giving it control over
            // continuing the driver startup.
            const continuePromise = createDeferredPromise();
            resolve({
                driver,
                mockPort,
                serial,
                continueStartup: () => continuePromise.resolve(),
            });
            return continuePromise;
        };
        // Usually we don't want logs in these tests
        if (!driverOptions.logConfig) {
            driverOptions.logConfig = {
                enabled: false,
            };
        }
        const testingHooks = {
            ...driverOptions.testingHooks,
            onSerialPortOpen,
        };
        driver = new Driver(bindingFactory, {
            ...driverOptions,
            testingHooks,
        });
        driver.start().catch(reject);
    });
}
export async function createAndStartTestingDriver(options = {}) {
    const { beforeStartup, skipControllerIdentification = false, skipFirmwareIdentification = true, skipNodeInterview = false, loadConfiguration = true, fs = (await import("#default_bindings/fs")).fs, ...internalOptions } = options;
    // Use a new fake serial port for each test
    const testId = Math.round(Math.random() * 0xffffffff)
        .toString(16)
        .padStart(8, "0");
    // internalOptions.portAddress ??= `/tty/FAKE${testId}`;
    if (skipControllerIdentification) {
        internalOptions.testingHooks ??= {};
        internalOptions.testingHooks.skipControllerIdentification = true;
    }
    if (skipNodeInterview) {
        internalOptions.testingHooks ??= {};
        internalOptions.testingHooks.skipNodeInterview = true;
    }
    if (!loadConfiguration) {
        internalOptions.testingHooks ??= {};
        internalOptions.testingHooks.loadConfiguration = false;
    }
    if (skipFirmwareIdentification) {
        internalOptions.testingHooks ??= {};
        internalOptions.testingHooks.skipFirmwareIdentification = true;
    }
    // TODO: Make sure we delete this from time to time
    const cacheDir = path.join(tmpdir(), "zwave-js-test-cache", testId);
    internalOptions.storage ??= {};
    internalOptions.storage.cacheDir = cacheDir;
    const { driver, continueStartup, mockPort, serial } = await createAndStartDriverWithMockPort(internalOptions);
    if (typeof beforeStartup === "function") {
        await beforeStartup(mockPort, serial);
    }
    // Make sure the mock FS gets restored when the driver is destroyed
    const originalDestroy = driver.destroy.bind(driver);
    driver.destroy = async () => {
        await originalDestroy();
        await fs.deleteDir(cacheDir);
    };
    return new Promise((resolve) => {
        driver.once("driver ready", () => {
            resolve({ driver, mockPort, serial });
        });
        continueStartup();
    });
}
//# sourceMappingURL=DriverMock.js.map