import { TransmitStatus, ZWaveError, ZWaveErrorCodes, } from "@zwave-js/core";
import { SendDataBridgeRequest, SendDataMulticastBridgeRequest, SendDataMulticastRequest, SendDataRequest, isSendData, isSendDataTransmitReport, } from "@zwave-js/serial/serialapi";
import { getEnumMemberName } from "@zwave-js/shared";
export function serialAPICommandErrorToZWaveError(reason, sentMessage, receivedMessage, transactionSource) {
    switch (reason) {
        case "CAN":
        case "NAK":
            return new ZWaveError(`Failed to execute controller command`, ZWaveErrorCodes.Controller_MessageDropped, reason, transactionSource);
        case "ACK timeout":
            return new ZWaveError(`Timeout while waiting for an ACK from the controller`, ZWaveErrorCodes.Controller_Timeout, "ACK", transactionSource);
        case "response timeout":
            return new ZWaveError(`Timeout while waiting for a response from the controller`, ZWaveErrorCodes.Controller_Timeout, "response", transactionSource);
        case "callback timeout":
            return new ZWaveError(`Timeout while waiting for a callback from the controller`, ZWaveErrorCodes.Controller_Timeout, "callback", transactionSource);
        case "response NOK": {
            if (isSendData(sentMessage)) {
                return new ZWaveError(`Failed to send the command: Transmission queue full`, ZWaveErrorCodes.Controller_MessageDropped, receivedMessage, transactionSource);
            }
            else {
                return new ZWaveError(`The controller response indicated failure`, ZWaveErrorCodes.Controller_ResponseNOK, receivedMessage, transactionSource);
            }
        }
        case "callback NOK": {
            if (isSendData(sentMessage)
                && isSendDataTransmitReport(receivedMessage)
                && receivedMessage.transmitStatus === TransmitStatus.Fail) {
                return new ZWaveError(`Failed to send the command, the controller is jammed`, ZWaveErrorCodes.Controller_Jammed, receivedMessage, transactionSource);
            }
            if (sentMessage instanceof SendDataRequest
                || sentMessage instanceof SendDataBridgeRequest) {
                const status = receivedMessage.transmitStatus;
                return new ZWaveError(`Failed to send the command (Status ${getEnumMemberName(TransmitStatus, status)})`, status === TransmitStatus.NoAck
                    ? ZWaveErrorCodes.Controller_CallbackNOK
                    : ZWaveErrorCodes.Controller_MessageDropped, receivedMessage, transactionSource);
            }
            else if (sentMessage instanceof SendDataMulticastRequest
                || sentMessage instanceof SendDataMulticastBridgeRequest) {
                const status = receivedMessage.transmitStatus;
                return new ZWaveError(`One or more nodes did not respond to the multicast request (Status ${getEnumMemberName(TransmitStatus, status)})`, status === TransmitStatus.NoAck
                    ? ZWaveErrorCodes.Controller_CallbackNOK
                    : ZWaveErrorCodes.Controller_MessageDropped, receivedMessage, transactionSource);
            }
            else {
                return new ZWaveError(`The controller callback indicated failure`, ZWaveErrorCodes.Controller_CallbackNOK, receivedMessage, transactionSource);
            }
        }
    }
}
export function createMessageDroppedUnexpectedError(original) {
    const ret = new ZWaveError(`Message dropped because of an unexpected error: ${original.message}`, ZWaveErrorCodes.Controller_MessageDropped);
    if (original.stack)
        ret.stack = original.stack;
    return ret;
}
// TODO: Do we still need this?
// function computeRetryDelay(attempts: number): number {
// 	return 100 + 1000 * (attempts - 1);
// }
//# sourceMappingURL=StateMachineShared.js.map