import { CacheBackedMap, NOT_KNOWN, SecurityClass, securityClassOrder, } from "@zwave-js/core";
import { getEnumMemberName } from "@zwave-js/shared";
import { cacheKeys } from "../../driver/NetworkCache.js";
import { NetworkRoleMixin } from "./01_NetworkRole.js";
export class NodeSecurityMixin extends NetworkRoleMixin {
    constructor(nodeId, driver, index, deviceClass, supportedCCs) {
        super(nodeId, driver, index, deviceClass, supportedCCs);
        this.securityClasses = new CacheBackedMap(this.driver.networkCache, {
            prefix: cacheKeys.node(this.id)._securityClassBaseKey + ".",
            suffixSerializer: (value) => getEnumMemberName(SecurityClass, value),
            suffixDeserializer: (key) => {
                if (key in SecurityClass
                    && typeof SecurityClass[key] === "number") {
                    return SecurityClass[key];
                }
            },
        });
    }
    /** @internal */
    // This a CacheBackedMap that's assigned in the constructor
    securityClasses;
    get isSecure() {
        const securityClass = this.getHighestSecurityClass();
        if (securityClass == undefined)
            return NOT_KNOWN;
        if (securityClass === SecurityClass.None)
            return false;
        return true;
    }
    hasSecurityClass(securityClass) {
        return this.securityClasses.get(securityClass);
    }
    setSecurityClass(securityClass, granted) {
        this.securityClasses.set(securityClass, granted);
    }
    getHighestSecurityClass() {
        if (this.securityClasses.size === 0)
            return undefined;
        let missingSome = false;
        for (const secClass of securityClassOrder) {
            if (this.securityClasses.get(secClass) === true)
                return secClass;
            if (!this.securityClasses.has(secClass)) {
                missingSome = true;
            }
        }
        // If we don't have the info for every security class, we don't know the highest one yet
        return missingSome ? NOT_KNOWN : SecurityClass.None;
    }
    /**
     * Should be `true` when an S2-capable node fails to S2-bootstrap.
     * This will skip the interview process (see `Driver.interviewNodeInternal` method).
     */
    get failedS2Bootstrapping() {
        return this.driver.cacheGet(cacheKeys.node(this.id).failedS2Bootstrapping)
            ?? false;
    }
    set failedS2Bootstrapping(value) {
        this.driver.cacheSet(cacheKeys.node(this.id).failedS2Bootstrapping, value);
    }
}
//# sourceMappingURL=05_Security.js.map