import { MultiChannelCCValues } from "@zwave-js/cc";
import { ZWaveError, ZWaveErrorCodes, } from "@zwave-js/core";
import { isArray, isObject } from "alcalzone-shared/typeguards";
import { DeviceClass } from "../DeviceClass.js";
import { Endpoint } from "../Endpoint.js";
import * as nodeUtils from "../utils.js";
import { NodeCapabilityValuesMixin } from "./41_CapabilityValues.js";
export class EndpointsMixin extends NodeCapabilityValuesMixin {
    get endpointCountIsDynamic() {
        return nodeUtils.endpointCountIsDynamic(this.driver, this.id);
    }
    get endpointsHaveIdenticalCapabilities() {
        return nodeUtils.endpointsHaveIdenticalCapabilities(this.driver, this.id);
    }
    get individualEndpointCount() {
        return nodeUtils.getIndividualEndpointCount(this.driver, this.id);
    }
    get aggregatedEndpointCount() {
        return nodeUtils.getAggregatedEndpointCount(this.driver, this.id);
    }
    /** Returns the device class of an endpoint. Falls back to the node's device class if the information is not known. */
    getEndpointDeviceClass(index) {
        const deviceClass = this.getValue(MultiChannelCCValues.endpointDeviceClass.endpoint(this.endpointsHaveIdenticalCapabilities ? 1 : index));
        if (deviceClass && this.deviceClass) {
            return new DeviceClass(this.deviceClass.basic, deviceClass.generic, deviceClass.specific);
        }
        // fall back to the node's device class if it is known
        return this.deviceClass;
    }
    getEndpointCCs(index) {
        const ret = this.getValue(MultiChannelCCValues.endpointCCs.endpoint(this.endpointsHaveIdenticalCapabilities ? 1 : index));
        // Workaround for the change in #1977
        if (isArray(ret)) {
            // The value is set up correctly, return it
            return ret;
        }
        else if (isObject(ret) && "supportedCCs" in ret) {
            return ret.supportedCCs;
        }
    }
    /**
     * Returns the current endpoint count of this node.
     *
     * If you want to enumerate the existing endpoints, use `getEndpointIndizes` instead.
     * Some devices are known to contradict themselves.
     */
    getEndpointCount() {
        return nodeUtils.getEndpointCount(this.driver, this.id);
    }
    /**
     * Returns indizes of all endpoints on the node.
     */
    getEndpointIndizes() {
        return nodeUtils.getEndpointIndizes(this.driver, this.id);
    }
    /** Whether the Multi Channel CC has been interviewed and all endpoint information is known */
    get isMultiChannelInterviewComplete() {
        return nodeUtils.isMultiChannelInterviewComplete(this.driver, this.id);
    }
    /** Cache for this node's endpoint instances */
    _endpointInstances = new Map();
    getEndpoint(index) {
        if (index < 0) {
            throw new ZWaveError("The endpoint index must be positive!", ZWaveErrorCodes.Argument_Invalid);
        }
        // Zero is the root endpoint - i.e. this node. Also accept undefined if an application misbehaves
        if (!index)
            return this;
        // Check if the Multi Channel CC interview for this node is completed,
        // because we don't have all the information before that
        if (!this.isMultiChannelInterviewComplete) {
            this.driver.driverLog.print(`Node ${this.id}, Endpoint ${index}: Trying to access endpoint instance before Multi Channel interview`, "error");
            return undefined;
        }
        // Check if the endpoint index is in the list of known endpoint indizes
        if (!this.getEndpointIndizes().includes(index))
            return undefined;
        // Create an endpoint instance if it does not exist
        if (!this._endpointInstances.has(index)) {
            this._endpointInstances.set(index, new Endpoint(this.id, this.driver, index, this.getEndpointDeviceClass(index), this.getEndpointCCs(index)));
        }
        return this._endpointInstances.get(index);
    }
    getEndpointOrThrow(index) {
        const ret = this.getEndpoint(index);
        if (!ret) {
            throw new ZWaveError(`Endpoint ${index} does not exist on Node ${this.id}`, ZWaveErrorCodes.Controller_EndpointNotFound);
        }
        return ret;
    }
    /** Returns a list of all endpoints of this node, including the root endpoint (index 0) */
    getAllEndpoints() {
        return nodeUtils.getAllEndpoints(this.driver, this);
    }
}
//# sourceMappingURL=50_Endpoints.js.map