'use strict';
import { module } from "./logger.js";
import { Server as SocketServer } from 'socket.io';
import { TypedEventEmitter } from "./EventEmitter.js";
import { inboundEvents } from "./SocketEvents.js";
const logger = module('Socket');
/**
 * The constructor
 */
class SocketManager extends TypedEventEmitter {
    io;
    activeSockets = new Map();
    authMiddleware;
    /**
     * Binds socket.io to `server`
     *
     */
    bindServer(server) {
        this.io = new SocketServer(server, {
            path: '/socket.io',
        });
        this.io.on('error', (err) => {
            logger.error(`Socket error: ${err.message}`);
        });
        this.io
            .use(this._authMiddleware())
            .on('connection', this._onConnection.bind(this));
    }
    _authMiddleware() {
        return (socket, next) => {
            if (this.authMiddleware !== undefined) {
                this.authMiddleware(socket, next);
            }
            else {
                next();
            }
        };
    }
    /**
     * Handles new socket connections
     *
     */
    _onConnection(socket) {
        logger.debug(`New connection ${socket.id}`);
        // add socket to active sockets
        this.activeSockets.set(socket.id, socket);
        this.emit('clients', 'connection', this.activeSockets);
        // register inbound events from this socket
        for (const k in inboundEvents) {
            const eventName = inboundEvents[k];
            // pass socket reference as first parameter
            socket.on(eventName, this._emitEvent.bind(this, eventName, socket));
        }
        // https://socket.io/docs/v4/server-socket-instance/#events
        socket.on('disconnect', (reason) => {
            logger.debug(`User disconnected from ${socket.id}: ${reason}`);
            this.activeSockets.delete(socket.id);
            this.emit('clients', 'disconnect', this.activeSockets);
        });
    }
    /**
     * Logs and emits the `eventName` with `socket` and `args` as parameters
     *
     */
    _emitEvent(eventName, socket, data) {
        logger.debug(`Event ${eventName} emitted to ${socket.id}`);
        this.emit(eventName, socket, data);
    }
}
export default SocketManager;
