/**
 * Protobuf wire types
 */
export declare const enum WireType {
    Varint = 0,
    Fixed64 = 1,
    LengthDelimited = 2,
    StartGroup = 3,// Deprecated
    EndGroup = 4,// Deprecated
    Fixed32 = 5
}
/**
 * Helper functions for protobuf wire format encoding/decoding
 */
/**
 * Encodes a protobuf field tag (field number and wire type)
 */
export declare function encodeTag(fieldNumber: number, wireType: WireType): Uint8Array;
/**
 * Decodes a protobuf field tag
 */
export declare function decodeTag(data: Uint8Array, offset?: number): {
    fieldNumber: number;
    wireType: WireType;
    bytesRead: number;
};
/**
 * Encodes a VarInt according to the Protocol Buffers specification
 */
export declare function encodeVarInt(value: number): Uint8Array;
/**
 * Decodes a VarInt according to the Protocol Buffers specification
 * Returns the decoded value and the number of bytes consumed
 */
export declare function decodeVarInt(data: Uint8Array, offset?: number): {
    value: number;
    bytesRead: number;
};
/**
 * Encodes a string field
 */
export declare function encodeStringField(fieldNumber: number, value: string | Uint8Array): Uint8Array;
/**
 * Decodes a string field at the current position
 */
export declare function decodeStringField(data: Uint8Array, offset: number): {
    value: string;
    bytesRead: number;
};
/**
 * Encodes a varint field (uint32, int32, bool, enum)
 */
export declare function encodeVarintField(fieldNumber: number, value: number): Uint8Array;
/**
 * Encodes a boolean field
 */
export declare function encodeBoolField(fieldNumber: number, value: boolean): Uint8Array;
/**
 * Encodes a fixed32 field
 */
export declare function encodeFixed32Field(fieldNumber: number, value: number): Uint8Array;
/**
 * Decodes a fixed32 field at the current position
 */
export declare function decodeFixed32Field(data: Uint8Array, offset: number): {
    value: number;
    bytesRead: number;
};
/**
 * Encodes a length-delimited field (for nested messages)
 */
export declare function encodeLengthDelimitedField(fieldNumber: number, value: Uint8Array): Uint8Array;
/**
 * Decodes a length-delimited field at the current position
 */
export declare function decodeLengthDelimitedField(data: Uint8Array, offset: number): {
    value: Uint8Array;
    bytesRead: number;
};
/**
 * Encodes a signed integer using zigzag encoding
 */
export declare function encodeSignedVarint(value: number): Uint8Array;
/**
 * Decodes a signed integer using zigzag encoding
 */
export declare function decodeSignedVarint(data: Uint8Array, offset?: number): {
    value: number;
    bytesRead: number;
};
/**
 * Encodes a signed varint field
 */
export declare function encodeSignedVarintField(fieldNumber: number, value: number): Uint8Array;
/**
 * Skips an unknown field based on its wire type
 */
export declare function skipField(data: Uint8Array, offset: number, wireType: WireType): number;
/**
 * Helper function to concatenate Uint8Arrays
 */
export declare function concatenateUint8Arrays(arrays: Uint8Array[]): Uint8Array;
/**
 * Parses a protobuf message and calls a field handler for each field
 */
export declare function parseProtobufMessage(data: Uint8Array, fieldHandler: (fieldNumber: number, wireType: WireType, data: Uint8Array, offset: number) => number): void;
//# sourceMappingURL=ProtobufHelpers.d.ts.map