import { CRC16_CCITT, ZWaveError, ZWaveErrorCodes, ZnifferProtocolDataRate, getZWaveChipType, validatePayload, } from "@zwave-js/core";
import { Bytes, getEnumMemberName } from "@zwave-js/shared";
import { ZnifferFrameType, ZnifferFunctionType, ZnifferMessageType, } from "./Constants.js";
export class ZnifferMessageRaw {
    type;
    functionType;
    payload;
    constructor(type, functionType, payload) {
        this.type = type;
        this.functionType = functionType;
        this.payload = payload;
    }
    static parse(data) {
        if (data.length < 3) {
            throw new ZWaveError("Incomplete Zniffer message", ZWaveErrorCodes.PacketFormat_Truncated);
        }
        const type = data[0];
        if (type === ZnifferMessageType.Command) {
            const functionType = data[1];
            const length = data[2];
            const payload = Bytes.view(data.subarray(3, 3 + length));
            return {
                raw: new ZnifferMessageRaw(type, functionType, payload),
                bytesRead: 3 + length,
            };
        }
        else if (type === ZnifferMessageType.Data) {
            // For frames received from the Zniffer hardware itself,
            // the ZnifferParser ensures we always get complete frames.
            // However, when reading from a ZLF file, especially the ones
            // written by the Zniffer application, data frames
            // don't necessarily align with ZLF frame boundaries.
            // We do some light inspection to figure out how many bytes we
            // need to read for a complete frame.
            let totalLength;
            // Byte 1 contains the frame type
            if (data.length < 2) {
                throw new ZWaveError("Incomplete Zniffer message", ZWaveErrorCodes.PacketFormat_Truncated);
            }
            const frameType = data[1];
            if (frameType === ZnifferFrameType.BeamStop) {
                // BeamStop frames always seem to be 7 bytes long, including the type byte
                totalLength = 7;
            }
            else {
                // The remaining length is stored at byte 9
                if (data.length < 10) {
                    throw new ZWaveError("Incomplete Zniffer message", ZWaveErrorCodes.PacketFormat_Truncated);
                }
                totalLength = data[9] + 10;
            }
            if (data.length < totalLength) {
                throw new ZWaveError("Incomplete Zniffer message", ZWaveErrorCodes.PacketFormat_Truncated);
            }
            // Cut off the type byte from the payload
            const payload = Bytes.view(data.subarray(1, totalLength));
            return {
                raw: new ZnifferMessageRaw(type, undefined, payload),
                bytesRead: totalLength,
            };
        }
        else {
            throw new ZWaveError(`Invalid Zniffer message type ${type}`, ZWaveErrorCodes.PacketFormat_InvalidPayload);
        }
    }
    withPayload(payload) {
        return new ZnifferMessageRaw(this.type, this.functionType, payload);
    }
}
/**
 * Retrieves the correct constructor for the next message in the given Buffer.
 * It is assumed that the buffer has been checked beforehand
 */
function getZnifferMessageConstructor(raw) {
    // We hardcode the list of constructors here, since the Zniffer protocol has
    // a very limited list of messages
    if (raw.type === ZnifferMessageType.Command) {
        switch (raw.functionType) {
            case ZnifferFunctionType.GetVersion:
                return ZnifferGetVersionResponse;
            case ZnifferFunctionType.SetFrequency:
                return ZnifferSetFrequencyResponse;
            case ZnifferFunctionType.GetFrequencies:
                return ZnifferGetFrequenciesResponse;
            case ZnifferFunctionType.Start:
                return ZnifferStartResponse;
            case ZnifferFunctionType.Stop:
                return ZnifferStopResponse;
            case ZnifferFunctionType.SetLRChannelConfig:
                return ZnifferSetLRChannelConfigResponse;
            case ZnifferFunctionType.GetLRChannelConfigs:
                return ZnifferGetLRChannelConfigsResponse;
            case ZnifferFunctionType.GetLRRegions:
                return ZnifferGetLRRegionsResponse;
            case ZnifferFunctionType.SetBaudRate:
                return ZnifferSetBaudRateResponse;
            case ZnifferFunctionType.GetFrequencyInfo:
                return ZnifferGetFrequencyInfoResponse;
            default:
                return ZnifferMessage;
        }
    }
    else if (raw.type === ZnifferMessageType.Data) {
        return ZnifferDataMessage;
    }
    else {
        return ZnifferMessage;
    }
}
/**
 * Represents a Zniffer message for communication with the serial interface
 */
export class ZnifferMessage {
    constructor(options) {
        this.type = options.type;
        this.functionType = options.functionType;
        this.payload = options.payload || new Bytes();
    }
    static parse(data) {
        const { raw, bytesRead } = ZnifferMessageRaw.parse(data);
        const Constructor = getZnifferMessageConstructor(raw);
        return {
            msg: Constructor.from(raw),
            bytesRead,
        };
    }
    /** Creates an instance of the message that is serialized in the given buffer */
    static from(raw) {
        return new this({
            type: raw.type,
            functionType: raw.functionType,
            payload: raw.payload,
        });
    }
    type;
    functionType;
    payload;
    /** Serializes this message into a Buffer */
    serialize() {
        if (this.type === ZnifferMessageType.Command) {
            return Bytes.concat([
                Bytes.from([
                    this.type,
                    this.functionType,
                    this.payload.length,
                ]),
                this.payload,
            ]);
        }
        else if (this.type === ZnifferMessageType.Data) {
            const ret = Bytes.concat([
                [this.type],
                this.payload,
            ]);
            ret[9] = this.payload.length - 10;
            return ret;
        }
        else {
            throw new ZWaveError(`Invalid Zniffer message type ${this.type}`, ZWaveErrorCodes.PacketFormat_InvalidPayload);
        }
    }
}
function computeChecksumXOR(buffer) {
    let ret = 0xff;
    for (let i = 0; i < buffer.length; i++) {
        ret ^= buffer[i];
    }
    return ret;
}
export class ZnifferDataMessage extends ZnifferMessage {
    constructor(options) {
        super({
            type: ZnifferMessageType.Data,
            payload: options.payload,
        });
        this.frameType = options.frameType;
        this.channel = options.channel;
        this.protocolDataRate = options.protocolDataRate;
        this.region = options.region;
        this.rssiRaw = options.rssiRaw;
        this.checksumOK = options.checksumOK;
    }
    static from(raw) {
        if (raw.payload.length < 6) {
            throw new ZWaveError("Incomplete Zniffer message", ZWaveErrorCodes.PacketFormat_Truncated);
        }
        const frameType = raw.payload[0];
        // bytes 1-2 are 0
        const channel = raw.payload[3] >>> 5;
        const protocolDataRate = raw.payload[3]
            & 0b11111;
        const checksumLength = protocolDataRate >= ZnifferProtocolDataRate.ZWave_100k
            ? 2
            : 1;
        const region = raw.payload[4];
        const rssiRaw = raw.payload[5];
        let checksumOK;
        let payload;
        if (frameType === ZnifferFrameType.Data) {
            if (raw.payload.length < 9 + checksumLength) {
                throw new ZWaveError("Incomplete Zniffer message", ZWaveErrorCodes.PacketFormat_Truncated);
            }
            validatePayload.withReason(`ZnifferDataMessage[6] = ${raw.payload[6]}`)(raw.payload[6] === 0x21);
            validatePayload.withReason(`ZnifferDataMessage[7] = ${raw.payload[7]}`)(raw.payload[7] === 0x03);
            const remainingLength = raw.payload[8];
            const mpduOffset = 9;
            if (raw.payload.length < mpduOffset + remainingLength) {
                throw new ZWaveError("Incomplete Zniffer message", ZWaveErrorCodes.PacketFormat_Truncated);
            }
            const mpdu = raw.payload.subarray(mpduOffset, mpduOffset + remainingLength - checksumLength);
            const checksum = raw.payload.readUIntBE(mpduOffset + remainingLength - checksumLength, checksumLength);
            // Compute checksum over the entire MPDU
            const expectedChecksum = checksumLength === 1
                ? computeChecksumXOR(mpdu)
                : CRC16_CCITT(mpdu);
            checksumOK = checksum === expectedChecksum;
            payload = mpdu;
        }
        else if (frameType === ZnifferFrameType.BeamStart) {
            if (raw.payload.length < 7) {
                throw new ZWaveError("Incomplete Zniffer message", ZWaveErrorCodes.PacketFormat_Truncated);
            }
            validatePayload.withReason(`ZnifferDataMessage[6] = ${raw.payload[6]}`)(raw.payload[6] === 0x55);
            // There is no checksum
            checksumOK = true;
            payload = raw.payload.subarray(6);
        }
        else if (frameType === ZnifferFrameType.BeamStop) {
            // This always seems to contain the same 2 bytes
            // There is no checksum
            checksumOK = true;
            payload = new Bytes();
        }
        else {
            validatePayload.fail(`Unsupported frame type ${getEnumMemberName(ZnifferFrameType, frameType)}`);
        }
        return new this({
            frameType,
            channel,
            protocolDataRate,
            region,
            rssiRaw,
            payload,
            checksumOK,
        });
    }
    frameType;
    channel;
    protocolDataRate;
    region;
    rssiRaw;
    checksumOK;
}
export class ZnifferGetVersionRequest extends ZnifferMessage {
    constructor() {
        super({
            type: ZnifferMessageType.Command,
            functionType: ZnifferFunctionType.GetVersion,
        });
    }
}
export class ZnifferGetVersionResponse extends ZnifferMessage {
    constructor(options) {
        super({
            type: ZnifferMessageType.Command,
            functionType: ZnifferFunctionType.GetVersion,
        });
        this.chipType = options.chipType;
        this.majorVersion = options.majorVersion;
        this.minorVersion = options.minorVersion;
    }
    static from(raw) {
        const chipType = getZWaveChipType(raw.payload[0], raw.payload[1]);
        const majorVersion = raw.payload[2];
        const minorVersion = raw.payload[3];
        return new this({
            chipType,
            majorVersion,
            minorVersion,
        });
    }
    chipType;
    majorVersion;
    minorVersion;
}
export class ZnifferSetFrequencyRequest extends ZnifferMessage {
    constructor(options) {
        super({
            type: ZnifferMessageType.Command,
            functionType: ZnifferFunctionType.SetFrequency,
        });
        this.frequency = options.frequency;
    }
    frequency;
    serialize() {
        this.payload = Bytes.from([this.frequency]);
        return super.serialize();
    }
}
export class ZnifferSetFrequencyResponse extends ZnifferMessage {
}
export class ZnifferGetFrequenciesRequest extends ZnifferMessage {
    constructor() {
        super({
            type: ZnifferMessageType.Command,
            functionType: ZnifferFunctionType.GetFrequencies,
        });
    }
}
export class ZnifferGetFrequenciesResponse extends ZnifferMessage {
    constructor(options) {
        super({
            type: ZnifferMessageType.Command,
            functionType: ZnifferFunctionType.GetFrequencies,
        });
        this.currentFrequency = options.currentFrequency;
        this.supportedFrequencies = options.supportedFrequencies;
    }
    static from(raw) {
        const currentFrequency = raw.payload[0];
        const supportedFrequencies = [
            ...raw.payload.subarray(1),
        ];
        return new this({
            currentFrequency,
            supportedFrequencies,
        });
    }
    currentFrequency;
    supportedFrequencies;
}
export class ZnifferStartRequest extends ZnifferMessage {
    constructor() {
        super({
            type: ZnifferMessageType.Command,
            functionType: ZnifferFunctionType.Start,
        });
    }
}
export class ZnifferStartResponse extends ZnifferMessage {
}
export class ZnifferStopRequest extends ZnifferMessage {
    constructor() {
        super({
            type: ZnifferMessageType.Command,
            functionType: ZnifferFunctionType.Stop,
        });
    }
}
export class ZnifferStopResponse extends ZnifferMessage {
}
export class ZnifferSetLRChannelConfigRequest extends ZnifferMessage {
    constructor(options) {
        super({
            type: ZnifferMessageType.Command,
            functionType: ZnifferFunctionType.SetLRChannelConfig,
        });
        this.channelConfig = options.channelConfig;
    }
    channelConfig;
    serialize() {
        this.payload = Bytes.from([this.channelConfig]);
        return super.serialize();
    }
}
export class ZnifferSetLRChannelConfigResponse extends ZnifferMessage {
}
export class ZnifferGetLRChannelConfigsRequest extends ZnifferMessage {
    constructor() {
        super({
            type: ZnifferMessageType.Command,
            functionType: ZnifferFunctionType.GetLRChannelConfigs,
        });
    }
}
export class ZnifferGetLRChannelConfigsResponse extends ZnifferMessage {
    constructor(options) {
        super({
            type: ZnifferMessageType.Command,
            functionType: ZnifferFunctionType.GetLRChannelConfigs,
        });
        this.currentConfig = options.currentConfig;
        this.supportedConfigs = options.supportedConfigs;
    }
    static from(raw) {
        const currentConfig = raw.payload[0];
        const supportedConfigs = [
            ...raw.payload.subarray(1),
        ];
        return new this({
            currentConfig,
            supportedConfigs,
        });
    }
    currentConfig;
    supportedConfigs;
}
export class ZnifferGetLRRegionsRequest extends ZnifferMessage {
    constructor() {
        super({
            type: ZnifferMessageType.Command,
            functionType: ZnifferFunctionType.GetLRRegions,
        });
    }
}
export class ZnifferGetLRRegionsResponse extends ZnifferMessage {
    constructor(options) {
        super({
            type: ZnifferMessageType.Command,
            functionType: ZnifferFunctionType.GetLRRegions,
        });
        this.regions = options.regions;
    }
    static from(raw) {
        return new this({
            regions: [...raw.payload],
        });
    }
    regions;
}
export class ZnifferSetBaudRateRequest extends ZnifferMessage {
    constructor(options) {
        super({
            type: ZnifferMessageType.Command,
            functionType: ZnifferFunctionType.SetBaudRate,
        });
        this.baudrate = options.baudrate;
    }
    baudrate;
    serialize() {
        this.payload = Bytes.from([this.baudrate]);
        return super.serialize();
    }
}
export class ZnifferSetBaudRateResponse extends ZnifferMessage {
}
export class ZnifferGetFrequencyInfoRequest extends ZnifferMessage {
    constructor(options) {
        super({
            type: ZnifferMessageType.Command,
            functionType: ZnifferFunctionType.GetFrequencyInfo,
        });
        this.frequency = options.frequency;
    }
    frequency;
    serialize() {
        this.payload = Bytes.from([this.frequency]);
        return super.serialize();
    }
}
export class ZnifferGetFrequencyInfoResponse extends ZnifferMessage {
    constructor(options) {
        super({
            type: ZnifferMessageType.Command,
            functionType: ZnifferFunctionType.GetFrequencyInfo,
        });
        this.frequency = options.frequency;
        this.numChannels = options.numChannels;
        this.frequencyName = options.frequencyName;
    }
    static from(raw) {
        const frequency = raw.payload[0];
        const numChannels = raw.payload[1];
        const frequencyName = raw.payload
            .subarray(2)
            .toString("ascii");
        return new this({
            frequency,
            numChannels,
            frequencyName,
        });
    }
    frequency;
    numChannels;
    frequencyName;
}
export class ZnifferGetLRChannelConfigInfoRequest extends ZnifferMessage {
    constructor(options) {
        super({
            type: ZnifferMessageType.Command,
            functionType: ZnifferFunctionType.GetLRChannelConfigInfo,
        });
        this.channelConfig = options.channelConfig;
    }
    channelConfig;
    serialize() {
        this.payload = Bytes.from([this.channelConfig]);
        return super.serialize();
    }
}
export class ZnifferGetLRChannelConfigInfoResponse extends ZnifferMessage {
    constructor(options) {
        super({
            type: ZnifferMessageType.Command,
            functionType: ZnifferFunctionType.GetLRChannelConfigInfo,
        });
        this.channelConfig = options.channelConfig;
        this.numChannels = options.numChannels;
        this.configName = options.configName;
    }
    static from(raw) {
        const channelConfig = raw.payload[0];
        const numChannels = raw.payload[1];
        const configName = raw.payload
            .subarray(2)
            .toString("ascii");
        return new this({
            channelConfig,
            numChannels,
            configName,
        });
    }
    channelConfig;
    numChannels;
    configName;
}
//# sourceMappingURL=ZnifferMessages.js.map