import { type LogConfig, ZnifferLRChannelConfig } from "@zwave-js/core";
import { type ZWaveSerialBindingFactory, type ZWaveSerialPortImplementation } from "@zwave-js/serial";
import { TypedEventTarget } from "@zwave-js/shared";
import type { ZWaveOptions } from "../driver/ZWaveOptions.js";
import { type CorruptedFrame, type Frame } from "./MPDU.js";
export interface ZnifferEventCallbacks {
    ready: () => void;
    error: (err: Error) => void;
    frame: (frame: Frame, rawData: Uint8Array) => void;
    "corrupted frame": (err: CorruptedFrame, rawData: Uint8Array) => void;
}
export type ZnifferEvents = Extract<keyof ZnifferEventCallbacks, string>;
export interface ZnifferOptions {
    /**
     * Optional log configuration
     */
    logConfig?: Partial<LogConfig>;
    /** Security keys for decrypting Z-Wave traffic */
    securityKeys?: ZWaveOptions["securityKeys"];
    /** Security keys for decrypting Z-Wave Long Range traffic */
    securityKeysLongRange?: ZWaveOptions["securityKeysLongRange"];
    host?: ZWaveOptions["host"];
    /**
     * The RSSI values reported by the Zniffer are not actual RSSI values.
     * They can be converted to dBm, but the conversion is chip dependent and not documented for 700/800 series Zniffers.
     *
     * Set this option to `true` enable the conversion. Otherwise the raw values from the Zniffer will be used.
     */
    convertRSSI?: boolean;
    /**
     * The frequency to initialize the Zniffer with. If not specified, the current setting will be kept.
     *
     * On 700/800 series Zniffers, this value matches the {@link ZnifferRegion}.
     *
     * On 400/500 series Zniffers, the value is firmware-specific.
     * Supported regions and their names have to be queried using the `getFrequencies` and `getFrequencyInfo(frequency)` commands.
     */
    defaultFrequency?: number;
    /**
     * The LR channel configuration to initialize the Zniffer with. If not specified, the current setting will be kept.
     *
     * This is only supported for 800 series Zniffers with LR support
     */
    defaultLRChannelConfig?: ZnifferLRChannelConfig;
    /** Limit the number of frames that are kept in memory. */
    maxCapturedFrames?: number;
}
export interface CapturedData {
    timestamp: Date;
    rawData: Uint8Array;
    frameData: Uint8Array;
    parsedFrame?: Frame | CorruptedFrame;
}
export interface CapturedFrame {
    timestamp: Date;
    frameData: Uint8Array;
    parsedFrame: Frame | CorruptedFrame;
}
export declare class Zniffer extends TypedEventTarget<ZnifferEventCallbacks> {
    private port;
    constructor(port: string | ZWaveSerialPortImplementation | ZWaveSerialBindingFactory, options?: ZnifferOptions);
    private _options;
    /**
     * The host bindings used to access file system etc.
     */
    private bindings;
    private serialFactory;
    /** The serial port instance */
    private serial;
    private parsingContext;
    private _destroyPromise;
    private get wasDestroyed();
    private _chipType;
    private _currentFrequency;
    /** The currently configured frequency */
    get currentFrequency(): number | undefined;
    private _supportedFrequencies;
    /** A map of supported frequency identifiers and their names */
    get supportedFrequencies(): ReadonlyMap<number, string>;
    private _lrRegions;
    /** A list regions that are Long Range capable */
    get lrRegions(): ReadonlySet<number>;
    private _currentLRChannelConfig;
    /** The currently configured Long Range channel configuration */
    get currentLRChannelConfig(): number | undefined;
    private _supportedLRChannelConfigs;
    /** A map of supported Long Range channel configurations and their names */
    get supportedLRChannelConfigs(): ReadonlyMap<number, string>;
    private _logContainer;
    private znifferLog;
    /** The security managers for each node */
    private securityManagers;
    /** A list of awaited messages */
    private awaitedMessages;
    private _active;
    /** Whether the Zniffer instance is currently capturing */
    get active(): boolean;
    private _capturedFrames;
    /** A list of raw captured frames that can be saved to a .zlf file later */
    get capturedFrames(): Readonly<CapturedFrame>[];
    init(): Promise<void>;
    private handleSerialData;
    /**
     * Is called when the serial port has received a Zniffer frame
     */
    private serialport_onData;
    /**
     * Is called when a Request-type message was received
     */
    private handleResponse;
    /**
     * Is called when a Request-type message was received
     */
    private handleDataMessage;
    /**
     * Waits until a certain serial message is received or a timeout has elapsed. Returns the received message.
     * @param timeout The number of milliseconds to wait. If the timeout elapses, the returned promise will be rejected
     * @param predicate A predicate function to test all incoming messages.
     */
    private waitForMessage;
    private getVersion;
    private getFrequencies;
    setFrequency(frequency: number): Promise<void>;
    private getFrequencyInfo;
    private getLRRegions;
    private getLRChannelConfigs;
    setLRChannelConfig(channelConfig: number): Promise<void>;
    private getLRChannelConfigInfo;
    /** Starts the capture and discards all previously captured frames */
    start(): Promise<void>;
    stop(): Promise<void>;
    private setBaudrate;
    private getSecurityManagers;
    /** Clears the list of captured frames */
    clearCapturedFrames(): void;
    /**
     * Get the captured frames in the official Zniffer application format.
     * @param frameFilter Optional predicate function to filter the frames included in the capture
     */
    getCaptureAsZLFBuffer(frameFilter?: (frame: CapturedFrame) => boolean): Uint8Array;
    /**
     * Saves the captured frames in a `.zlf` file that can be read by the official Zniffer application.
     * @param frameFilter Optional predicate function to filter the frames included in the capture
     */
    saveCaptureToFile(filePath: string, frameFilter?: (frame: CapturedFrame) => boolean): Promise<void>;
    /**
     * Terminates the Zniffer instance and closes the underlying serial connection.
     * Must be called under any circumstances.
     */
    destroy(): Promise<void>;
    /**
     * Loads captured frames from a `.zlf` file that was written by the official Zniffer application or Z-Wave JS.
     */
    loadCaptureFromFile(filePath: string): Promise<void>;
    /**
     * Load captured frames from a buffer
     */
    loadCaptureFromBuffer(buffer: Uint8Array): Promise<void>;
    private parseFrame;
}
//# sourceMappingURL=Zniffer.d.ts.map