import { type CCAPIHost, CommandClass, type InterviewContext, type InterviewOptions, type PersistValuesContext, type Powerlevel, type RefreshValueTimeouts, type RefreshValuesContext, type SchedulePollOptions, type UserPreferences } from "@zwave-js/cc";
import { ConfigManager, type DeviceConfig } from "@zwave-js/config";
import { type CCId, CommandClasses, type LogConfig, type LogNodeOptions, type MaybeNotKnown, SecurityClass, SecurityManager, SecurityManager2, type SendCommandOptions, type SendCommandReturnType, type SendMessageOptions, TransmitStatus, type ValueDB, type ValueID } from "@zwave-js/core";
import { type BootloaderChunk, type CLIChunk, FunctionType, Message, MessageHeaders, type ZWaveSerialBindingFactory, type ZWaveSerialPortImplementation } from "@zwave-js/serial";
import { type ContainsCC, SendDataBridgeRequest, type SendDataMessage, SendDataMulticastBridgeRequest, SendDataMulticastRequest, SendDataRequest } from "@zwave-js/serial/serialapi";
import { type Timer, TypedEventTarget } from "@zwave-js/shared";
import { ZWaveController } from "../controller/Controller.js";
import { type FirmwareUpdateInfo } from "../controller/_Types.js";
import type { Endpoint } from "../node/Endpoint.js";
import type { ZWaveNode } from "../node/Node.js";
import { type ZWaveNodeEventCallbacks } from "../node/_Types.js";
import { type AppInfo } from "../telemetry/statistics.js";
import { Bootloader } from "./Bootloader.js";
import { DriverMode } from "./DriverMode.js";
import { EndDeviceCLI } from "./EndDeviceCLI.js";
import { TaskScheduler } from "./Task.js";
import { Transaction } from "./Transaction.js";
import type { EditableZWaveOptions, PartialZWaveOptions, ZWaveOptions } from "./ZWaveOptions.js";
import { type OTWFirmwareUpdateProgress, type OTWFirmwareUpdateResult } from "./_Types.js";
export declare const libVersion: string;
export declare const libName: string;
/**
 * Function signature for a message handler. The return type signals if the
 * message was handled (`true`) or further handlers should be called (`false`)
 */
export type RequestHandler<T extends Message = Message> = (msg: T) => boolean | Promise<boolean>;
interface AwaitedThing<T> {
    handler: (thing: T) => void;
    timeout?: Timer;
    predicate: (msg: T) => boolean;
    refreshPredicate?: (msg: T) => boolean;
}
export type AwaitedBootloaderChunkEntry = AwaitedThing<BootloaderChunk>;
type PrefixedNodeEvents = {
    [K in keyof ZWaveNodeEventCallbacks as K extends string ? `node ${K}` : never]: ZWaveNodeEventCallbacks[K];
};
export interface DriverEventCallbacks extends PrefixedNodeEvents {
    "driver ready": () => void;
    "bootloader ready": () => void;
    "cli ready": () => void;
    "all nodes ready": () => void;
    "firmware update progress": (progress: OTWFirmwareUpdateProgress) => void;
    "firmware update finished": (result: OTWFirmwareUpdateResult) => void;
    error: (err: Error) => void;
}
export type DriverEvents = Extract<keyof DriverEventCallbacks, string>;
/**
 * The driver is the core of this library. It controls the serial interface,
 * handles transmission and receipt of messages and manages the network cache.
 * Any action you want to perform on the Z-Wave network must go through a driver
 * instance or its associated nodes.
 */
export declare class Driver extends TypedEventTarget<DriverEventCallbacks> implements CCAPIHost, InterviewContext, RefreshValuesContext, PersistValuesContext {
    private port;
    constructor(port: string | ZWaveSerialPortImplementation | ZWaveSerialBindingFactory, ...optionsAndPresets: (PartialZWaveOptions | undefined)[]);
    private serialFactory;
    /** The serial port instance */
    private serial;
    private messageEncodingContext;
    private getEncodingContext;
    private getMessageParsingContext;
    private getCCParsingContext;
    private queue;
    private immediateQueue;
    private serialAPIQueue;
    /** Gives access to the transaction queues, ordered by priority */
    private get queues();
    private initTransactionQueues;
    private destroyTransactionQueues;
    private _scheduler;
    get scheduler(): TaskScheduler;
    private queuePaused;
    /** Used to immediately abort ongoing Serial API commands */
    private abortSerialAPICommand;
    private initSerialAPIQueue;
    private destroySerialAPIQueue;
    private _queuesBusyFlags;
    private _queueIdle;
    /** Whether the queue is currently idle */
    get queueIdle(): boolean;
    private set queueIdle(value);
    /** A map of handlers for all sorts of requests */
    private requestHandlers;
    /** A list of awaited message headers */
    private awaitedMessageHeaders;
    /** A list of awaited messages */
    private awaitedMessages;
    /** A list of awaited commands */
    private awaitedCommands;
    /** A list of awaited chunks from the bootloader */
    private awaitedBootloaderChunks;
    /** A list of awaited chunks from the end device CLI */
    private awaitedCLIChunks;
    /** A map of Node ID -> ongoing sessions */
    private nodeSessions;
    private ensureNodeSessions;
    private _requestStorage;
    readonly cacheDir: string;
    private _valueDB;
    private _metadataDB;
    private _networkCache;
    private _configManager;
    get configManager(): ConfigManager;
    get configVersion(): string;
    private _logContainer;
    private _driverLog;
    private _controllerLog;
    logNode(nodeId: number, message: string, level?: LogNodeOptions["level"]): void;
    logNode(nodeId: number, options: LogNodeOptions): void;
    private _controller;
    /** Encapsulates information about the Z-Wave controller and provides access to its nodes */
    get controller(): ZWaveController;
    /** While in bootloader mode, this encapsulates information about the bootloader and its state */
    private _bootloader;
    get bootloader(): Bootloader;
    private _cli;
    /** While in end device CLI mode, this encapsulates information about the CLI and its state */
    get cli(): EndDeviceCLI;
    /** Determines which kind of Z-Wave application the driver is currently communicating with */
    get mode(): DriverMode;
    private _recoveryPhase;
    private _securityManager;
    /**
     * **!!! INTERNAL !!!**
     *
     * Not intended to be used by applications
     */
    get securityManager(): SecurityManager | undefined;
    private _securityManager2;
    /**
     * **!!! INTERNAL !!!**
     *
     * Not intended to be used by applications
     */
    get securityManager2(): SecurityManager2 | undefined;
    private _securityManagerLR;
    /**
     * **!!! INTERNAL !!!**
     *
     * Not intended to be used by applications
     */
    get securityManagerLR(): SecurityManager2 | undefined;
    private _learnModeAuthenticatedKeyPair;
    /**
     * **!!! INTERNAL !!!**
     *
     * Not intended to be used by applications. Use `controller.homeId` instead!
     */
    get homeId(): number;
    /**
     * **!!! INTERNAL !!!**
     *
     * Not intended to be used by applications. Use `controller.ownNodeId` instead!
     */
    get ownNodeId(): number;
    tryGetNode(msg: Message): ZWaveNode | undefined;
    tryGetEndpoint(cc: CommandClass): Endpoint | undefined;
    /**
     * **!!! INTERNAL !!!**
     *
     * Not intended to be used by applications
     */
    getValueDB(nodeId: number): ValueDB;
    /**
     * **!!! INTERNAL !!!**
     *
     * Not intended to be used by applications
     */
    tryGetValueDB(nodeId: number): ValueDB | undefined;
    getDeviceConfig(nodeId: number): DeviceConfig | undefined;
    lookupManufacturer(manufacturerId: number): string | undefined;
    getHighestSecurityClass(nodeId: number): MaybeNotKnown<SecurityClass>;
    hasSecurityClass(nodeId: number, securityClass: SecurityClass): MaybeNotKnown<boolean>;
    /**
     * **!!! INTERNAL !!!**
     *
     * Not intended to be used by applications
     */
    setSecurityClass(nodeId: number, securityClass: SecurityClass, granted: boolean): void;
    /** Updates the logging configuration without having to restart the driver. */
    updateLogConfig(config: Partial<LogConfig>): void;
    /** Returns the current logging configuration. */
    getLogConfig(): LogConfig;
    /** Updates the preferred sensor scales to use for node queries */
    setPreferredScales(scales: ZWaveOptions["preferences"]["scales"]): void;
    /**
     * **!!! INTERNAL !!!**
     *
     * Not intended to be used by applications
     */
    getUserPreferences(): UserPreferences;
    /**
     * **!!! INTERNAL !!!**
     *
     * Not intended to be used by applications
     */
    getInterviewOptions(): InterviewOptions;
    /**
     * **!!! INTERNAL !!!**
     *
     * Not intended to be used by applications
     */
    getRefreshValueTimeouts(): RefreshValueTimeouts;
    /**
     * Enumerates all existing serial ports.
     * @param local Whether to include local serial ports
     * @param remote Whether to discover remote serial ports using an mDNS query for the `_zwave._tcp` domain
     */
    static enumerateSerialPorts({ local, remote, }?: {
        local?: boolean;
        remote?: boolean;
    }): Promise<string[]>;
    /** Updates a subset of the driver options on the fly */
    updateOptions(options: EditableZWaveOptions): void;
    private _options;
    get options(): Readonly<ZWaveOptions>;
    /**
     * The host bindings used to access file system etc.
     */
    private bindings;
    private _wasStarted;
    private _isOpen;
    /** Start the driver */
    start(): Promise<void>;
    private detectMode;
    private _controllerInterviewed;
    private _nodesReady;
    private _nodesReadyEventEmitted;
    private _isOpeningSerialPort;
    private openSerialport;
    /** Indicates whether all nodes are ready, i.e. the "all nodes ready" event has been emitted */
    get allNodesReady(): boolean;
    private getJsonlDBOptions;
    private initNetworkCache;
    private initValueDBs;
    private performCacheMigration;
    /**
     * Initializes the variables for controller and nodes,
     * adds event handlers and starts the interview process.
     */
    private initializeControllerAndNodes;
    private autoRefreshNodeValueTimers;
    private retryNodeInterviewTimeouts;
    /** Adds the necessary event handlers for a node instance */
    private addNodeEventHandlers;
    /** Removes a node's event handlers that were added with addNodeEventHandlers */
    private removeNodeEventHandlers;
    /** Is called when a node wakes up */
    private onNodeWakeUp;
    /** Is called when a node goes to sleep */
    private onNodeSleep;
    /** Is called when a previously dead node starts communicating again */
    private onNodeAlive;
    /** Is called when a node is marked as dead */
    private onNodeDead;
    /** Is called when a node is ready to be used */
    private onNodeReady;
    /** Checks if all nodes are ready and emits the "all nodes ready" event if they are */
    private checkAllNodesReady;
    private _statisticsEnabled;
    /** Whether reporting usage statistics is currently enabled */
    get statisticsEnabled(): boolean;
    private statisticsAppInfo;
    private userAgentComponents;
    /**
     * Updates individual components of the user agent. Versions for individual applications can be added or removed.
     * @param components An object with application/module/component names and their versions. Set a version to `null` or `undefined` explicitly to remove it from the user agent.
     */
    updateUserAgent(components: Record<string, string | null | undefined>): void;
    /**
     * Returns the effective user agent string for the given components.
     * The driver name and version is automatically prepended and the statisticsAppInfo data is automatically appended if no components were given.
     */
    private getEffectiveUserAgentString;
    private _userAgent;
    /** Returns the user agent string used for service requests */
    get userAgent(): string;
    /** Returns the user agent string combined with the additional components (if given) */
    getUserAgentStringWithComponents(components?: Record<string, string | null | undefined>): string;
    /**
     * Enable sending usage statistics. Although this does not include any sensitive information, we expect that you
     * inform your users before enabling statistics.
     */
    enableStatistics(appInfo: Pick<AppInfo, "applicationName" | "applicationVersion">): void;
    /**
     * Disable sending usage statistics
     */
    disableStatistics(): void;
    private statisticsTimeout;
    private compileAndSendStatistics;
    /** Is called when a node interview is completed */
    private onNodeInterviewCompleted;
    /** This is called when a new node was found and is being added to the network */
    private onNodeFound;
    /** This is called when a new node has been added to the network */
    private onNodeAdded;
    /** This is called when a node was removed from the network */
    private onNodeRemoved;
    private onControllerStatusChanged;
    private onNetworkFound;
    private onNetworkJoined;
    private onNetworkLeft;
    private recreateNetworkCacheAndValueDBs;
    /**
     * Returns the time in seconds to actually wait after a firmware upgrade, depending on what the device said.
     * This number will always be a bit greater than the advertised duration, because devices have been found to take longer to actually reboot.
     */
    getConservativeWaitTimeAfterFirmwareUpdate(advertisedWaitTime: number | undefined): number;
    /** This is called when the firmware on one of a node's firmware targets was updated */
    private onNodeFirmwareUpdated;
    /** This is called when a node emits a `"notification"` event */
    private onNodeNotification;
    /** Checks if there are any pending messages for the given node */
    private hasPendingMessages;
    /** Checks if there are any pending transactions that match the given predicate */
    hasPendingTransactions(predicate: (t: Transaction) => boolean): boolean;
    /**
     * Retrieves the maximum version of a command class the given endpoint supports.
     * Returns 0 when the CC is not supported. Also returns 0 when the node was not found.
     * Falls back to querying the root endpoint if an endpoint was not found on the node
     *
     * @param cc The command class whose version should be retrieved
     * @param nodeId The node for which the CC version should be retrieved
     * @param endpointIndex The endpoint in question
     */
    getSupportedCCVersion(cc: CommandClasses, nodeId: number, endpointIndex?: number): number;
    /**
     * Retrieves the maximum version of a command class that can be used to communicate with a node.
     * Returns the highest implemented version if the node's CC version is unknown.
     * Returns `undefined` for CCs that are not implemented in this library yet.
     *
     * @param cc The command class whose version should be retrieved
     * @param nodeId The node for which the CC version should be retrieved
     * @param endpointIndex The endpoint for which the CC version should be retrieved
     */
    getSafeCCVersion(cc: CommandClasses, nodeId: number, endpointIndex?: number): number | undefined;
    /**
     * Determines whether a CC must be secure for a given node and endpoint.
     *
     * @param ccId The command class in question
     * @param nodeId The node for which the CC security should be determined
     * @param endpointIndex The endpoint for which the CC security should be determined
     */
    isCCSecure(ccId: CommandClasses, nodeId: number, endpointIndex?: number): boolean;
    /**
     * **!!! INTERNAL !!!**
     *
     * Not intended to be used by applications.
     * Needed for compatibility with CCAPIHost
     */
    schedulePoll(nodeId: number, valueId: ValueID, options: SchedulePollOptions): boolean;
    private isSoftResetting;
    private maySoftReset;
    /**
     * Soft-resets the controller if the feature is enabled
     */
    trySoftReset(): Promise<void>;
    /**
     * Instruct the controller to soft-reset.
     *
     * **Warning:** USB modules will reconnect, meaning that they might get a new address.
     *
     * **Warning:** This call will throw if soft-reset is not enabled.
     */
    softReset(): Promise<void>;
    private softResetInternal;
    /** Soft-reset the Z-Wave module and restart the driver instance */
    softResetAndRestart(): Promise<void>;
    /**
     * Checks whether recovering an unresponsive controller is enabled
     * and whether the driver is in a state where it makes sense.
     */
    private mayRecoverUnresponsiveController;
    private ensureSerialAPI;
    private _ensureCLIReadyPromise;
    private ensureCLIReady;
    /**
     * Performs a hard reset on the controller. This wipes out all configuration!
     *
     * The returned Promise resolves when the hard reset has been performed.
     * It does not wait for the initialization process which is started afterwards.
     */
    hardReset(): Promise<void>;
    /**
     * Instructs the Z-Wave API to shut down in order to safely remove the power.
     * This will destroy the driver instance if it succeeds.
     */
    shutdown(): Promise<boolean>;
    private _destroyPromise;
    private get wasDestroyed();
    /**
     * Ensures that the driver is ready to communicate (serial port open and not destroyed).
     * If desired, also checks that the controller interview has been completed.
     */
    private ensureReady;
    /** Indicates whether the driver is ready, i.e. the "driver ready" event was emitted */
    get ready(): boolean;
    private destroyWithMessage;
    /**
     * Terminates the driver instance and closes the underlying serial connection.
     * Must be called under any circumstances.
     */
    destroy(): Promise<void>;
    /** Cleanly destroy the controller instance, but not the entire driver */
    private destroyController;
    private closeDatabases;
    private clearAllTimeouts;
    private handleSerialData;
    private handleSerialPortClosedUnexpectedly;
    /**
     * Is called when the serial port has received a single-byte message or a complete message buffer
     */
    private serialport_onData;
    /** Handles a decoding error and returns the desired reply to the stick */
    private handleDecodeError;
    private mustReplyWithSecurityS2MOS;
    private handleSecurityS2DecodeError;
    /** Checks if a transaction failed because a node didn't respond in time */
    private isMissingNodeACK;
    private shouldRequestWakeupOnDemand;
    private partialCCSessions;
    private getPartialCCSession;
    /**
     * Assembles partial CCs of in a message body. Returns `true` when the message is complete and can be handled further.
     * If the message expects another partial one, this returns `false`.
     */
    private assemblePartialCCs;
    /** Is called when a Transport Service command is received */
    private handleTransportServiceCommand;
    /**
     * Is called when a message is received that does not belong to any ongoing transactions
     * @param msg The decoded message
     */
    private handleUnsolicitedMessage;
    /**
     * Is called when the Serial API restart unexpectedly.
     */
    private handleSerialAPIStartedUnexpectedly;
    /**
     * Registers a handler for messages that are not handled by the driver as part of a message exchange.
     * The handler function needs to return a boolean indicating if the message has been handled.
     * Registered handlers are called in sequence until a handler returns `true`.
     *
     * @param fnType The function type to register the handler for
     * @param handler The request handler callback
     * @param oneTime Whether the handler should be removed after its first successful invocation
     */
    registerRequestHandler<T extends Message>(fnType: FunctionType, handler: RequestHandler<T>, oneTime?: boolean): void;
    /**
     * Unregisters a message handler that has been added with `registerRequestHandler`
     * @param fnType The function type to unregister the handler for
     * @param handler The previously registered request handler callback
     */
    unregisterRequestHandler(fnType: FunctionType, handler: RequestHandler): void;
    /**
     * Checks whether a CC has a lower than expected security level and needs to be discarded
     */
    private isSecurityLevelTooLow;
    /** Checks whether a CC should be discarded */
    private shouldDiscardCC;
    /**
     * Is called when a Response-type message was received
     */
    private handleResponse;
    /**
     * Is called when a Request-type message was received
     */
    private handleRequest;
    private hasLoggedNoNetworkKey;
    private handleSecurityNonceGet;
    /**
     * Is called when a nonce report is received that does not belong to any transaction.
     * The received nonce reports are stored as "free" nonces
     */
    private handleSecurityNonceReport;
    private handleSecurityCommandsSupportedGet;
    /** Handles a nonce request for S2 */
    private handleSecurity2NonceGet;
    /**
     * Is called when a nonce report is received that does not belong to any transaction.
     */
    private handleSecurity2NonceReport;
    private handleSecurity2CommandsSupportedGet;
    /**
     * Returns the next callback ID. Callback IDs are used to correlate requests
     * to the controller/nodes with its response
     */
    readonly getNextCallbackId: () => number;
    private readonly supervisionSessionIDs;
    /**
     * Returns the next session ID for Supervision CC
     */
    getNextSupervisionSessionId(nodeId: number): number;
    /**
     * Returns the next session ID for Transport Service CC
     */
    readonly getNextTransportServiceSessionId: () => number;
    private encapsulateCommands;
    unwrapCommands(msg: Message & ContainsCC): void;
    private shouldPersistCCValues;
    /** Persists the values contained in a Command Class in the corresponding nodes's value DB */
    private persistCCValues;
    /**
     * Gets called whenever any Serial API command succeeded or a SendData command had a negative callback.
     */
    private handleSerialAPICommandResult;
    private shouldUseLowPriorityForSupervisionReport;
    private mayStartTransaction;
    private markQueueBusy;
    private drainTransactionQueue;
    /** Steps through the message generator of a transaction. Throws an error if the transaction should fail. */
    private executeTransaction;
    /**
     * Provides access to the result Promise for the currently executed serial API command
     */
    private _currentSerialAPICommandPromise;
    /** Handles sequencing of queued Serial API commands */
    private drainSerialAPIQueue;
    private triggerQueues;
    /** Puts a message on the serial API queue and returns or throws the command result */
    private queueSerialAPICommand;
    private mayRetrySerialAPICommand;
    /**
     * Executes a Serial API command and returns or throws the result.
     * This method should not be called outside of {@link drainSerialAPIQueue}.
     */
    private executeSerialAPICommand;
    private getQueueForTransaction;
    /**
     * Sends a message to the Z-Wave stick.
     * @param msg The message to send
     * @param options (optional) Options regarding the message transmission
     */
    sendMessage<TResponse extends Message = Message>(msg: Message, options?: SendMessageOptions): Promise<TResponse>;
    /** Wraps a CC in the correct SendData message to use for sending */
    createSendDataMessage(command: CommandClass, options?: Omit<SendCommandOptions, keyof SendMessageOptions>): SendDataMessage & ContainsCC;
    /**
     * Sends a command to a Z-Wave node. If the node returns a command in response, that command will be the return value.
     * If the command expects no response **or** the response times out, nothing will be returned
     * @param command The command to send. It will be encapsulated in a SendData[Multicast]Request.
     * @param options (optional) Options regarding the message transmission
     */
    private sendCommandInternal;
    /**
     * Sends a supervised command to a Z-Wave node. When status updates are requested, the passed callback will be executed for every non-final update.
     * @param command The command to send
     * @param options (optional) Options regarding the message transmission
     */
    private sendSupervisedCommand;
    /**
     * Sends a command to a Z-Wave node. The return value depends on several factors:
     * * If the node returns a command in response, that command will be the return value.
     * * If the command is a SET-type command and Supervision CC can and should be used, a {@link SupervisionResult} will be returned.
     * * If the command expects no response **or** the response times out, nothing will be returned.
     *
     * @param command The command to send. It will be encapsulated in a SendData[Multicast]Request.
     * @param options (optional) Options regarding the message transmission
     */
    sendCommand<TResponse extends CCId | undefined = undefined>(command: CommandClass, options?: SendCommandOptions): Promise<SendCommandReturnType<TResponse>>;
    private abortSendData;
    /**
     * Sends a low-level message like ACK, NAK or CAN immediately
     * @param header The low-level message to send
     */
    private writeHeader;
    /** Sends a raw datagram to the serialport (if that is open) */
    private writeSerial;
    waitForMessageHeader<T extends MessageHeaders>(predicate: (header: MessageHeaders) => header is T, timeout?: number, abortSignal?: AbortSignal): Promise<T>;
    waitForMessageHeader(predicate: (header: MessageHeaders) => boolean, timeout?: number, abortSignal?: AbortSignal): Promise<MessageHeaders>;
    waitForMessage<T extends Message>(predicate: (msg: Message) => msg is T, timeout?: number, refreshPredicate?: (msg: Message) => boolean, abortSignal?: AbortSignal): Promise<T>;
    waitForMessage<T extends Message>(predicate: (msg: Message) => boolean, timeout?: number, refreshPredicate?: (msg: Message) => boolean, abortSignal?: AbortSignal): Promise<T>;
    waitForCommand<T extends CCId, U extends T>(predicate: (cc: CCId) => cc is U, timeout?: number, abortSignal?: AbortSignal): Promise<U>;
    waitForCommand<T extends CCId>(predicate: (cc: CCId) => boolean, timeout?: number, abortSignal?: AbortSignal): Promise<T>;
    /**
     * Calls the given handler function every time a CommandClass is received that matches the given predicate.
     * @param predicate A predicate function to test all incoming command classes
     */
    registerCommandHandler<T extends CCId>(predicate: (cc: CCId) => boolean, handler: (cc: T) => void): {
        unregister: () => void;
    };
    private handleFailedTransaction;
    private rejectTransaction;
    private resolveTransaction;
    /** Checks if a message is allowed to go into the wakeup queue */
    private mayMoveToWakeupQueue;
    /** Moves all messages for a given node into the wakeup queue */
    private moveMessagesToWakeupQueue;
    /**
     * Pauses the send queue, avoiding commands to be sent to the controller
     */
    private pauseSendQueue;
    /**
     * Unpauses the send queue, allowing commands to be sent to the controller again
     */
    private unpauseSendQueue;
    private reduceQueues;
    private reduceQueue;
    private cachePurge;
    /**
     * Restores a previously stored Z-Wave network state from cache to speed up the startup process
     */
    restoreNetworkStructureFromCache(): Promise<void>;
    private sendNodeToSleepTimers;
    /** Computes the maximum net CC payload size for the given CC or SendDataRequest */
    computeNetCCPayloadSize(commandOrMsg: CommandClass | (SendDataRequest | SendDataBridgeRequest) & ContainsCC, ignoreEncapsulation?: boolean): number;
    /** Computes the maximum payload size that can be transmitted with the given message */
    getMaxPayloadLength(msg: SendDataMessage): number;
    exceedsMaxPayloadLength(msg: SendDataMessage): Promise<boolean>;
    /** Determines time in milliseconds to wait for a report from a node */
    getReportTimeout(msg: Message): number;
    /** Returns the preferred constructor to use for singlecast SendData commands */
    getSendDataSinglecastConstructor(): typeof SendDataRequest | typeof SendDataBridgeRequest;
    /** Returns the preferred constructor to use for multicast SendData commands */
    getSendDataMulticastConstructor(): typeof SendDataMulticastRequest | typeof SendDataMulticastBridgeRequest;
    /**
     * Checks whether there is a compatible update for the currently installed config package.
     * Returns the new version if there is an update, `undefined` otherwise.
     */
    checkForConfigUpdates(silent?: boolean): Promise<string | undefined>;
    private _installConfigUpdatePromise;
    /**
     * Installs an update for the embedded configuration DB if there is a compatible one.
     * Returns `true` when an update was installed, `false` otherwise.
     *
     * **Note:** Bugfixes and changes to device configuration generally require a restart or re-interview to take effect.
     */
    installConfigUpdate(): Promise<boolean>;
    private installConfigUpdateInternal;
    private _otwFirmwareUpdateInProgress;
    /**
     * Returns whether a firmware update is in progress for the Z-Wave module.
     */
    isOTWFirmwareUpdateInProgress(): boolean;
    /**
     * Updates the firmware of the controller using the given firmware file.
     * The file can be provided as binary data or as a {@link FirmwareUpdateInfo} object as returned
     * from the firmware update service. The latter will be downloaded automatically.
     *
     * The return value indicates whether the update was successful.
     * **WARNING:** After a successful update, the Z-Wave driver will destroy itself so it can be restarted.
     *
     * **WARNING:** A failure during this process may put your controller in recovery mode, rendering it unusable until a correct firmware image is uploaded. Use at your own risk!
     */
    firmwareUpdateOTW(data: Uint8Array | FirmwareUpdateInfo): Promise<OTWFirmwareUpdateResult>;
    /**
     * Downloads the desired firmware update from the Z-Wave JS firmware update service and extracts the firmware data.
     * @param updateInfo The desired entry from the updates array that was returned by {@link getAvailableFirmwareUpdates}.
     * Before applying the update, Z-Wave JS will check whether the device IDs, firmware version and region match.
     */
    private extractOTWUpdateInfo;
    private firmwareUpdateOTW500;
    private firmwareUpdateOTW700;
    private firmwareUpdateOTW700Internal;
    private _enteringBootloader;
    private _enterBootloaderPromise;
    enterBootloader(): Promise<void>;
    private enterBootloaderFromSerialAPI;
    private enterBootloaderFromCLI;
    private leaveBootloaderInternal;
    /**
     * Leaves the bootloader by running the application.
     */
    leaveBootloader(): Promise<void>;
    private serialport_onBootloaderData;
    waitForBootloaderChunk<T extends BootloaderChunk>(predicate: (chunk: BootloaderChunk) => chunk is T, timeout?: number, abortSignal?: AbortSignal): Promise<T>;
    waitForBootloaderChunk<T extends BootloaderChunk>(predicate: (chunk: BootloaderChunk) => boolean, timeout?: number, abortSignal?: AbortSignal): Promise<T>;
    waitForCLIChunk<T extends CLIChunk>(predicate: (chunk: CLIChunk) => chunk is T, timeout?: number, abortSignal?: AbortSignal): Promise<T>;
    waitForCLIChunk<T extends CLIChunk>(predicate: (chunk: CLIChunk) => boolean, timeout?: number, abortSignal?: AbortSignal): Promise<T>;
    private serialport_onCLIData;
    private pollBackgroundRSSITimer;
    private lastBackgroundRSSITimestamp;
    private handleQueueIdleChange;
    private _powerlevelTestNodeContext;
    /** Sends a NOP Power frame to the given node and returns the transmit status if the frame was sent */
    sendTestFrame(nodeId: number, powerlevel: Powerlevel): Promise<TransmitStatus | undefined>;
    /**
     * Resets the S2 singlecast encryption state (SPAN) for the given node, which forces
     * a re-synchronization on the next communication attempt.
     */
    resetSPAN(nodeId: number): void;
    /**
     * Resets the S2 singlecast encryption state (SPAN) for all nodes, which forces
     * a re-synchronization on the next communication attempt.
     */
    resetAllSPANs(): void;
}
export {};
//# sourceMappingURL=Driver.d.ts.map