import { pick } from "@zwave-js/shared";
import { isArray, isObject } from "alcalzone-shared/typeguards";
import { throwInvalidConfig } from "../utils_safe.js";
import { conditionApplies, evaluateDeep, } from "./ConditionalItem.js";
import { parseConditionalPrimitive, } from "./ConditionalPrimitive.js";
export class ConditionalDeviceMetadata {
    constructor(filename, definition) {
        for (const prop of [
            "wakeup",
            "inclusion",
            "exclusion",
            "reset",
            "manual",
        ]) {
            if (prop in definition) {
                this[prop] = parseConditionalPrimitive(filename, "string", prop, definition[prop], "The metadata entry ");
            }
        }
        if ("comments" in definition) {
            const value = definition.comments;
            const isComment = (opt) => isObject(opt)
                && typeof opt.level === "string"
                && typeof opt.text === "string";
            if (isComment(value)) {
                this.comments = new ConditionalDeviceComment(value.level, value.text, value.$if);
            }
            else if (isArray(value) && value.every(isComment)) {
                this.comments = value.map((c) => new ConditionalDeviceComment(c.level, c.text, c.$if));
            }
            else {
                throwInvalidConfig("devices", `packages/config/config/devices/${filename}:
The metadata entry comments is invalid!`);
            }
        }
    }
    condition;
    evaluateCondition(deviceId) {
        if (!conditionApplies(this, deviceId))
            return;
        const ret = {};
        for (const prop of [
            "wakeup",
            "inclusion",
            "exclusion",
            "reset",
            "manual",
        ]) {
            if (this[prop]) {
                const evaluated = evaluateDeep(this[prop], deviceId);
                if (evaluated)
                    ret[prop] = evaluated;
            }
        }
        const comments = evaluateDeep(this.comments, deviceId, true);
        if (comments)
            ret.comments = comments;
        return ret;
    }
    /** How to wake up the device manually */
    wakeup;
    /** Inclusion instructions */
    inclusion;
    /** Exclusion instructions */
    exclusion;
    /** Instructions for resetting the device to factory defaults */
    reset;
    /** A link to the device manual */
    manual;
    /** Comments for this device */
    comments;
}
export class ConditionalDeviceComment {
    level;
    text;
    condition;
    constructor(level, text, condition) {
        this.level = level;
        this.text = text;
        this.condition = condition;
    }
    evaluateCondition(deviceId) {
        if (!conditionApplies(this, deviceId))
            return;
        return pick(this, ["level", "text"]);
    }
}
//# sourceMappingURL=DeviceMetadata.js.map