import { tryParseParamNumber } from "@zwave-js/core";
import { ObjectKeyMap, pick, } from "@zwave-js/shared";
import { isArray, isObject } from "alcalzone-shared/typeguards";
import { throwInvalidConfig } from "../utils_safe.js";
import { conditionApplies, evaluateDeep, validateCondition, } from "./ConditionalItem.js";
export class ConditionalParamInformation {
    constructor(parent, parameterNumber, valueBitMask, definition) {
        this.parent = parent;
        this.parameterNumber = parameterNumber;
        this.valueBitMask = valueBitMask;
        // No need to validate here, this should be done one level higher
        this.condition = definition.$if;
        if (typeof definition.label !== "string") {
            throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber} has a non-string label`);
        }
        this.label = definition.label;
        if (definition.description != undefined
            && typeof definition.description !== "string") {
            throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber} has a non-string description`);
        }
        this.description = definition.description;
        if (typeof definition.valueSize !== "number"
            || definition.valueSize <= 0) {
            throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber} has an invalid value size`);
        }
        this.valueSize = definition.valueSize;
        if (definition.minValue != undefined
            && typeof definition.minValue !== "number") {
            throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber} has a non-numeric property minValue`);
        }
        this.minValue = definition.minValue;
        if (definition.maxValue != undefined
            && typeof definition.maxValue !== "number") {
            throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber} has a non-numeric property maxValue`);
        }
        this.maxValue = definition.maxValue;
        if (definition.unsigned != undefined
            && typeof definition.unsigned !== "boolean") {
            throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber} has a non-boolean property unsigned`);
        }
        this.unsigned = definition.unsigned === true;
        if (definition.unit != undefined
            && typeof definition.unit !== "string") {
            throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber} has a non-string unit`);
        }
        this.unit = definition.unit;
        if (definition.readOnly != undefined && definition.readOnly !== true) {
            throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
		Parameter #${parameterNumber}: readOnly must true or omitted!`);
        }
        this.readOnly = definition.readOnly;
        if (definition.writeOnly != undefined
            && definition.writeOnly !== true) {
            throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
		Parameter #${parameterNumber}: writeOnly must be true or omitted!`);
        }
        this.writeOnly = definition.writeOnly;
        if (definition.defaultValue == undefined) {
            if (!this.readOnly) {
                throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber} is missing defaultValue, which is required unless the parameter is readOnly`);
            }
        }
        else if (typeof definition.defaultValue !== "number") {
            throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber} has a non-numeric property defaultValue`);
        }
        this.defaultValue = definition.defaultValue;
        if (definition.recommendedValue != undefined
            && typeof definition.recommendedValue !== "number") {
            throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber} has a non-numeric property recommendedValue`);
        }
        this.recommendedValue = definition.recommendedValue;
        if (definition.allowManualEntry != undefined
            && definition.allowManualEntry !== false) {
            throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber}: allowManualEntry must be false or omitted!`);
        }
        // Default to allowing manual entry, except if the param is readonly
        this.allowManualEntry = definition.allowManualEntry ?? !this.readOnly;
        if (definition.destructive != undefined
            && typeof definition.destructive !== "boolean") {
            throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber} has a non-boolean property destructive`);
        }
        this.destructive = definition.destructive;
        if (isArray(definition.options)
            && !definition.options.every((opt) => isObject(opt)
                && typeof opt.label === "string"
                && typeof opt.value === "number")) {
            throwInvalidConfig("devices", `packages/config/config/devices/${parent.filename}:
Parameter #${parameterNumber}: options is malformed!`);
        }
        this.options = definition.options?.map((opt) => new ConditionalConfigOption(opt.value, opt.label, opt.$if)) ?? [];
    }
    parent;
    parameterNumber;
    valueBitMask;
    label;
    description;
    valueSize;
    minValue;
    maxValue;
    unsigned;
    defaultValue;
    recommendedValue;
    unit;
    readOnly;
    writeOnly;
    allowManualEntry;
    destructive;
    options;
    condition;
    evaluateCondition(deviceId) {
        if (!conditionApplies(this, deviceId))
            return;
        const ret = {
            ...pick(this, [
                "parameterNumber",
                "valueBitMask",
                "label",
                "description",
                "valueSize",
                "minValue",
                "maxValue",
                "unsigned",
                "defaultValue",
                "recommendedValue",
                "unit",
                "readOnly",
                "writeOnly",
                "allowManualEntry",
                "destructive",
            ]),
            options: evaluateDeep(this.options, deviceId, true),
        };
        // Infer minValue from options if possible
        if (ret.minValue == undefined) {
            if (ret.allowManualEntry === false && ret.options.length > 0) {
                ret.minValue = Math.min(...ret.options.map((o) => o.value));
            }
            else {
                throwInvalidConfig("devices", `packages/config/config/devices/${this.parent.filename}:
Parameter #${this.parameterNumber} is missing required property "minValue"!`);
            }
        }
        if (ret.maxValue == undefined) {
            if (ret.allowManualEntry === false && ret.options.length > 0) {
                ret.maxValue = Math.max(...ret.options.map((o) => o.value));
            }
            else {
                throwInvalidConfig("devices", `packages/config/config/devices/${this.parent.filename}:
Parameter #${this.parameterNumber} is missing required property "maxValue"!`);
            }
        }
        // @ts-expect-error TS doesn't seem to understand that we do set min/maxValue
        return ret;
    }
}
export class ConditionalConfigOption {
    value;
    label;
    condition;
    constructor(value, label, condition) {
        this.value = value;
        this.label = label;
        this.condition = condition;
    }
    evaluateCondition(deviceId) {
        if (!conditionApplies(this, deviceId))
            return;
        return pick(this, ["value", "label"]);
    }
}
export function parseConditionalParamInformationMap(definition, parent, errorPrefix = "") {
    const paramInformation = new ObjectKeyMap();
    const filename = parent.filename;
    if (isArray(definition.paramInformation)) {
        // Check that every param has a param number
        if (!definition.paramInformation.every((entry) => "#" in entry)) {
            throwInvalidConfig(`device`, `packages/config/config/devices/${filename}: 
${errorPrefix}required property "#" missing in at least one entry of paramInformation`);
        }
        // And a valid $if condition
        for (const entry of definition.paramInformation) {
            validateCondition(filename, entry, `${errorPrefix}At least one entry of paramInformation contains an`);
        }
        for (const paramDefinition of definition.paramInformation) {
            const { ["#"]: paramNo, ...defn } = paramDefinition;
            const key = tryParseParamNumber(paramNo);
            if (!key) {
                throwInvalidConfig(`device`, `packages/config/config/devices/${filename}: 
${errorPrefix}found invalid param number "${paramNo}" in paramInformation`);
            }
            if (!paramInformation.has(key))
                paramInformation.set(key, []);
            paramInformation
                .get(key)
                .push(new ConditionalParamInformation(parent, key.parameter, key.valueBitMask, defn));
        }
    }
    else if (isObject(definition.paramInformation)) {
        // Silently ignore this old format
    }
    else {
        throwInvalidConfig(`device`, `packages/config/config/devices/${filename}:
${errorPrefix}paramInformation must be an array!`);
    }
    return paramInformation;
}
//# sourceMappingURL=ParamInformation.js.map