"use strict";
var __defProp = Object.defineProperty;
var __getOwnPropDesc = Object.getOwnPropertyDescriptor;
var __getOwnPropNames = Object.getOwnPropertyNames;
var __hasOwnProp = Object.prototype.hasOwnProperty;
var __name = (target, value) => __defProp(target, "name", { value, configurable: true });
var __export = (target, all) => {
  for (var name in all)
    __defProp(target, name, { get: all[name], enumerable: true });
};
var __copyProps = (to, from, except, desc) => {
  if (from && typeof from === "object" || typeof from === "function") {
    for (let key of __getOwnPropNames(from))
      if (!__hasOwnProp.call(to, key) && key !== except)
        __defProp(to, key, { get: () => from[key], enumerable: !(desc = __getOwnPropDesc(from, key)) || desc.enumerable });
  }
  return to;
};
var __toCommonJS = (mod) => __copyProps(__defProp({}, "__esModule", { value: true }), mod);
var primitives_browser_exports = {};
__export(primitives_browser_exports, {
  primitives: () => primitives
});
module.exports = __toCommonJS(primitives_browser_exports);
var import_shared = require("@zwave-js/shared");
var import_shared2 = require("../shared.js");
function randomBytes(length) {
  const buffer = new Uint8Array(length);
  return crypto.getRandomValues(buffer);
}
__name(randomBytes, "randomBytes");
async function encryptAES128ECB(plaintext, key) {
  return encryptAES128CBC(plaintext, key, new Uint8Array(import_shared2.BLOCK_SIZE).fill(0));
}
__name(encryptAES128ECB, "encryptAES128ECB");
async function encryptAES128CBC(plaintext, key, iv) {
  const cryptoKey = await crypto.subtle.importKey("raw", key, { name: "AES-CBC" }, true, ["encrypt"]);
  const ciphertext = await crypto.subtle.encrypt({
    name: "AES-CBC",
    iv
  }, cryptoKey, plaintext);
  const paddedLength = Math.ceil(plaintext.length / import_shared2.BLOCK_SIZE) * import_shared2.BLOCK_SIZE;
  return new Uint8Array(ciphertext, 0, paddedLength);
}
__name(encryptAES128CBC, "encryptAES128CBC");
async function decryptAES256CBC(ciphertext, key, iv) {
  const cryptoKey = await crypto.subtle.importKey("raw", key, { name: "AES-CBC" }, true, ["decrypt"]);
  const plaintext = await crypto.subtle.decrypt({
    name: "AES-CBC",
    iv
  }, cryptoKey, ciphertext);
  return new Uint8Array(plaintext);
}
__name(decryptAES256CBC, "decryptAES256CBC");
async function encryptAES128OFB(plaintext, key, iv) {
  const cryptoKey = await crypto.subtle.importKey("raw", key, { name: "AES-CTR" }, true, [
    "encrypt",
    "decrypt"
  ]);
  const ret = new Uint8Array(plaintext.length);
  let counter = (0, import_shared2.zeroPad)(iv, import_shared2.BLOCK_SIZE).output;
  for (let offset = 0; offset < plaintext.length - 1; offset += import_shared2.BLOCK_SIZE) {
    const input = plaintext.slice(offset, offset + import_shared2.BLOCK_SIZE);
    const ciphertextBuffer = await crypto.subtle.encrypt({
      name: "AES-CTR",
      counter,
      length: import_shared2.BLOCK_SIZE * 8
    }, cryptoKey, input);
    const ciphertext = new Uint8Array(ciphertextBuffer);
    ret.set(ciphertext, offset);
    counter = (0, import_shared2.zeroPad)((0, import_shared2.xor)(ciphertext, input), import_shared2.BLOCK_SIZE).output;
  }
  return ret;
}
__name(encryptAES128OFB, "encryptAES128OFB");
async function decryptAES128OFB(ciphertext, key, iv) {
  const cryptoKey = await crypto.subtle.importKey("raw", key, { name: "AES-CTR" }, true, [
    "encrypt",
    "decrypt"
  ]);
  const ret = new Uint8Array(ciphertext.length);
  let counter = (0, import_shared2.zeroPad)(iv, import_shared2.BLOCK_SIZE).output;
  for (let offset = 0; offset < ciphertext.length - 1; offset += import_shared2.BLOCK_SIZE) {
    const input = ciphertext.slice(offset, offset + import_shared2.BLOCK_SIZE);
    const plaintextBuffer = await crypto.subtle.decrypt({
      name: "AES-CTR",
      counter,
      length: import_shared2.BLOCK_SIZE * 8
    }, cryptoKey, input);
    const plaintext = new Uint8Array(plaintextBuffer);
    ret.set(plaintext, offset);
    counter = (0, import_shared2.zeroPad)((0, import_shared2.xor)(plaintext, input), import_shared2.BLOCK_SIZE).output;
  }
  return ret;
}
__name(decryptAES128OFB, "decryptAES128OFB");
async function encryptAES128CCM(plaintext, key, iv, additionalData, authTagLength) {
  const M = authTagLength - 2 >> 1;
  const L = 15 - iv.length;
  const hasAData = additionalData.length > 0;
  const plaintextBlocks = getCCMPlaintextBlocks(plaintext);
  const B = getCCMAuthenticationBlocks(hasAData, M, L, iv, plaintext, additionalData, plaintextBlocks);
  const X = await computeCBCMac(B, key);
  const A0 = new Uint8Array(import_shared2.BLOCK_SIZE);
  A0[0] = L - 1 & 7;
  A0.set(iv, 1);
  const cryptoKey = await crypto.subtle.importKey("raw", key, { name: "AES-CTR" }, true, ["encrypt"]);
  const encryptionInput = import_shared.Bytes.concat([X, plaintextBlocks]);
  const encryptionOutput = await crypto.subtle.encrypt({
    name: "AES-CTR",
    counter: A0,
    length: import_shared2.BLOCK_SIZE * 8
  }, cryptoKey, encryptionInput);
  const authTagAndCiphertext = new Uint8Array(encryptionOutput);
  const authTag = authTagAndCiphertext.slice(0, authTagLength);
  const ciphertext = authTagAndCiphertext.slice(import_shared2.BLOCK_SIZE).slice(0, plaintext.length);
  return { ciphertext, authTag };
}
__name(encryptAES128CCM, "encryptAES128CCM");
async function computeCBCMac(B, key) {
  const macOutput = await encryptAES128CBC(B, key, new Uint8Array(import_shared2.BLOCK_SIZE).fill(0));
  const X = macOutput.subarray(-import_shared2.BLOCK_SIZE);
  return X;
}
__name(computeCBCMac, "computeCBCMac");
function getCCMPlaintextBlocks(plaintext) {
  const plaintextBlocks = new import_shared.Bytes(
    // plaintext | ...padding
    Math.ceil(plaintext.length / import_shared2.BLOCK_SIZE) * import_shared2.BLOCK_SIZE
  );
  plaintextBlocks.set(plaintext, 0);
  return plaintextBlocks;
}
__name(getCCMPlaintextBlocks, "getCCMPlaintextBlocks");
function getCCMAuthenticationBlocks(hasAData, M, L, iv, plaintext, additionalData, plaintextBlocks) {
  const B0 = new import_shared.Bytes(import_shared2.BLOCK_SIZE);
  B0[0] = (hasAData ? 64 : 0) | (M & 7) << 3 | L - 1 & 7;
  B0.set(iv, 1);
  B0.writeUIntBE(plaintext.length, 16 - L, L);
  let aDataLength;
  if (additionalData.length === 0) {
    aDataLength = new import_shared.Bytes(0);
  } else if (additionalData.length < 65280) {
    aDataLength = new import_shared.Bytes(2);
    aDataLength.writeUInt16BE(additionalData.length, 0);
  } else if (additionalData.length <= 4294967295) {
    aDataLength = new import_shared.Bytes(6);
    aDataLength.writeUInt16BE(65534, 0);
    aDataLength.writeUInt32BE(additionalData.length, 2);
  } else {
    aDataLength = new import_shared.Bytes(10);
    aDataLength.writeUInt16BE(65535, 0);
    aDataLength.writeBigUInt64BE(BigInt(additionalData.length), 2);
  }
  const aDataBlocks = new import_shared.Bytes(
    // B0 | aDataLength | additionalData | ...padding
    Math.ceil((import_shared2.BLOCK_SIZE + aDataLength.length + additionalData.length) / import_shared2.BLOCK_SIZE) * import_shared2.BLOCK_SIZE
  );
  aDataBlocks.set(B0, 0);
  aDataBlocks.set(aDataLength, import_shared2.BLOCK_SIZE);
  aDataBlocks.set(additionalData, import_shared2.BLOCK_SIZE + aDataLength.length);
  const B = import_shared.Bytes.concat([aDataBlocks, plaintextBlocks]);
  return B;
}
__name(getCCMAuthenticationBlocks, "getCCMAuthenticationBlocks");
async function decryptAES128CCM(ciphertext, key, iv, additionalData, authTag) {
  const M = authTag.length - 2 >> 1;
  const L = 15 - iv.length;
  const hasAData = additionalData.length > 0;
  const A0 = new Uint8Array(import_shared2.BLOCK_SIZE);
  A0[0] = L - 1 & 7;
  A0.set(iv, 1);
  const cryptoKey = await crypto.subtle.importKey("raw", key, { name: "AES-CTR" }, true, ["decrypt"]);
  const paddedAuthTag = new import_shared.Bytes(import_shared2.BLOCK_SIZE);
  paddedAuthTag.set(authTag, 0);
  const decryptionInput = import_shared.Bytes.concat([paddedAuthTag, ciphertext]);
  const decryptionOutput = await crypto.subtle.decrypt({
    name: "AES-CTR",
    counter: A0,
    length: import_shared2.BLOCK_SIZE * 8
  }, cryptoKey, decryptionInput);
  const plaintextAndT = new Uint8Array(decryptionOutput);
  const T = plaintextAndT.slice(0, authTag.length);
  const plaintext = plaintextAndT.slice(import_shared2.BLOCK_SIZE);
  const plaintextBlocks = getCCMPlaintextBlocks(plaintext);
  const B = getCCMAuthenticationBlocks(hasAData, M, L, iv, plaintext, additionalData, plaintextBlocks);
  const X = await computeCBCMac(B, key);
  const expectedAuthTag = X.subarray(0, authTag.length);
  const emptyPlaintext = new Uint8Array();
  let result = 0;
  if (T.length !== expectedAuthTag.length) {
    return { plaintext: emptyPlaintext, authOK: false };
  }
  for (let i = 0; i < T.length; i++) {
    result |= T[i] ^ expectedAuthTag[i];
  }
  if (result === 0) {
    return { plaintext, authOK: true };
  } else {
    return { plaintext: emptyPlaintext, authOK: false };
  }
}
__name(decryptAES128CCM, "decryptAES128CCM");
async function digest(algorithm, data) {
  if (algorithm === "md5") {
    algorithm = "sha-256";
  }
  const output = await crypto.subtle.digest(algorithm, data);
  return new Uint8Array(output);
}
__name(digest, "digest");
async function generateECDHKeyPair() {
  const pair = await crypto.subtle.generateKey("X25519", true, ["deriveKey"]);
  const publicKey = new Uint8Array(await crypto.subtle.exportKey("raw", pair.publicKey));
  const privateKey = (0, import_shared2.decodeX25519KeyDER)(new Uint8Array(await crypto.subtle.exportKey("pkcs8", pair.privateKey)));
  return { publicKey, privateKey };
}
__name(generateECDHKeyPair, "generateECDHKeyPair");
async function keyPairFromRawECDHPrivateKey(privateKey) {
  const privateKeyObject = await crypto.subtle.importKey("pkcs8", (0, import_shared2.encodeX25519KeyDERPKCS8)(privateKey), "X25519", true, ["deriveKey"]);
  const jwk = await crypto.subtle.exportKey("jwk", privateKeyObject);
  delete jwk.d;
  const publicKeyObject = await crypto.subtle.importKey("jwk", jwk, "X25519", true, []);
  const publicKey = new Uint8Array(await crypto.subtle.exportKey("raw", publicKeyObject));
  return { publicKey, privateKey };
}
__name(keyPairFromRawECDHPrivateKey, "keyPairFromRawECDHPrivateKey");
async function deriveSharedECDHSecret(keyPair) {
  const publicKey = await crypto.subtle.importKey("raw", keyPair.publicKey, "X25519", true, []);
  const privateKey = await crypto.subtle.importKey("pkcs8", (0, import_shared2.encodeX25519KeyDERPKCS8)(keyPair.privateKey), "X25519", true, ["deriveBits"]);
  const secret = await crypto.subtle.deriveBits({
    name: "X25519",
    public: publicKey
  }, privateKey, null);
  return new Uint8Array(secret);
}
__name(deriveSharedECDHSecret, "deriveSharedECDHSecret");
const primitives = {
  randomBytes,
  encryptAES128ECB,
  encryptAES128CBC,
  encryptAES128OFB,
  decryptAES128OFB,
  encryptAES128CCM,
  decryptAES128CCM,
  decryptAES256CBC,
  digest,
  generateECDHKeyPair,
  keyPairFromRawECDHPrivateKey,
  deriveSharedECDHSecret
};
// Annotate the CommonJS export names for ESM import in node:
0 && (module.exports = {
  primitives
});
//# sourceMappingURL=primitives.browser.js.map
