/**
 * Connects a readable stream and a writable stream
 * while allowing the writable stream to be temporarily disconnected
 * or un-writable.
 *
 * Data that is emitted while no writable stream is connected is discarded.
 */
export class Faucet {
    /** Creates a new Faucet. The data starts flowing immediately */
    constructor(readable, writable) {
        this.#readable = readable;
        this.#writable = writable;
        void this.#flow();
    }
    #readable;
    #writable;
    #abort;
    /**
     * Connects a new writable to the faucet.
     * If a writable is already connected, it is replaced.
     */
    connect(writable) {
        this.#writable = writable;
    }
    /**
     * Disconnects the current writable from the faucet.
     */
    disconnect() {
        this.#writable = undefined;
    }
    async #flow() {
        const reader = this.#readable.getReader();
        const abort = new AbortController();
        this.#abort = abort;
        try {
            const abortPromise = new Promise((resolve) => {
                abort.signal.addEventListener("abort", () => {
                    resolve();
                });
            });
            while (true) {
                const result = await Promise.race([
                    reader.read(),
                    abortPromise.then(() => ({ done: true, value: undefined })),
                ]);
                if (result.done || result.value == undefined)
                    break;
                let writer;
                try {
                    writer = this.#writable?.getWriter();
                    await writer?.write(result.value);
                }
                catch {
                    // ignore
                }
                finally {
                    writer?.releaseLock();
                }
            }
        }
        finally {
            reader.releaseLock();
        }
    }
    /** Closes the faucet and stops reading data */
    close() {
        this.#abort?.abort();
    }
}
//# sourceMappingURL=Faucet.js.map