#!/usr/bin/env node
import { resolve } from "path";
import { Driver, ZWaveError, ZWaveErrorCodes, driverPresets, } from "zwave-js";
import { ZwavejsServer } from "../lib/server.js";
import { createMockDriver } from "../mock/index.js";
import { parseArgs } from "../util/parse-args.js";
import { createRequire } from "node:module";
const require = createRequire(import.meta.url);
const normalizeKey = (key, keyName) => {
    if (Buffer.isBuffer(key))
        return key;
    if (key.length === 32)
        return Buffer.from(key, "hex");
    // Convert from OpenZWave format
    key = key.toLowerCase();
    if (key.includes("0x"))
        return Buffer.from(key.replace(/0x/g, "").replace(/, /g, ""), "hex");
    throw new Error(`Invalid key format for ${keyName} option`);
};
const getDriverParams = () => {
    const args = parseArgs([
        "_",
        "config",
        "mock-driver",
        "port",
        "host",
        "disable-dns-sd",
        "reconnect",
    ]);
    if (args.port) {
        if (typeof args["port"] !== "number") {
            throw new Error("port must be a valid integer");
        }
    }
    if (!args["mock-driver"] && args._.length < 1) {
        console.error("Error: Missing path to serial port");
        return;
    }
    const serialPort = args._[0];
    let configPath = args.config;
    if (configPath && configPath.substring(0, 1) !== "/") {
        configPath = resolve(process.cwd(), configPath);
    }
    let options;
    let presetNames;
    if (configPath) {
        try {
            // Pull presets out of options so we can pass them to the driver
            ({ presets: presetNames, ...options } = require(configPath));
            // If both securityKeys.S0_Legacy and networkKey are defined, throw an error.
            if (options.securityKeys?.S0_Legacy && options.networkKey) {
                throw new Error("Both `networkKey` and `securityKeys.S0_Legacy` options are present in the " +
                    "config. Remove `networkKey`.");
            }
            const securityKeyNames = [
                "S0_Legacy",
                "S2_AccessControl",
                "S2_Authenticated",
                "S2_Unauthenticated",
            ];
            // We prefer the securityKeys option over the networkKey one
            if (options.securityKeys) {
                for (const key of securityKeyNames) {
                    if (key in options.securityKeys) {
                        options.securityKeys[key] = normalizeKey(options.securityKeys[key], `securityKeys.${key}`);
                    }
                }
            }
            // If we get here, securityKeys.S0_Legacy is not defined, so we can safely use networkKey
            // make sure that networkKey is passed as buffer and accept both zwave2mqtt format and ozw format
            if (options.networkKey) {
                if (!options.securityKeys)
                    options.securityKeys = {};
                options.securityKeys.S0_Legacy = normalizeKey(options.networkKey, "networkKey");
                console.warn("The `networkKey` option is deprecated in favor of `securityKeys` option. To eliminate " +
                    "this warning, move your networkKey into the securityKeys.S0_Legacy option. Refer to " +
                    "the Z-Wave JS docs for more information");
                delete options.networkKey;
            }
            else if (!options.securityKeys?.S0_Legacy)
                throw new Error("Error: `securityKeys.S0_Legacy` key is missing.");
            // Support long range keys
            const securityKeysLongRangeNames = [
                "S2_AccessControl",
                "S2_Authenticated",
            ];
            if (options.securityKeysLongRange) {
                for (const key of securityKeysLongRangeNames) {
                    if (key in options.securityKeysLongRange) {
                        options.securityKeysLongRange[key] = normalizeKey(options.securityKeysLongRange[key], `securityKeysLongRange.${key}`);
                    }
                }
            }
        }
        catch (err) {
            console.error(`Error: failed loading config file ${configPath}`);
            console.error(err);
            return;
        }
    }
    if (!options) {
        options = { emitValueUpdateAfterSetValue: true };
    }
    else if (!("emitValueUpdateAfterSetValue" in options)) {
        options["emitValueUpdateAfterSetValue"] = true;
    }
    else if (!options["emitValueUpdateAfterSetValue"]) {
        console.warn("Because `emitValueUpdateAfterSetValue` is set to false, multi-client setups will not work " +
            "as expected. In particular, clients will not see value updates that are initiated by " +
            "another client.");
    }
    // Normalize the presets
    let presets;
    if (presetNames !== undefined) {
        if (typeof presetNames === "string") {
            presetNames = [presetNames];
        }
        else if (!Array.isArray(presetNames) ||
            !presetNames.every((p) => typeof p === "string")) {
            throw new Error("presets must be an array of strings or a string if provided");
        }
        presets = presetNames
            .map((name) => driverPresets[name])
            .filter((preset) => preset !== undefined);
    }
    return {
        args,
        serialPort,
        options,
        presets,
    };
};
const logMessage = (...message) => {
    const now = new Date();
    const hours = now.getHours().toString().padStart(2, "0");
    const minutes = now.getMinutes().toString().padStart(2, "0");
    const seconds = now.getSeconds().toString().padStart(2, "0");
    const milliseconds = now.getMilliseconds().toString().padStart(3, "0");
    console.log(`${hours}:${minutes}:${seconds}.${milliseconds} SERVER   ${message.join(" ")}`);
};
const debug = false;
const logger = {
    debug: (msg) => (debug ? logMessage("[DEBUG]", msg) : undefined),
    info: (msg) => logMessage(msg),
    warn: (msg) => logMessage("[WARNING]", msg),
    error: (msg) => logMessage("[ERROR]", msg.toString()),
};
(() => {
    const params = getDriverParams();
    if (!params) {
        process.exit(1);
    }
    let driver;
    let server;
    let retryCount = 0;
    let retryTimer;
    // Exponential backoff: 1s, 2s, 4s, 8s, 16s, 32s, then cap at 60s
    const getRetryDelay = () => Math.min(Math.pow(2, retryCount) * 1000, 60000);
    const startDriverWithRetry = async () => {
        // We only want to show the logo the very first connection attempt.
        if (retryCount > 0 &&
            (!params.options.logConfig || params.options.logConfig.showLogo)) {
            if (!params.options.logConfig) {
                params.options.logConfig = {
                    showLogo: false,
                };
            }
            else {
                params.options.logConfig.showLogo = false;
            }
        }
        if (retryTimer) {
            clearTimeout(retryTimer);
            retryTimer = undefined;
        }
        if (driver) {
            logger.info("stopping driver");
            if (server) {
                await server.destroy();
            }
            if (driver) {
                await driver.destroy();
            }
        }
        driver = params.args["mock-driver"]
            ? createMockDriver()
            : new Driver(params.serialPort, params.options, ...(params.presets ?? []));
        driver.on("error", (e) => {
            logger.error("error in driver", e);
            // Driver_Failed cannot be recovered by zwave-js so we restart
            if (e instanceof ZWaveError && e.code === ZWaveErrorCodes.Driver_Failed) {
                if (params.args["reconnect"]) {
                    startDriverWithRetry();
                }
                else {
                    logger.error("driver failure is unrecoverable, exiting...");
                    process.exit(1);
                }
            }
        });
        driver.once("driver ready", async () => {
            logger.info("driver ready - starting server");
            retryCount = 0; // Reset retry count on successful connection
            server = new ZwavejsServer(driver, {
                port: params.args.port,
                host: params.args.host,
                enableDNSServiceDiscovery: !params.args["disable-dns-sd"],
                logger,
            });
            await server.start(true);
        });
        try {
            await driver.start();
        }
        catch (e) {
            await driver.destroy();
            driver = undefined;
            if (!params.args["reconnect"]) {
                logger.error("failed starting driver.");
                throw e;
            }
            retryCount++;
            const retryDelay = getRetryDelay();
            logger.error(`failed ${retryCount > 1 ? `attempt ${retryCount} ` : ""}starting driver. Retrying in ${Math.round(retryDelay / 1000)}s...`);
            setTimeout(() => {
                startDriverWithRetry();
            }, retryDelay);
        }
    };
    let closing = false;
    const handleShutdown = async (exitCode = 0) => {
        // Pressing ctrl+c twice.
        if (closing) {
            process.exit(exitCode);
        }
        // Close gracefully
        closing = true;
        console.log(); // User pressed ctrl+c - move to new line
        logger.info("shutting down");
        // Clear any pending retry timer
        if (retryTimer) {
            clearTimeout(retryTimer);
            retryTimer = undefined;
        }
        if (server) {
            await server.destroy();
        }
        if (driver) {
            await driver.destroy();
        }
        process.exit(exitCode);
    };
    process.on("SIGINT", () => handleShutdown(0));
    process.on("SIGTERM", () => handleShutdown(0));
    startDriverWithRetry();
})();
